/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/common/IContent.java,v 1.3 2004/07/30 06:51:52 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:52 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.common;

import java.io.InputStream;
import org.apache.slide.content.NodeRevisionContent;
import org.apache.slide.store.tamino.datastore.XContentWrapper;
import org.apache.slide.util.XException;

/**
 * <p>IContent represents the content of the resource. It provides methods for setting
 * and retrieving the content (NodeRevisionContent) as well as methods about the kind
 * of the object (XML or non-XML). From the point of view of Slide, one IContent object
 * is uniquely identified by a URI and a RevisionDescriptor. IDbSession uses a String
 * to identify  the physical location of the content. This would be somehow the ino:id.
 * This contentId must be stored within the correlated RevisionDescriptor object as a
 * live property.</p>
 * <p>IContent uses the stream accessor of Tamino API. Depending on the content type
 * (XML or non-XML) the stream is stored to the non-XML schema or in the supplied schema. </p>
 * @author    martin.wallmer@softwareag.com
 *
 * @version   $Revision: 1.3 $
 */
public interface IContent {
    
    /**
     * nodeRevisionContent accessor.
     *
     * @return     the NodeRevisionContent object represented by this Content object
     *
     * @exception  XException when underlying service reports an error
     */
    NodeRevisionContent getContent () throws XException;
    
    /**
     * nodeRevisionContent mutator.
     *
     * @pre        (nrc != null)
     *
     * @param      nrc the NodeRevisionContent object,
     *
     * @exception  XException when underlying service reports an error
     */
    void setContent (NodeRevisionContent nrc) throws XException;
    
    /**
     * <p>Returns the tamino schema, in which the content shall be stored. The return
     * value gives a hint, whether the content is XML or non-XML data: </p>
     * If the return value is
     * <ul>
     *  <li> null, the data is definitly non-XML data. </li>
     *  <li> not null, the data may be valid XML. There is no guarantee, that the
     *      schema exists in tamino </li>
     * </ul>
     *
     * @exception  XException when underlying service reports an error
     *
     * @return     the schema for this content object or null.
     */
    String getSchema () throws XException;
    
    /**
     * length accessor.
     *
     * @return     the length of the content
     */
    long getLength();
    
    /**
     * set length.
     *
     *@param length sets the length of this Content object
     */
    void setLength (long length);
    
    /**
     * contentId accessor. With the contentId DbSession is able to find the content.
     *
     * @return     the contentId
     */
    String getContentId ();
    
    /**
     * contentId mutator
     *
     * @pre        (id != null)
     *
     * @param      id the content id of this content object
     */
    void setContentId (String id);
    
    /**
     * retrieves the content as InputStream.
     *
     * @return     the InputStream representing the content
     *
     * @exception  XException when underlying service reports an error
     */
    InputStream streamContent () throws XException;
    
    /**
     * Retrieves the unique name for this content. Consists of uri and revision.
     *
     * @return     the unique name for this content
     *
     */
    String getRevisionName ();
    
    /**
     * Retrieves the MIME type of this content.
     *
     * @return     the unique name for this content
     *
     */
    String getDocumentType ();
   /**
     * Returns the ContentWrapper.
     *
     * @pre        true
     * @post       true
     *
     * @return     The content wrapper.
     *
     */
    XContentWrapper getContentWrapper ();
    
    /**
     * Method isEmptyContent
     *
     * @return   a boolean
     *
     */
    boolean isEmptyContent ();
    
    /**
     *  Buffer the content so that this object can be cached.
     * @return content as byte array
     */
    byte[] bufferContent();
    
    /**
     * Method setIsXml
     *
     * @param    isXml               a  boolean
     *
     */
    void setIsXml (boolean isXml);
    
    /**
     * Method getIsXml
     *
     * @return   a boolean
     *
     */
    boolean getIsXml ();
}

