/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/common/IContentCache.java,v 1.3 2004/07/30 06:51:52 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:52 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.common;

import java.util.Collection;


/**
 * The content cache is a sub-component of the content handler and supports
 * caching of small content for optimization reasons. XContentCache
 * avoids out-of-memory exceptions by using techniques such as soft
 * references and LRU queues. Within the cache, content instances are
 * uniquely identified by a CONTENT_ID.
 *
 * @author    peter.nevermann@softwareag.com
 *
 * @version   $Revision: 1.3 $
 *
 * @see       IContent
 */
public interface IContentCache {

    /**
     * The initial capacity for the global cache
     */
    final int INITIAL_CAPACITY = 11;
    
    /**
     * Initializes this component.
     */
    void initialize();
    
    /**
     * Get the objects identified by given CONTENT_ID.
     *
     * @pre        contentId != null
     * @post
     *
     * @param      contentId   the CONTENT_ID
     *
     * @return     the object identified by contentId; null, if not found.
     */
    IContent get( String contentId );
    
    /**
     * Get all objects contained in cache.
     *
     * @pre
     * @post
     *
     * @return     collection of all objects contained in cache;
     *             empty collection if none.
     */
    Collection getAll();
    
    /**
     * Check whether the objects identified by given content ID is contained in cache.
     *
     * @pre        contentId != null
     * @post
     *
     * @param      contentId   the content ID
     *
     * @return     true, if object identified by contentId is contained;
     *             false, otherwise.
     */
    boolean contains( String contentId );
    
    /**
     * Add the given object to cache. If the cache previously contained
     * an entry for the content ID in question, this will be displaced.
     *
     * @pre        d != null
     * @pre        d.getUri() != null
     * @post       d == get( d.getUri() )
     *
     * @param      d   the object
     */
    void add( IContent d );
    
    /**
     * Add all objects in the given collection to cache. If the cache previously
     * contained entries for some of the content IDs in question, these will be displaced.
     *
     * @pre        pre of add for each d in dl
     * @post       post of add for each d in dl
     *
     * @param      dl   the collection of objects
     */
    void addAll( Collection dl );
    
    /**
     * Remove the objects identified by given content ID.
     *
     * @pre        contentId != null
     * @post       get( contentId ) == null
     *
     * @param      contentId   the content ID
     *
     * @return     the (removed) object identified by contentId; null, if not found.
     */
    IContent remove( String contentId );
    
    /**
     * Remove all objects contained in cache.
     *
     * @pre
     * @post       isEmpty() == true
     */
    void removeAll();
}
