/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/common/IContentHandler.java,v 1.3 2004/07/30 06:51:52 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:52 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.common;

import org.apache.slide.content.NodeRevisionContent;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.util.XException;


/**
 * A thin layer that routes CRUD calls from XContentStore to the correct XDbSession.
 * This component also exists for symmetry reasons (XDescriptorsHandler). If we�d
 * decide to implement caching for content objects, this would be the place for.
 *
 * @author    martin.wallmer@softwareag.com
 *
 * @version   $Revision: 1.3 $
 */
public interface IContentHandler {

    /**
     * Initializes this component.
     */
    public void initialize();

    /**
     ** Returns true if this store is in repair mode (i.e. read_only).
     ** @return     true if this store is in repair mode
     **/
    public boolean isInRepairMode();

    /**
     ** Set repair mode for this store .
     ** @param     onOpenTA 0=WAIT, 1=ERROR, 2=ROLLBACK
     ** @param     waitTimeout the timeout for onOpenTA=0 (WAIT)
     * @exception  XException when underlying service reports an error
     **/
    public void setRepairMode( int onOpenTA, long waitTimeout ) throws XException;

    /**
     ** Release repair mode for this store.
     * @exception  XException when underlying service reports an error
     **/
    public void releaseRepairMode() throws XException;

    /**
     * Creates a NodeRevisionContent object within datastore.
     *
     * @pre        (nrc != null)
     *
     * @param      nrc the NodeRevisionContent object to be made persistent
     * @param      nrd the NodeRevisionDescriptor object for this content
     *
     * @return     the IContent object wrapping the NodeRevisionContent object
     *
     * @exception  XException when underlying service reports an error
     */
    IContent create (NodeRevisionContent nrc, NodeRevisionDescriptor nrd) throws XException;

    /**
     * Retrieves an IContent object from the datastore.
     *
     * @pre        contentId is unique within datastore
     *
     * @param      contentId that identifies the content within datastore
     *
     * @return     the IContent object wrapping the NodeRevisionContent object
     *
     * @exception  XException when underlying service reports an error
     */
    IContent read (String contentId) throws XException;

    /**
     * Updates a NodeRevisionContent object within datastore.
     *
     * @pre        (nrc != null)
     * @pre         contentId exists in datastore
     *
     * @param      contentId identifying the object in datastore
     * @param      nrc the NodeRevisionContent object to be updated within datastore
     * @param      nrd the NodeRevisionDescriptor object for this content
     *
     * @return     the IContent object wrapping the NodeRevisionContent object
     *
     * @exception  XException when underlying service reports an error
     */
    IContent update (String contentId, NodeRevisionContent nrc,
                     NodeRevisionDescriptor nrd) throws XException;

    /**
     * Deletes a NodeRevisionContent object from datastore.
     *
     * @pre         contentId exists in datastore
     *
     * @param      contentId identifying the object in datastore
     *
     * @exception  XException when underlying service reports an error
     */
     void delete (String contentId) throws XException;

    /**
     * <p>Commit the global transaction associated to the current thread.
     * <p>If database session is still available: commit database and clean-up session.
     * <p>Clean-up delta cache: remove entries marked deleted; call commitEvent for
     * non-deleted entries and move them to the global cache; remove delta cache.
     *
     * @param      onePhase   If true, a one-phase commit protocol should be used to commit
     *                        the work done within the current thread.
     *
     * @exception  XException
     */
    void commit( boolean onePhase ) throws XException;

    /**
     * <p>Roll back work done on behalf of a transaction branch.
     * <p>If database session is still available: rollback database and clean-up session.
     * <p>Clean-up delta cache: remove all entries; remove delta cache.
     *
     * @exception  XException
     */
    void rollback() throws XException;

    /**
     * is called in the prepare step of the two phase commit.
     *
     * @exception  XException
     */
    void prepare () throws XException;

}
