/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/common/IDescriptorsHandler.java,v 1.3 2004/07/30 06:51:52 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:52 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.common;

import org.apache.slide.util.XException;

/**
 * <p>
 * The Descriptors Handler controls caching and storage of descriptors
 * objects. As all metadata aspects of a resource are contained in a
 * single descriptors object, the descriptors handler has to collect all
 * aspects first before sending the data to underlying database. To do so,
 * the descriptors handler keeps uncommitted descriptors in a special
 * cache called "delta cache". A separate delta cache is required in each
 * thread. Moreover, the decriptors handler can cache committed descriptors in a
 * global cache in order to reduce the number of accesses to the database.</p>
 * <p>
 * The Descriptors Handler mainly exports an interface consisting of a get
 * method, extended by transaction handling methods  (prepare, commit,
 * rollback).</p>
 * <p>
 * The "get" method supports specific semantics and never returns null.
 * Creation and deletion of descriptors objects is handled automatically.
 * To create a new metadata aspect, the store has to get the descriptors
 * object first. Then, it can use the corresponding set and remove methods
 * to create its store-specific aspect within it. So, even if the store is
 * creating its aspect, the descriptors object may already exist as other
 * stores have already created their aspects into it. For update the
 * situation is similar. Each store has to get the descriptors object
 * first. Then, it can update its store-specific aspect. Descriptors
 * objects are automatically marked deleted when all containing aspects
 * have been removed. At commit time, such objects are then deleted from
 * the database and removed from the caches.</p>
 * <p>
 * <b>Transaction handling.</b><br>
 * The Descriptors Handler, at commit time, needs to coordinate with the Content
 * Handler. The handler called first performs commit and cleans-up the database
 * session. The behavior of the Descriptors Handler at prepare and commit time is
 * specified as follows:
 * <li><b>create/update/delete:</b> descriptors objects are kept in delta cache
 * <li><b>prepare:</b> descriptors objects are flushed to the database
 * <li><b>commit:</b> delta cache is cleaned-up; if the DB Session has not yet been
 *                    cleaned-up, commit is sent to Tamino.
 * </p>
 *
 * @author    peter.nevermann@softwareag.com
 *
 * @version   $Revision: 1.3 $
 */
public interface IDescriptorsHandler {

    /**
     * The initial capacity for the pool of delta caches
     */
    final int INITIAL_CAPACITY = 11;


    /**
     * Initializes this component.
     */
    public void initialize();

    /**
     ** @return true if binding is enabled for the underlying store
     **/
    public boolean useBinding();
    
    /**
     ** Returns true if this store is in repair mode (i.e. read_only).
     ** @return     true if this store is in repair mode
     **/
    public boolean isInRepairMode();

    /**
     ** Set repair mode for this store .
     ** @param     onOpenTA 0=WAIT, 1=ERROR, 2=ROLLBACK
     ** @param     waitTimeout the timeout for onOpenTA=0 (WAIT)
     ** @exception XException xdav error
     **/
    public void setRepairMode( int onOpenTA, long waitTimeout ) throws XException;

    /**
     ** Release repair mode for this store.
     ** @exception XException xdav error
     **/
    public void releaseRepairMode() throws XException;

    /**
     * <p>
     * Get descriptors object for the requested Unique URI.
     * The access sequence for get is specified as follows:
     * <li> search in delta cache for the current thread; if not found...</li>
     * <li> search in global cache; if not found...</li>
     * <li> retrieve from database; if not found...</li>
     * <li> create a new descriptors object</li></p>
     * <p>
     * Moreover, the following holds:
     * If (readOnly == true) and the descritors object was...
     * <li> found in delta cache: return it! (Note, the returned object is modifiable!)
     * <li> found in global cache: return it!
     * <li> retrieved from database: add it to the global cache, mark it read-only and
     *      return it!
     * <li> created: add it to the global cache, mark it read-only and
     *      return it! (<b>this is to fulfill "never returns null" - makes sense?</b>)</p>
     * <p>
     * Else If (readOnly == false) and the descritors object was...
     * <li> found in delta cache: return it!
     * <li> found in global cache: clone it, mark clone modifiable, add clone to delta
     *      cache and return the clone!
     * <li> retrieved from database: add it to global cache, clone it, mark clone as
     *      modifiable, add clone to delta cache and return the clone!
     * <li> created: add it to global cache, clone it, mark clone as
     *      modifiable, add clone to delta cache and return the clone!</p>
     *
     * @pre        uri != null
     * @post
     *
     * @param      uri        the requested URI
     *
     * @return     the requested descriptors object; null if not found
     *
     * @exception  XException xdav error
     */
    public IDescriptors lookup( String uri, String uuri, int lockType ) throws XException;
    public IDescriptors lookupUriCache( int lockType, String uri ) throws XTLockedException;
    public IDescriptors create(String uri) throws XConflictException;

    /**
     * Prepare for a transaction commit of the transaction associated to the current
     * thread. Flush uncommitted descriptors objects to the database.
     *
     * @exception  XException xdav error
     */
    void prepare() throws XException;

    /**
     * <p>Commit the global transaction associated to the current thread.
     * <p>If database session is still available: commit database and clean-up session.
     * <p>Clean-up delta cache: remove entries marked deleted; call commitEvent for
     * non-deleted entries and move them to the global cache; remove delta cache.
     *
     * @param      onePhase   If true, a one-phase commit protocol should be used to commit
     *                        the work done within the current thread.
     *
     * @exception  XException xdav error
     */
    void commit( boolean onePhase ) throws XException;

    /**
     * <p>Roll back work done on behalf of a transaction branch.
     * <p>If database session is still available: rollback database and clean-up session.
     * <p>Clean-up delta cache: remove all entries; remove delta cache.
     *
     * @exception  XException xdav error
     */
    void rollback() throws XException;
}


