/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/IDbSession.java,v 1.3 2004/07/30 06:51:54 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:54 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.datastore;

import java.util.Collection;
import org.apache.slide.store.tamino.common.IContent;
import org.apache.slide.store.tamino.common.IDescriptors;
import org.apache.slide.store.tamino.store.ISlideAccessor;
import org.apache.slide.util.XException;

/**
 * Offers a kind of CRUD interface for IContent and IDescriptors objects,
 * as well as methods for transactional behaviour.
 * This interface hides the physical store. The implementing class is the
 * only one to talk to the physical store (Tamino).
 *
 * @author martin.wallmer@softwareag.com
 *
 * @version $Revision: 1.3 $
 */
public interface IDbSession {

    /**
     * Set the DB handler maaging this session.
     *
     * @pre        dbHandler != null
     * @post
     *
     * @param      dbHandler the DB handler for this session
     */
    public void setDbHandler( XDbHandler dbHandler );

    /**
     * triggers the DBSession to retrieve its parameters from slide
     *
     * @pre        slide != null
     * @post
     *
     * @param      slide the SlideAccessor from where to retrieve
     * @exception  XException xdav error
     */
    public void setParameter (ISlideAccessor slide) throws XException ;

    /**
     * <p>Initialize this DB session.
     * <p>Checks the existence of the descriptors schema associated to the current tamino
     * collection. The schema is created if not found.
     *
     * @exception  XException xdav error
     */
    public void initialize () throws XException;

    /**
     * writes the content of a resource to database. The content is flushed
     * to database, but not yet committed.
     *
     * @pre        content != null
     * @post
     *
     * @param      content   the content object to be made persistent
     *
     * @return     the content (possibly slightly modified by tamino)
     *
     * @exception  XException xdav error
     */
    public IContent createContent (IContent content) throws XException;

    /**
     * reads the content of a resource from database
     *
     * @pre        (contentId != null and !(contentId.equals ("")))
     * @post
     *
     * @param      contentId   the content object to be made persistent
     *
     * @return     IContent object representing the content
     *
     * @exception  XException xdav error
     */
    public IContent readContent (String contentId) throws XException;

    /**
     * updates a content object within the database. The new content is flushed
     * to the database, but not yet committed
     *
     * @pre        (contentId != null)
     * @pre        (!(contentId.equals ("")))
     * @pre        (content != null)
     * @post       true
     *
     * @param      content     the content object to be made persistent
     * @param      contentId   the contentId identifying the content object
     * @return     the content (possibly slightly modified by tamino)
     *
     * @exception  XException xdav error
     */
    public IContent updateContent (IContent content, String contentId)
        throws XException;

    /**
     * deletes a content object within the database, not yet committed
     *
     * @pre        (contentId != null)
     * @pre        (!(contentId.equals ("")))
     * @post       true
     *
     * @param      contentId   the contentId identifying the content object
     *
     * @exception  XException xdav error
     */
    public void deleteContent (String contentId)
        throws XException;

    /**
     * reads a descriptors object from databse
     *
     * @pre        (uuri != null)
     * @post
     *
     * @param      uuri   the uuri identifying this resource
     * @return     the descriptors
     * @exception  XException xdav error
     */
    public IDescriptors readDescriptor (String uuri)
        throws XException;

    /**
     * Processes a collection of descriptors objects. The action on each element
     * (delete, update, create) depends on the object's state.
     *
     * @pre        (descriptors != null)
     * @post
     *
     * @param      descriptors   a collection of IDescriptorsDocument objects
     *
     * @exception  XException xdav error
     *
     * @see IDescriptors
     */
    public void processDescriptors (Collection descriptors)
        throws XException;

    /**
     * commits a database transaction
     *
     * @param onePhase indicates if one- or two phase commit
     *
     * @exception  XException
     */
    public void commit (boolean onePhase)
        throws XException;

    /**
     * commits a database transaction. One phase commit is assumed.
     *
    * @exception  XException xdav error
     */
    public void commit ()
        throws XException;

    /**
     * aborts a database transaction
     *
     * @exception  XException xdav error
     */
    public void rollback ()
        throws XException;

    /**
     * prepare a database transaction
     *
     * @exception  XException xdav error
     */
    public void prepare ()
        throws XException;

    /**
     * Set the repair mode
     * @param  repairMode true/false
     */
    public void setRepairMode( boolean repairMode );
}
