/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/XConnectionKey.java,v 1.4 2005/01/07 14:58:17 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/01/07 14:58:17 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore;

import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.store.tamino.store.ISlideAccessor;
import org.apache.slide.store.tamino.tools.stores.XDomainFileHandler;
import org.apache.slide.store.tamino.tools.stores.XNamespaceConfig;
import org.apache.slide.util.XException;
import org.apache.slide.util.XUri;

/**
 * Unique key for an XConnection. Comprises of threadname, dbUri and
 * tamino user
 *
 * @author martin.wallmer@softwareag.com
 *
 * @version $Revision: 1.4 $
 *
 */
public class XConnectionKey implements XGlobals {
    private String key;
    private String databaseUri;
    private String domain;
    private String user;
    private String pwd;

    private static int counter = 0;

    /**
     * Create an XConnectionKey.
     *
     * @param databaseUri the databaseUri
     * @param user        the taminoo user
     * @param pwd         the user pwd
     *
     */
    public XConnectionKey(String databaseUri, String domain, String user, String pwd) {
        init (databaseUri, domain, user, pwd, null);
    }

    /**
     * Creates a connectionKey with a private key
     *
     * @param    databaseUri         a  String
     * @param    domain              a  String
     * @param    user                a  String
     * @param    pwd                 a  String
     * @param    privateKey          a  String
     *
     */
    public  XConnectionKey (String databaseUri, String domain,
                            String user, String pwd,
                            String privateKey) {

        init(databaseUri, domain, user, pwd, privateKey);
    }

    /**
     * Create an XConnectionKey.
     *
     * @param slide    the store description
     */
    public XConnectionKey (ISlideAccessor slide) throws XException {
        this (slide, null);
    }

    public XConnectionKey(ISlideAccessor slide, String privateKey) throws XException {
        String databaseUri = slide.getParameter(TAMINO_BASE);
        String database   = slide.getParameter(TAMINO_DATABASE);
        StringBuffer sb = new StringBuffer (databaseUri);
        if (!databaseUri.endsWith (XUri.SEP))
            sb.append (XUri.SEP);
        sb.append (database);

        String namespace = slide.getNamespaceName();
        XNamespaceConfig config = XDomainFileHandler.get().getNamespace(namespace).getConfig();
        String s_user = slide.getParameter(USER_ID);
        String user = s_user == null ? config.getAdminUser() : s_user;
        String s_password = slide.getParameter(PASSWORD);
        String pwd = s_password == null ? config.getAdminPwd() : s_password;

        String s_domain = slide.getParameter(TAMINO_DOMAIN);
        String domain = null;
        if (s_user != null)  domain = s_domain ==null ? "" : s_domain;
        else domain = s_domain ==null ? config.getAdminDomain() : s_domain;

        init (sb.toString(), domain, user, pwd, privateKey);
    }


    /**
     * initializes this XConnectionKey. If privateKey is set (not null),
     * this string is appended to the key to guarantee to get a "private"
     * connection.
     *
     * @param    databaseUri         a  String
     * @param    domain              a  String
     * @param    user                a  String
     * @param    pwd                 a  String
     * @param    isAdmin             a  boolean
     * @param    privateKey          a  String
     *
     */
    private void init (String databaseUri, String domain,
                       String user, String pwd, String privateKey)
    {
        this.databaseUri = databaseUri;
        this.user = user;
        this.pwd = pwd;
        this.domain = domain;

        String threadId = Thread.currentThread().getName();
        key = threadId + "@" + databaseUri + "@" + domain + "@" + user;
        if (privateKey != null) {
            key += "@" + privateKey;
        }

        counter++;
    }



    /**
     * Method equals
     *
     * @param    other               the other object to compare against
     *
     * @return   true if this returns the same hashCode as the other object
     *
     */
    public boolean equals (Object other) {
        boolean result = false;
        if( other instanceof XConnectionKey )
            result = ((XConnectionKey)other).key.equals( key );
        return result;
    }

    /**
     * Method hashCode
     *
     * @return   an int
     *
     */
    public int hashCode() {
        return key.hashCode();
    }

    /**
     * Creates a unique pool name for the associated TConnectionPool
     *
     * @return   a String
     *
     */
    public String getTPoolName () {
        return user + '@' + databaseUri;
    }

    /**
     * databaseUri accessor.
     *
     * @return   the databaseUri
     *
     */
    public String getDatabaseUri () {
        return databaseUri;
    }

    /**
     * Method getDomain
     *
     * @return   the domain
     *
     */
    public String getDomain () {
        return domain;
    }

    /**
     * user accessor.
     *
     * @return   the user
     *
     */
    public String getUser () {
        return user;
    }

    /**
     * pwd accessor.
     *
     * @return   the password
     *
     */
    public String getPwd () {
        return pwd;
    }

    /**
     * Method toString
     *
     * @return   String representation of this key
     *
     */
    public String toString () {
        return key;
    }
}

