/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/XUtilDbSession.java,v 1.3 2004/07/30 06:51:55 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:55 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.datastore;

import com.softwareag.common.instrumentation.logging.Level;
import com.softwareag.common.instrumentation.logging.Logger;
import com.softwareag.common.instrumentation.logging.LoggerFactory;
import com.softwareag.common.instrumentation.logging.LoggerUtil;
import com.softwareag.tamino.db.api.common.TException;
import org.apache.slide.store.tamino.tools.nixe.configure.NixeAdmin;
import org.apache.slide.store.tamino.tools.nixe.configure.NixeConfigException;
import org.apache.slide.store.tamino.common.XDatastoreException;
import org.apache.slide.store.tamino.datastore.schema.XSchemaFactory;
import org.apache.slide.util.ClassName;
import org.apache.slide.util.XException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;


/**
 * Offers some helpfull Tamino access methods used internally
 * or by tools.
 *
 * @author martin.wallmer@softwareag.com
 *
 * @version $Revision: 1.3 $
 */
public class XUtilDbSession extends XDbSession {

    private static final String LOGNAME = LoggerUtil.getThisClassName();
    private static final String CLASSNAME = new ClassName(LOGNAME).getPlainName();
    private static Logger logger = LoggerFactory.getLogger(LOGNAME);

    /**
     * Creates the schema for desrciptors in Tamino. The schema has the name of the
     * content collection and is stored in xdav:metadata.
     *
     * @pre        (descriptors != null)
     * @post
     *
     * @param      contentCollection the name of the content collection to be webDAV enabled.
     *
     * @exception  XException
     */
    public void createMetaDataSchema (String contentCollection) throws XException {
        if (logger.isLoggable (Level.FINE))
            logger.entering (CLASSNAME, "createMetaDataSchema", new Object[] {contentCollection});

        utilAccessor.createMetaDataSchema(contentCollection);

        if (logger.isLoggable(Level.FINE))
            logger.exiting (CLASSNAME, "createMetaDataSchema");
    }

    /**
     * creates the schema for nonXML data within the content collection.
     *
     * @pre        (descriptors != null)
     * @post
     *
     * @param      contentCollection the name of the content collection to be webDAV enabled.
     *
     * @exception  XException
     */
    public void createNonXMLSchema (String contentCollection) throws XException {
        if (logger.isLoggable (Level.FINE))
            logger.entering (CLASSNAME, "createNonXMLSchema", new Object[] {contentCollection});

        try {
            if (useNixe) {

                NixeAdmin nixe = new NixeAdmin (xConnection.getTConnection());
                if (!nixe.isNixeInstalled()) {
                    throw new XException ("Non XML Indexer is not installed.");
                }

                Document [] nixeSchemas = nixe.getNixeSchemasAsDocuments
                    (contentCollection, NONXML_CONTENT);

                for (int i = 0; i < nixeSchemas.length; i++) {
                    utilAccessor.createSchema (contentCollection, nixeSchemas [i]);
                }
            }
            else {
                utilAccessor.createSchema(contentCollection,
                                          XSchemaFactory.getNonXMLContentSchema
                                              (contentCollection, NONXML_CONTENT)
                                         );
            }
        }
        catch (TException e) {
            throw new XException (e);
        }
        catch (NixeConfigException e) {
            throw new XException (e);
        }

        if (logger.isLoggable(Level.FINE))
            logger.exiting (CLASSNAME, "createNonXMLSchema");
    }


    /**
     * Method getFullQualifiedCollection
     *
     * @return   an IFullQualifiedCollection
     *
     */
    public IContentCollection getContentCollection () {

        return new ContentCollectionImpl (this.contentCollection);
    }

    public XUtilDBAccessor getUtilDBAccessor() {
        return utilAccessor;
    }

    /**
     *
     */
    private class ContentCollectionImpl implements IContentCollection {
        private String collection;


        ContentCollectionImpl (String collection) {
            this.collection = collection;
        }

        /**
         * Method getCollectionName
         *
         * @return   a String
         *
         */
        public String getCollectionName() {
            return collection;
        }

        /**
         * Method getFullQualifiedName
         *
         * @return   a String
         *
         */
        public String getUri() {
            return taminoBase + "/" + database + "/" + collection;
        }


        /**
         * Method getSchemaVersion
         *
         * @return   a String
         *
         */
        public String getSchemaVersion() throws XDatastoreException {
            return getSchemaLanguage (collection);
        }
    }

    //--

    private static final String ID = "id";
    public static final Namespace NS = Namespace.getNamespace("ino", "http://namespaces.softwareag.com/tamino/response2");

    /** @return null for no id */
    public static String getInoId(Element element) {
        return element.getAttributeValue(ID, NS);
    }

    /** @return null for no id */
    public static void setInoId(Element element, String id) {
        if (id == null) {
            element.removeAttribute(ID, NS);
        } else {
            element.setAttribute(ID, id, NS);
        }
    }
}



