/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/schema/XDoctype.java,v 1.3 2004/07/30 06:51:56 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:56 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore.schema;
import org.apache.slide.store.tamino.common.XGlobals;


/**
 * <p>Represents meta information for doctypes as defined in Tamino. A doctype
 * consists of a name, a namespace uri and a namespace prefix. A doctype is
 * uniquely identified by the tuple name and uri. </p>
 * <p>However, within one collection the tuple prefix:name must be unique as
 * well. So a XDoctype may be "strong" (that means "certified" against Tamino)
 * If it is weak, there is no guarantee, that pre:name is unique. </p>
 *
 * @author martin.wallmer@softwareag.com
 *
 * @version $Revision: 1.3 $ *
 */

public class XDoctype {
    private boolean isWeak = true;
    
    boolean isWeak() {
        return isWeak;
    }
    
    protected String prefixedName;
    protected String name;
    protected String namespacePrefix;
    protected String namespaceUri;
    protected String fullQualifiedName;
    
    protected XDoctype ()  {};
    
    /**
     * Constructor
     *
     * @param    prefixedName               a  String in the format prefix:name
     *
     */
    public XDoctype (String prefixedName, String namespaceUri) {
        this (prefixedName, namespaceUri, true);
    }
    
    /**
     * Constructor
     *
     * @param    prefixedName        a  String in the format prefix:name
     * @param    namespaceUri        the namespaceUri
     *
     */
    public XDoctype (String prefixedName, String namespaceUri, boolean isWeak) {
        
        this.isWeak = isWeak;
        
        //      if (namespaceUri == null || namespaceUri.length() == 0)
        //          namespaceUri = XGlobals.DUMMY_NAMESPACE_URI;
        
        int pos = prefixedName.indexOf (':');
        if (pos != -1) {
            namespacePrefix = prefixedName.substring (0, pos);
            name = prefixedName.substring (pos + 1);
        }
        else if (namespaceUri != null && namespaceUri.length() > 0) {
            this.name = prefixedName;
            namespacePrefix = XGlobals.DUMMY_NAMESPACE_PREFIX;
            prefixedName = namespacePrefix + ":" + prefixedName;
        }
        else {
            name = prefixedName;
            namespacePrefix = null;
        }
        this.namespaceUri = namespaceUri;
        this.prefixedName = prefixedName;
    }
    
    
    /**
     * Method makeStrong
     *
     * @param    prefix              a  String
     *
     */
    void makeStrong (String prefix) {
        namespacePrefix = prefix;
        if (prefix != null)
            prefixedName = namespacePrefix + ":" + name;
        else
            prefixedName = name;
        
        isWeak = false;
    }
    
    
    
    /**
     * Method getNamespacePrefix
     *
     * @return   a String
     *
     */
    public String getNamespacePrefix() {
        return namespacePrefix;
    }
    
    /**
     * Method getName
     *
     * @return   a String
     *
     */
    public String getName() {
        return name;
    }
    
    /**
     * Method getNamespaceUri
     *
     * @return   a String
     *
     */
    public String getNamespaceUri() {
        return namespaceUri;
    }
    
    /**
     * Method getPrefixedName
     *
     * @return   a String
     *
     */
    public String getPrefixedName() {
        return prefixedName;
    }
    
    
    /**
     * If a namespace URI is set, returns namespaceuri:name,
     * else the prefixed name
     *
     * @return   a String
     *
     */
    public String getFullQualifiedName() {
        
        if (fullQualifiedName == null) {
            
            if (namespaceUri != null && namespaceUri.length() > 0)
                fullQualifiedName =  name + "@" + namespaceUri;
                
            else
                fullQualifiedName = prefixedName;
        }
        
        return fullQualifiedName;
    }
    
}


