/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/schema/XDoctypeHandler.java,v 1.3 2004/07/30 06:51:56 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:56 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.datastore.schema;

import com.softwareag.tamino.db.api.connection.TConnection;
import java.util.Iterator;
import java.util.Set;
import org.apache.slide.store.tamino.common.XDatastoreException;
import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.store.tamino.datastore.IContentCollection;
import org.apache.slide.store.tamino.datastore.XUtilDBAccessor;
import org.apache.slide.store.tamino.datastore.schema.XSchema;
import org.apache.slide.util.XException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;

/**
 * returns a content doctype from the cache. If not in cache and autocreate schema
 * allowed, the doctype is created in Tamino, put to cache and returned
 *
 * @author martin.wallmer@softwareag.com
 *
 * @version $Revision: 1.3 $ *
 */
public class XDoctypeHandler {

    private XDoctypeCache doctypeCache = XDoctypeCache.getSingleInstance();
    private TConnection tCon;

    public XDoctypeHandler (TConnection tCon) {
        this.tCon = tCon;
    }

    /**
     * Method getDoctype
     *
     * @param    collection          a  String
     * @param    prefixedName        a  String
     * @param    namespaceUri        a  String
     *
     * @return   a XDoctype
     *
     * @throws   XException
     *
     */
    public XDoctype getDoctype (IContentCollection collection,
                                String prefixedName,
                                String namespaceUri,
                                boolean autocreateSchema)
        throws XException {

        return getDoctype (collection,
                           new XDoctype (prefixedName, namespaceUri),
                           autocreateSchema);
    }


    public XDoctype getDoctype (IContentCollection collection,
                                XDoctype weakDoctype,
                                boolean autocreateSchema)
        throws XException {
        XDoctype result =
            doctypeCache.getDoctype (collection,
                                     weakDoctype.getFullQualifiedName());

        if (result == null && autocreateSchema) {

            XDoctype strongDoctype = doctypeCache.createStrongDoctype
                (collection, weakDoctype);


            XUtilDBAccessor utilDb = new XUtilDBAccessor (tCon);

            String colName = collection.getCollectionName();
            String prefixedName = strongDoctype.getPrefixedName();
            String namespaceUri = strongDoctype.getNamespaceUri();

            utilDb.createSchema (colName,
                                 prefixedName,
                                 namespaceUri);

            result = strongDoctype;
            doctypeCache.addDoctype (collection, result);
        }

        return result;
    }

    /**
     * reads all schemas defined in the specified collection, gets the doctypes
     * and put them in the cache. Call this once for each webdav enabled collection.
     *
     * @param    coll                an IFullQualifiedCollection
     *
     * @throws   XDatastoreException in case of DB errors.
     *
     */
    public void initCache (IContentCollection coll) throws XDatastoreException {
        XUtilDBAccessor utilDb = new XUtilDBAccessor (tCon);
        Namespace xqlNamespace = Namespace.getNamespace (XGlobals.XQL_NAMESPACE_URI);

        Iterator it = utilDb.getSchemas(coll.getCollectionName()).iterator();

        while (it.hasNext()) {
            Document resultDoc = (Document) it.next();

            Element xqlResult = resultDoc.getRootElement().
                getChild ("result", xqlNamespace);

            Element schemaElement = (Element)xqlResult.getChildren().get (0);

            XSchema xSchema = XSchema.newInstance (schemaElement);
            Set docTypes = xSchema.getDoctypes();
            XDoctypeCache.getSingleInstance().addDoctype (coll, docTypes);
        }
    }
}


