/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/search/XCompareExpression.java,v 1.3 2004/07/30 06:51:58 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:58 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore.search;

import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.store.tamino.datastore.schema.XKnownPropertyHelper;
import org.apache.slide.search.InvalidQueryException;
import org.apache.slide.search.basic.IBasicExpression;
import org.apache.slide.search.basic.expression.ComparedProperty;
import org.jdom.Element;

/**
 *  Class for compare expressions (property compares, contains).
 *
 * @author <a href="mailto:martin.wallmer@softwareag.com">Martin Wallmer</a>
 * @version $Revision: 1.3 $
 */
public class XCompareExpression extends XPropertyExpression {

    /** the operator for this compare, i. e. "=" */
    protected String operator;

    /** the property to be compared */
    protected ComparedProperty comparedProperty;

    /** indicates, if this is not-xxx expression */
    protected boolean isNot;

    /**
     * dummy constructor, called by is-collection (as no prop / literal
     * is passed).
     */
    XCompareExpression () {}


    /**
     * Creates a compare expression according to Element e
     *
     * @param e jdom element, that describes the expression
     * @param operator the operator for this compare
     *
     */
    XCompareExpression (Element e, String operator) throws InvalidQueryException {
        comparedProperty = new ComparedProperty (e);
        this.operator = operator;
    }

    /**
     * Creates a compare expression according to Element e
     *
     * @param e jdom element, that describes the expression
     * @param operator the operator for this compare
     * @param isNot if true, a NOT- compare expression is created
     *
     */
    XCompareExpression (Element e, String operator, boolean isNot) throws InvalidQueryException {
        comparedProperty = new ComparedProperty (e);
        this.operator = operator;
        this.isNot = isNot;
    }

    /**
     * creates the filter for this query string
     *
     * @return   an xpath filter
     *
     */
    protected String getXpathFilter() {
        StringBuffer sb = new StringBuffer ();

        String propName = comparedProperty.getProperty();
        String propNamespace = comparedProperty.getPropNamespace();
        boolean isKnown = knownProperties.isKnown (propName, propNamespace);

        createPropNamespace (sb, propName, propNamespace, isKnown);

        if (!isKnown)
            sb.append ("and ");
        else
            sb.append (" [");


        if (isNot) {
            sb.append ("not (");
            sb.append (createExpression ("shadow", XGlobals.MAX_SHADOW_SIZE));
            sb.append (" or ");
            sb.append (createExpression ("text()"));
            sb.append (")");
        }
        else {
            sb.append (createExpression ("shadow", XGlobals.MAX_SHADOW_SIZE));
            sb.append (" and ");
            sb.append (createExpression ("text()"));
        }

        sb.append ("]");
        return sb.toString();
    }

    /**
     * Method createExpression
     *
     * @param    context             a  String
     *
     * @return   a String
     *
     */
    protected String createExpression (String context) {
        return createExpression (context, -1);
    }

    /**
     * Method createExpression
     *
     * @param    context             a  String
     * @param    cutToLen            an int
     *
     * @return   a String
     *
     */
    protected String createExpression (String context, int cutToLen) {
        StringBuffer expression = new StringBuffer ();
        expression.append (context);

        String literal = comparedProperty.getLiteral();

        String op = operator;

        if (cutToLen != -1) {
            if (literal.length() >= XGlobals.MAX_SHADOW_SIZE) {
                literal = literal.substring (0, XGlobals.MAX_SHADOW_SIZE);
            }
            if (operator.equals (XBasicExpressionFactory.OP_GT)) op = XBasicExpressionFactory.OP_GTE;
            if (operator.equals (XBasicExpressionFactory.OP_LT)) op = XBasicExpressionFactory.OP_LTE;
        }

        if (comparedProperty.isNumeric()) {
            expression.append (op);
            expression.append (literal);
        }
        else {
            expression.append (op).append ("\"");
            expression.append (literal);
            expression.append ("\"");
        }
        return expression.toString();
    }

    /**
     * Checks, if another expression may be merged (and, or) with this
     * expression into one Tamino query.
     *
     * @param    otherExpression     an IBasicExpression
     *
     * @return   true if otherExpression is instanceof XCompareExpression
     *
     */
    protected boolean isMergeable (IBasicExpression otherExpression) {

        if (otherExpression instanceof XPropertyExpression)
            return true;
        else
            return false;
    }
}
