/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/search/XMergeExpression.java,v 1.3 2004/07/30 06:51:58 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:58 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore.search;

import java.util.Collection;
import java.util.Iterator;

import org.apache.slide.search.SearchException;
import org.apache.slide.search.BadQueryException;
import org.apache.slide.search.basic.IBasicExpression;

/**
 * Abstract base class for compare expressions (property compares, contains).
 *
 * @author <a href="mailto:martin.wallmer@softwareag.com">Martin Wallmer</a>
 * @version $Revision: 1.3 $
 */
public class XMergeExpression extends XPropertyExpression {

    /** the operator for this compare, i. e. "=" */
    protected String mergeOperator;

    /** expressions to be merged */
    protected Collection children;

    /** the first child within the list of expressions */
    protected XBasicExpression firstChild;

    /**
     * dummy constructor, called by is-collection (as no prop / literal
     * is passed).
     */
    XMergeExpression () {}


    /**
     * Creates a compare expression according to Element e
     *
     * @param mergeOperator the operator for this compare
     *
     */
    XMergeExpression (Collection children, String mergeOperator)
        throws BadQueryException
    {
        this.children = children;
        this.mergeOperator = mergeOperator;

        Iterator it = children.iterator();
        firstChild = (XBasicExpression)it.next();
        if (firstChild == null)
            throw new BadQueryException (mergeOperator + " needs at least one nested element");
    }

    /**
     * creates the filter for this query string
     *
     * @return   an xpath filter
     *
     */
    protected String getXpathFilter() throws SearchException {
        StringBuffer sb = new StringBuffer ();
        Iterator it = children.iterator();

        while (it.hasNext()) {
            sb.append ("(");
            XPropertyExpression exp = (XPropertyExpression)it.next();
            sb.append (exp.getXpathFilter());
            sb.append (")");

            if (it.hasNext())
                sb.append (" ").append (mergeOperator).append (" ");
        }

        return sb.toString();
    }

    /**
     * Checks, if another expression may be merged (and, or) with this
     * expression into one Tamino query. Returns true, if otherExpression is
     * mergeable to the first expression of this expressions children.
     *
     * @param    otherExpression     an IBasicExpression
     *
     * @return   true if mergeable
     *
     */
    protected boolean isMergeable (IBasicExpression otherExpression) {
        boolean result = true;

        if (otherExpression instanceof XBasicExpression) {
            result = firstChild.isMergeable (otherExpression);
        }
        else
            result = false;

        return result;
    }
}
