/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/search/XPropertyContainsExpression.java,v 1.3 2004/07/30 06:51:58 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:58 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore.search;

import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.store.tamino.datastore.schema.XKnownPropertyHelper;
import java.util.Iterator;
import org.apache.slide.search.InvalidQueryException;
import org.apache.slide.search.SearchException;
import org.apache.slide.search.basic.ComparableResource;
import org.apache.slide.search.basic.IBasicExpression;
import org.apache.slide.search.basic.IBasicResultSet;
import org.apache.slide.search.basic.Literals;
import org.jdom.Element;

/**
 * Represents a propcontains expression. propcontains is a not documented
 * extension to DASL.
 *
 * @author <a href="mailto:martin.wallmer@softwareag.com">Martin Wallmer</a>
 * @version $Revision: 1.3 $
 */
public class XPropertyContainsExpression extends XCompareExpression {

    private String unCensoredLiteral;
    private boolean needPostProcessing = false;

    /**
     * Creates a property-contains expression.
     *
     * @param e jdom element, that describes the expression
     *
     */
    protected XPropertyContainsExpression (Element e) throws InvalidQueryException {
        this (e, false);
    }

    /**
     * Creates a property-contains expression.
     *
     * @param e jdom element, that describes the expression
     * @param isNot if true, a not-property-contains expression is created (currently not supported)
     *
     */
    protected XPropertyContainsExpression (Element e, boolean isNot)
        throws InvalidQueryException {
        super (e, XBasicExpressionFactory.OP_LIKE, isNot);

        unCensoredLiteral = comparedProperty.getLiteral();

        StringBuffer lit = new StringBuffer ("*");
        if (isNot) {
            throw new InvalidQueryException
                ("NOT propcontains is not supported");
        }

        if (unCensoredLiteral.length() > XGlobals.MAX_TEXT_SEARCH_SIZE) {
            lit.append (unCensoredLiteral.substring (0, XGlobals.MAX_TEXT_SEARCH_SIZE));
            needPostProcessing = true;
        }
        else {
            lit.append (unCensoredLiteral);
        }
        lit.append ("*");
        comparedProperty.setLiteral (lit.toString());
    }


    /**
     * This implementation does not use the indexed shadow.
     *
     * @return   a String
     *
     */
    protected String getXpathFilter() {
        StringBuffer sb = new StringBuffer ();

        String propName = comparedProperty.getProperty();
        String propNamespace = comparedProperty.getPropNamespace();
        boolean isKnown = knownProperties.isKnown (propName, propNamespace);

        createPropNamespace (sb, propName, propNamespace, isKnown);

        if (!isKnown)
            sb.append ("and ");
        else
            sb.append (" [");

        if (isNot) {
            sb.append ("not (");
            sb.append (createExpression ("text()"));
            sb.append (")");
        }
        else {
            sb.append (createExpression ("text()"));
        }

        sb.append ("]");
        return sb.toString();
    }



    /**
     * If the literal was longer than MAX_TEXT_SEARCH_SIZE, a postselect is done.
     *
     * @return   an IBasicResultSet
     *
     * @throws   SearchException
     *
     */
    public IBasicResultSet execute () throws SearchException {
        IBasicResultSet result = super.execute();
        if (needPostProcessing) {
            Iterator it = result.iterator();
            while (it.hasNext()) {
                ComparableResource resource = (ComparableResource)it.next();
                if (resource.propContains (comparedProperty.getProperty(),
                                           comparedProperty.getPropNamespace(),
                                           unCensoredLiteral)
                    == Literals.FALSE) {
                    it.remove();
                }
            }
        }

        return result;
    }

    /**
     * Method isMergeable
     *
     * @param    otherExpression     an IBasicExpression
     *
     * @return   a boolean
     *
     */
    protected boolean isMergeable (IBasicExpression otherExpression) {
        boolean result;
        if (needPostProcessing)
            result = false;
        else
            result = super.isMergeable(otherExpression);

        return result;
    }
}

