/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/datastore/search/XPropertyExpression.java,v 1.3 2004/07/30 06:51:58 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:58 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.slide.store.tamino.datastore.search;

import org.apache.slide.store.tamino.datastore.schema.XKnownPropertyHelper;
import java.util.List;
import org.apache.slide.search.InvalidQueryException;
import org.apache.slide.search.SearchException;
import org.apache.slide.search.basic.IBasicResultSet;
import org.jdom.Attribute;
import org.jdom.Element;

/**
 * An XBasicExpression represents exactly one executable Tamino query
 * It may have the state resolved or unresolved. If it is resolved, it
 * contains a Set of RequestedResource, that represents the result set of
 * this XBasicExpression.
 *
 * @author <a href="mailto:martin.wallmer@softwareag.com">Martin Wallmer</a>
 * @version $Revision: 1.3 $
 */
public abstract class XPropertyExpression extends XBasicExpression {

    /**
     * prevent from being instantiated from outside, use factory method
     */
    protected XPropertyExpression () {}

    /**
     * constructor. Only called by the conrecte expressions
     *
     * @param e the jdom element representing this expression.
     */
    protected XPropertyExpression (Element e) throws InvalidQueryException {
        // TODO: super(e);
        List attrList = e.getAttributes();
        if (attrList.size() != 0) {
            Attribute attr = (Attribute) attrList.get(0);
            throw new InvalidQueryException (attr.getQualifiedName() +
                                                 " is an unprocessable entity");
        }
    }

    /**
     * executes this basic expression. It must be guaranteed, that all
     * subsequent expressions can be expressed in one Tamino Query.
     *
     * @return   an IBasicResultSet
     *
     * @throws   SearchException
     *
     */
    public IBasicResultSet execute () throws SearchException {

        String queryString = createQueryString (query.getSlidePath());
        IQueryExecuter executer = new XQueryExecuter (slideAccessor, query);

        return postProcessor (executer.execute (queryString));
    }

    /**
     * Creates the query string.
     *
     * @param    slidePath           a  String
     *
     * @return   a String
     *
     * @throws   InvalidScopeException
     *
     */
    protected String createQueryString (String slidePath) throws SearchException {
        StringBuffer sb = new StringBuffer ();

        sb.append (getCollectionPrefix());

        sb.append (" ");

        sb.append ("[");
        sb.append (getScopeFilter (slidePath));
        sb.append (" and (");
        sb.append (getXpathFilter());
        sb.append (")] ");
        sb.append (getSortBy ());

        return sb.toString();
    }
    
    private String getSortBy() {

        XOrderBy ob = (XOrderBy) query.getOrderBy();

        if (ob != null && ob.isMetaPropertyOrder ())
            return ob.getMetaSortByExpression ();

        return "";
    }

    /**
     * creates the common part of a property expression. Knows about the
     * different location of indexed and generic properties.
     *
     * @param    sb                  a  StringBuffer
     * @param    name                name of the property
     * @param    namespace           namespace of the property
     *
     */
    protected void createPropNamespace (StringBuffer sb, String name,
                                        String namespace, boolean isKnown) {

         if (isKnown) {
            String prefix = knownProperties.getPrefix (namespace);
            sb.append ("//propertyList/");
            sb.append (prefix).append(":");
            sb.append (name);
        }
        else {
            sb.append ("//nodeProperty[@name=\"");
            sb.append (name);
            sb.append ("\" and ");

            sb.append ("@namespace=\"");
            sb.append (namespace);
            sb.append ("\" ");
        }
    }

    /**
     * Method getXpathFilter
     *
     * @return   a String
     *
     */
    abstract String getXpathFilter () throws SearchException;

}

