/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/report/XAbstractReport.java,v 1.1 2004/09/15 14:58:25 pnever Exp $
 * $Revision: 1.1 $
 * $Date: 2004/09/15 14:58:25 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.report;

import java.io.IOException;
import java.util.Enumeration;
import java.util.List;
import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.SlideException;
import org.apache.slide.common.SlideToken;
import org.apache.slide.common.SlideTokenWrapper;
import org.apache.slide.structure.ActionNode;
import org.apache.slide.structure.ObjectNode;
import org.apache.slide.structure.SubjectNode;
import org.apache.slide.webdav.WebdavServletConfig;
import org.apache.slide.webdav.method.report.AbstractReport;
import org.apache.slide.webdav.util.AclConstants;
import org.apache.slide.webdav.util.PreconditionViolationException;
import org.apache.slide.webdav.util.ViolatedPrecondition;
import org.apache.slide.webdav.util.WebdavStatus;
import org.jdom.Element;


/**
 * URM-based report implementationimplementation.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.1 $
 */

public abstract class XAbstractReport extends AbstractReport implements AclConstants {
    
    /**
     * Constructor
     *
     * @param    slideToken          a  SlideToken
     * @param    token               a  NamespaceAccessToken
     * @param    config              a  WebdavServletConfig
     * @param    serverUrl           a  String
     * @param    contextPath         a  String
     */
    public XAbstractReport(SlideToken slideToken, NamespaceAccessToken token, WebdavServletConfig config, String slideContextPath) {
        super(slideToken, token, config, slideContextPath);
    }
    
    protected String getPrincipalName(Element principalElm, String resourcePath) throws PreconditionViolationException {
        List children = principalElm.getChildren(E_HREF, DNSP);
        if (children.size() != 1) {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("exactly-one-href-element",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         "the DAV:principal element must contain exactly one DAV:href element"),
                resourcePath
            );
        }
        ObjectNode principalNode = new SubjectNode(((Element)children.get(0)).getTextTrim());
        return principalNode.getPath().lastSegment();
    }
    
    protected String getPrivilegeName(Element privilegeElm, String resourcePath) throws PreconditionViolationException {
        List children = privilegeElm.getChildren();
        if (children.size() != 1) {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("exactly-one-privilege",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         "the DAV:privilege element must contain exactly one privilege"),
                resourcePath
            );
        }
        return ((Element)privilegeElm.getChildren().get(0)).getName();
    }
    
    /**
     * Execute report and add results to given multistatus element
     *
     * @param    resourcePath        a  String
     * @param    multistatusElm      an Element
     * @param    depth               an int
     *
     * @throws   SlideException
     * @throws   IOException
     */
    public void execute(String resourcePath, Element multistatusElm, int depth) throws SlideException, IOException {
        if (depth < 0) {
            return;
        }
        writeReport(resourcePath, multistatusElm);
        ObjectNode onode = structure.retrieve(slideToken, resourcePath);
        Enumeration childrenEnum = structure.getChildren(slideToken, onode);
        while  (childrenEnum.hasMoreElements()) {
            ObjectNode cnode = (ObjectNode)childrenEnum.nextElement();
            execute(cnode.getUri(), multistatusElm, depth-1);
        }
    }
    
    protected abstract void writeReport( String resourcePath, Element multistatusElm) throws SlideException;
    
    protected void checkPermissions(String resourcePath) throws SlideException {
        SlideToken insecureSlideToken = new SlideTokenWrapper(slideToken);
        insecureSlideToken.setForceStoreEnlistment(false);
        insecureSlideToken.setForceSecurity(false);
        insecureSlideToken.setForceLock(false);
        String actionsPath = token.getNamespaceConfig().getActionsPath();
        String readAclPath = null;
        if (actionsPath == null || actionsPath.length() == 0) {
            throw new IllegalStateException("Could not determine actionspath from namespace config");
        }
        else if (actionsPath.endsWith("/")) {
            readAclPath = actionsPath+"read-acl";
        }
        else {
            readAclPath = actionsPath+"/read-acl";
        }
        ObjectNode objectNode = structure.retrieve(insecureSlideToken, resourcePath);
        ActionNode readAclNode = (ActionNode)structure.retrieve(insecureSlideToken, readAclPath);
        security.checkCredentials(slideToken, objectNode, readAclNode);
    }
}



