/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/report/XPropfindDepthOneReport.java,v 1.1 2004/09/15 14:58:25 pnever Exp $
 * $Revision: 1.1 $
 * $Date: 2004/09/15 14:58:25 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.report;

import org.apache.slide.common.*;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import org.apache.slide.content.NodeRevisionNumber;
import org.apache.slide.search.RequestedResource;
import org.apache.slide.search.SearchQuery;
import org.apache.slide.search.basic.Literals;
import org.apache.slide.webdav.WebdavServletConfig;
import org.apache.slide.webdav.method.report.AbstractReport;
import org.apache.slide.webdav.util.PreconditionViolationException;
import org.apache.slide.webdav.util.ViolatedPrecondition;
import org.apache.slide.webdav.util.WebdavStatus;
import org.apache.slide.webdav.util.WebdavUtils;
import org.jdom.Element;
import org.jdom.Namespace;


/**
 * Query-based report implementation.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.1 $
 */

public final class XPropfindDepthOneReport extends AbstractReport {
    
    protected RequestedProperties requestedProperties = null;
    
    /**
     * Constructor
     *
     * @param    slideToken          a  SlideToken
     * @param    token               a  NamespaceAccessToken
     * @param    config              a  WebdavServletConfig
     * @param    serverUrl           a  String
     * @param    contextPath         a  String
     */
    public XPropfindDepthOneReport(SlideToken slideToken, NamespaceAccessToken token, WebdavServletConfig config, String slideContextPath) {
        super(slideToken, token, config, slideContextPath);
    }
    
    /**
     * Initialize report worker with specified report element
     *
     * @param    resourcePath        a  String
     * @param    versionTreeElm      an Element
     *
     * @throws   PreconditionViolationException
     */
    public void init(String resourcePath, Element propfDepthOneElm) throws PreconditionViolationException {
        List childrenList = propfDepthOneElm.getChildren();
        if (childrenList.size() != 1) {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("invalid-propfind-depth-one",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         "DAV:propfind-depth-one element must have 1 child: DAV:prop"),
                resourcePath
            );
        }
        
        Element propElm = null;
        if (E_PROP.equals(((Element)childrenList.get(0)).getName())) {
            propElm = (Element)childrenList.get(0);
        }
        else {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("invalid-propfind-depth-one",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         "DAV:propfind-depth-one element must have 1 child: DAV:prop"),
                resourcePath
            );
        }
        
        // report can only be applied to collection
        if ( ! WebdavUtils.isCollection(token, slideToken, resourcePath) ) {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("must-be-collection",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         "the request-URI must specify a collection"),
                resourcePath
            );
        }
        
        try {
            this.requestedProperties = new RequestedPropertiesImpl(propElm);
        }
        catch (PropertyParseException e) {
            throw new PreconditionViolationException(
                new ViolatedPrecondition("invalid-prop",
                                         WebdavStatus.SC_BAD_REQUEST,
                                         e.getMessage()),
                resourcePath
            );
        }
    }
    
    /**
     * Execute report and add results to given multistatus element
     *
     * @param    resourcePath        a  String
     * @param    multistatusElm      an Element
     * @param    depth               an int
     *
     * @throws   SlideException
     * @throws   IOException
     */
    public void execute(String resourcePath, Element multistatusElm, int depth) throws SlideException, IOException {
        if (depth < 0) {
            return;
        }
        Element queryElm = getQueryElement(resourcePath);
        // new XMLOutputter("  ", true).output(queryElm, System.out);
        
        String absUri = WebdavUtils.getAbsolutePath (resourcePath, slideContextPath, config);
        
        SearchQuery query =
            search.createSearchQuery(queryElm.getNamespaceURI(),
                                     queryElm,
                                     slideToken,
                                     config.getDepthLimit(),
                                     absUri);
        
        Iterator result = query.execute().iterator();
        while (result.hasNext()) {
            RequestedResource r = (RequestedResource)result.next();
            multistatusElm.addContent(getResponseElement(slideToken, r.getUri(), new NodeRevisionNumber(), requestedProperties));
        }
        
    }
    
    private Element getQueryElement(String resourcePath) {
        Element result = new Element(Literals.BASICSEARCH, DNSP);
        // select
        Element selectElm = new Element(Literals.SELECT, DNSP);
        result.addContent(selectElm);
        Element propElm = new Element(E_PROP, DNSP);
        selectElm.addContent(propElm);
        Iterator props = requestedProperties.getRequestedProperties();
        while (props.hasNext()) {
            RequestedProperty p = (RequestedProperty)props.next();
            Namespace nsp = (DNSP.getURI().equals(p.getNamespace()))
                ? DNSP
                : Namespace.getNamespace(p.getNamespace());
            propElm.addContent(new Element(p.getName(), nsp));
        }
        // from
        Element fromElm = new Element(Literals.FROM, DNSP);
        result.addContent(fromElm);
        Element scopeElm = new Element(Literals.SCOPE, DNSP);
        fromElm.addContent(scopeElm);
        Element hrefElm = new Element(E_HREF, DNSP);
        hrefElm.setText(WebdavUtils.getAbsolutePath (resourcePath, slideContextPath, config));
        Element depthElm = new Element(E_DEPTH, DNSP);
        depthElm.setText("1");
        scopeElm.addContent(hrefElm);
        scopeElm.addContent(depthElm);
        return result;
    }
    
    /**
     * Method checkPreconditions
     * @param    resourcePath        a  String
     * @param    depth               an int
     * @throws   PreconditionViolationException
     * @throws   ServiceAccessException
     */
    public void checkPreconditions(String resourcePath, int depth) throws PreconditionViolationException, ServiceAccessException {
        //        if (depth != 0) {
        //            throw new PreconditionViolationException(
        //                new ViolatedPrecondition("depth-must-be-zero",
        //                                         WebdavStatus.SC_BAD_REQUEST,
        //                                         "This report is only defined for depth=0."),
        //                resourcePath
        //            );
        //        }
    }
}


