/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/security/admin/URMParent.java,v 1.3 2004/07/30 06:52:01 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:01 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.security.admin;

import org.apache.slide.urm.URMException;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.authenticator.rolemanager.URMRole;
import org.apache.slide.urm.authenticator.userdb.URMGroup;
import org.apache.slide.urm.authenticator.userdb.URMUser;
import org.apache.slide.urm.common.URMConstants;
import org.apache.slide.urm.common.URMSetRoleException;
import org.apache.slide.store.tamino.common.XForbiddenException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Set;
import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.macro.ConflictException;
import org.apache.slide.macro.ForbiddenException;

/**
 ** URMParent adds user groups or roles to a parent.
 **
 ** @author    josef.haiduk@softwareag.com
 ** @version   $Revision: 1.3 $
 **
 **/

class URMParent
{
	private String uri = null;
	private URMUserDBStore parent = null;
	private Hashtable groupGates = new Hashtable();
	private Hashtable roleGates = new Hashtable();
	
	/**
	 * Constructor
	 *
	 * @param    parent              an URMUserDBStore
	 *
	 */
	URMParent(URMUserDBStore parent) {
		this.parent = parent;
	}
	
	/**
	 * Method prepareToAdd. Checks if the specified subject cann successfully be added to a parent
	 * spefyfied by the parentUri.
	 *
	 * @param    subject             an URMSubject
	 * @param    parentUri           a  String representing the Uri of the parent
	 *
	 * @throws   ServiceAccessException
	 *
	 */
	void prepareToAdd(URMSubject subject, String parentUri)
		throws ServiceAccessException
	{
		try {
			String subjectType = parentUri.substring(URMUserDBStore.getUserDBPath().length()+1);
			if ( subjectType.startsWith(URMUserDBStore.groupCollection) ) {
				URMGroupsGate gate = new URMGroupsGate(parent, parentUri);
				URMGroup admin = gate.getUrmAdministratorWithAdminRole().getGroup(gate.getId(), parent.domain);
				groupGates.put(subject,admin);
				return;
			}
			
			if ( subjectType.startsWith(URMUserDBStore.roleCollection) ) {
				URMRolesGate gate = new URMRolesGate(parent, parentUri);
				URMRole admin = gate.getUrmAdministratorWithAdminRole().getRole(gate.getId());
				roleGates.put(subject,admin);
				return;
			}
			
			throw new ServiceAccessException(parent,
				new ConflictException(parentUri,
					new XForbiddenException( "A parent must exist before it can be used" )));
		}
		catch (URMForbiddenException e) {
            throw new ServiceAccessException(parent,
                    new ForbiddenException(uri.toString(), e));
        }
        catch (URMSetRoleException e) {
            throw new ServiceAccessException(parent,
                    new ForbiddenException(uri.toString(), e));
        }
        catch (URMException e) {
            throw new ServiceAccessException(parent, e);
        }
		
	}
	
	/**
	 * Method add adds all the URMSubjectes passed to parent in the prepareToAdd method.
	 *
	 * @throws   ServiceAccessException
	 *
	 */
	void add()
		throws ServiceAccessException
	{
		try
		{
			// add to groups
			Enumeration keys = groupGates.keys();
			while (keys.hasMoreElements())
			{
				URMSubject subject = (URMSubject)keys.nextElement();
				URMGroup admin = (URMGroup)groupGates.get(subject);
				
				if ( subject.getType() == URMConstants.USER ) {
					if ( admin.isUserMember(subject.getName(), parent.domain) )
						continue;
					admin.addUserToGroup(subject.getName(), parent.domain);
				}
				if ( subject.getType() == URMConstants.GROUP ) {
					if ( admin.isGroupMember(subject.getName(), parent.domain) )
						continue;
					admin.addGroupToGroup(subject.getName(), parent.domain);
				}
			}
			// add to groups
			keys = roleGates.keys();
			while (keys.hasMoreElements())
			{
				URMSubject subject = (URMSubject)keys.nextElement();
				URMRole admin = (URMRole)roleGates.get(subject);
				
				if ( subject.getType() == URMConstants.USER ) {
					Set set = admin.getUserMapping();
					if ( set.contains((URMUser)subject) )
						continue;
					admin.addUserLink(subject.getName(), parent.domain);
				}
				if ( subject.getType() == URMConstants.GROUP ) {
					Set set = admin.getGroupMapping();
					if ( set.contains((URMGroup)subject) )
						continue;
					admin.addGroupLink(subject.getName(), parent.domain);
				}
				if ( subject.getType() == URMConstants.ROLE ) {
					Set set = admin.getAncestors();
					if ( set.contains((URMRole)subject) )
						continue;
					admin.addAncestor(subject.getName());
				}
			}
		}
		catch (URMForbiddenException e) {
            throw new ServiceAccessException(parent,
                    new ForbiddenException(uri.toString(), e));
        }
        catch (URMException e) {
            throw new ServiceAccessException(parent, e);
        }
	}
}

