/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/store/XActionStore.java,v 1.3 2004/07/30 06:52:03 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:03 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.store;

import com.softwareag.common.instrumentation.logging.Level;
import com.softwareag.common.instrumentation.logging.Logger;
import com.softwareag.common.instrumentation.logging.LoggerFactory;
import com.softwareag.common.instrumentation.logging.LoggerUtil;
import java.util.Enumeration;
import java.util.Vector;
import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.common.Uri;
import org.apache.slide.content.NodeRevisionContent;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.content.NodeRevisionNumber;
import org.apache.slide.content.RevisionDescriptorNotFoundException;
import org.apache.slide.content.RevisionNotFoundException;
import org.apache.slide.store.ContentStore;
import org.apache.slide.store.LockStore;
import org.apache.slide.store.NodeStore;
import org.apache.slide.store.RevisionDescriptorStore;
import org.apache.slide.store.RevisionDescriptorsStore;
import org.apache.slide.store.SecurityStore;
import org.apache.slide.store.tamino.jdomobjects.XFactory;
import org.apache.slide.store.tamino.security.XSecurityStore;
import org.apache.slide.store.tamino.tools.stores.ActionDeclaration;
import org.apache.slide.structure.ActionNode;
import org.apache.slide.structure.ObjectNode;
import org.apache.slide.structure.ObjectNotFoundException;
import org.apache.slide.util.ClassName;


/**
 ** OSSecurity store. This non-persistent store serves as the Security Store for
 ** an authentication environment agains the current operationg system.
 **
 ** @author    eckehard.hermann@softwareag.com
 ** @version   $Revision: 1.3 $
 **
 **/
public class XActionStore extends XSecurityStore implements ContentStore, LockStore,
NodeStore, RevisionDescriptorsStore, RevisionDescriptorStore, SecurityStore {

    private static final String LOGNAME = LoggerUtil.getThisClassName();
    private static final String CLASSNAME = new ClassName(LOGNAME).getPlainName();
    private static Logger logger = LoggerFactory.getLogger(LOGNAME);

    public final static String ACTIONS = ActionDeclaration.SCOPE;
    public final static String READ = ActionDeclaration.SCOPE + "/read";
    public final static String WRITE = ActionDeclaration.SCOPE + "/write";
    public final static String READ_ACL = ActionDeclaration.SCOPE + "/read-acl";
    public final static String WRITE_ACL = ActionDeclaration.SCOPE + "/write-acl";
    public final static String MANAGE = ActionDeclaration.SCOPE + "/manage";

    private java.util.HashSet actions = new java.util.HashSet();

    /**
     ** Default constructor.
     **/
    public XActionStore() {
        if( logger.isLoggable(Level.FINE) )
            logger.entering( CLASSNAME, "<init>" );

        if( logger.isLoggable(Level.FINE) )
            logger.exiting( CLASSNAME, "<init>" );

        init();
    }

    /** initialize the actions HashSet
     **/
    private void init() {
        actions.add(READ);
        actions.add(READ_ACL);
        actions.add(WRITE);
        actions.add(WRITE_ACL);
        actions.add(MANAGE);
        actions.add(ACTIONS);

    }

    /**
     * Retrieve an object from the Descriptors Store.
     *
     * @param uri Uri of the object we want to retrieve
     * @return an object node
     * @exception ServiceAccessException Error accessing the Descriptors Store
     * @exception ObjectNotFoundException The object to retrieve was not found
     */
    public ObjectNode retrieveObject(Uri uri)
    throws ServiceAccessException, ObjectNotFoundException {
        if( logger.isLoggable(Level.FINE) ) logger.entering( CLASSNAME, "retrieveObject",
            new Object[] {uri} );

        String uriStr = uri.toString();
        Vector children = new Vector();
        ObjectNode result = null;

        if (!actions.contains(uriStr)){
            throw new ObjectNotFoundException(uri);
        }
        result = new ActionNode( uriStr, children, new Vector() );

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME, "retrieveObject",
            (result!=null ? result.getUri() : null) );
        return result;
   }


    /**
     * Retrieve a revision descriptors.
     *
     * @param uri Uri
     * @return node revision descriptors
     * @exception ServiceAccessException Service access error
     * @exception RevisionDescriptorNotFoundException Revision descriptor
     * was not found
     */
    public NodeRevisionDescriptors retrieveRevisionDescriptors(Uri uri)
    throws ServiceAccessException, RevisionDescriptorNotFoundException {
        if( logger.isLoggable(Level.FINE) ) logger.entering( CLASSNAME,
            "retrieveRevisionDescriptors", new Object[] {uri} );

        String uriStr = uri.toString();
        if (!actions.contains(uriStr)){
            throw new RevisionDescriptorNotFoundException(uriStr);
        }
        NodeRevisionDescriptors result = XFactory.createNRDs( uriStr );

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME,
                                                           "retrieveRevisionDescriptors" );
        return result;
    }


    // ------------------------------------------------------------------------
    // RevisionDescriptorStore interface
    // ------------------------------------------------------------------------

    /**
     * Retrieve revision descriptor.
     *
     * @param uri uri
     * @param revisionNumber Node revision number
     * @return a node revision descriptor
     * @exception ServiceAccessException Service access error
     * @exception RevisionDescriptorNotFoundException revision descriptor not found
     */
    public NodeRevisionDescriptor retrieveRevisionDescriptor
    (Uri uri, NodeRevisionNumber revisionNumber)
    throws ServiceAccessException, RevisionDescriptorNotFoundException {
        if( logger.isLoggable(Level.FINE) ) logger.entering( CLASSNAME,
                                                            "retrieveRevisionDescriptor", new Object[] {uri, revisionNumber} );

        String uriStr = uri.toString();

        NodeRevisionDescriptor result = null;
        if (!actions.contains(uriStr)){
            throw new RevisionDescriptorNotFoundException(uriStr);
        }
        result = XFactory.createCollectionNRD( uriStr );

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME,
            "retrieveRevisionDescriptor", (result!=null
            ? result.getRevisionNumber()+"@"+result.getBranchName()
            : null) );
        return result;
    }

    /**
     * Enumerate permissions on the resource given by uri.
     *
     * @param uri Uri of the resource
     * @return the permissions as enumeration
     * @exception ServiceAccessException Error accessing the Descriptors Store
     */
    public Enumeration enumeratePermissions(Uri uri) throws ServiceAccessException {
        if( logger.isLoggable(Level.FINE) ) logger.entering( CLASSNAME,
            "enumeratePermissions", new Object[] {uri} );

        Vector permissionsVector = null;
        permissionsVector = new Vector();

        Enumeration result = permissionsVector.elements();

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME,
            "enumeratePermissions", result );
        return result;
    }
    /**
     * Returns the list of locks put on a subject.
     *
     * @param uri Uri of the resource
     * @return Enumeration List of locks which have been put on the subject
     * @exception ServiceAccessException Service access error
     */
    public Enumeration enumerateLocks(Uri uri) throws ServiceAccessException {
        if( logger.isLoggable(Level.FINE) ) logger.entering( CLASSNAME, "enumerateLocks",
            new Object[] {uri} );
        Vector locksVector = null;
        locksVector = new Vector();

        Enumeration result = locksVector.elements();

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME, "enumerateLocks",
            result);
        return result;
    }

            /**
     * Retrieve revision content.
     *
     * @param uri Uri
     * @param revisionDescriptor revision descriptor
     * @return a node revision content
     * @exception ServiceAccessException Error accessing the revision descriptor
     * @exception RevisionNotFoundException Error accessing the revision descriptor
     */
    public NodeRevisionContent retrieveRevisionContent(
    Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException, RevisionNotFoundException {
        if( logger.isLoggable(Level.FINE) )
            logger.entering( CLASSNAME, "retrieveRevisionContent", new Object[] {uri,
            (revisionDescriptor!=null
               ? revisionDescriptor.getRevisionNumber()+"@"+revisionDescriptor.getBranchName()
                 : null) } );

        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME,
            "NodeRevisionContent" );
        throw new RevisionNotFoundException (uri.toString(),revisionDescriptor.getRevisionNumber());
    }
}



