/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/store/XAdministrationStore.java,v 1.3 2004/07/30 06:52:03 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:03 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.store;

import com.softwareag.common.instrumentation.logging.Level;
import com.softwareag.common.instrumentation.logging.Logger;
import com.softwareag.common.instrumentation.logging.LoggerFactory;
import com.softwareag.common.instrumentation.logging.LoggerUtil;
import java.util.Hashtable;
import org.apache.slide.common.Namespace;
import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.Scope;
import org.apache.slide.common.Service;
import org.apache.slide.common.ServiceInitializationFailedException;
import org.apache.slide.common.ServiceParameterErrorException;
import org.apache.slide.common.ServiceParameterMissingException;
import org.apache.slide.store.StandardStore;
import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.util.ClassName;


/**
 ** XDAV-specific extension of org.apache.slide.store.StandardStore
 ** for the administration stores.
 **
 ** @author    peter.nevermann@softwareag.com
 **
 ** @version   $Revision: 1.3 $
 **/
public class XAdministrationStore extends StandardStore
implements XGlobals {
    
    private static final String LOGNAME = LoggerUtil.getThisClassName();
    private static final String CLASSNAME = new ClassName(LOGNAME).getPlainName();
    private static Logger logger = LoggerFactory.getLogger(LOGNAME);

    
    /** True, if this store is initialized. */
    protected boolean initialized = false;
    
    /** The parameters (from Domain.xml). */
    private Hashtable parameters = null;

    /** The namespace access token */
    protected NamespaceAccessToken nsaToken = null;

    
    /**
     ** Default constructor.
     **/
    public XAdministrationStore() {
        if( logger.isLoggable(Level.FINE) )
            logger.entering( CLASSNAME, "<init>" );
        
        if( logger.isLoggable(Level.FINE) )
            logger.exiting( CLASSNAME, "<init>" );
    }

    /**
     * Returns true, if this store is initialized.
     * @return true, if this store is initialized
     */
    public boolean isInitialized() {
        return initialized;
    }
    
    /**
     ** Get the associated namespace.
     **
     ** @return     return the namespace
     **/
    public Namespace getNamespace() {
        return namespace;
    }
    
    /**
     ** Get the associated scope.
     **
     ** @return     return the scope
     **/
    public Scope getScope() {
        return scope;
    }
    
    /**
     ** Get the associated child store.
     **
     ** @return     return the content store
     **/
    public Service getChildStore() {
        return (Service)nodeStore;
    }
    
    /**
     * Initializes descriptors store.
     * @param token name space token
     * @exception ServiceInitializationFailedException Throws an exception
     * if the descriptors store has already been initialized before
     */
    public void initialize(NamespaceAccessToken token)
    throws ServiceInitializationFailedException {
        if( logger.isLoggable(Level.FINE) )
            logger.entering( CLASSNAME, "initialize", new Object[]{token} );

        if( initialized )
            throw new ServiceInitializationFailedException( this, "Already initialized" );
        
        this.nsaToken = token;
        initialized = true;

        // disable SEARCH for this store
        // parameters.put (BasicSearchLanguage.BASIC_QUERY_CLASS, DUMMY_QUERY);
        try {
            super.initialize(token); // this comes after monitoring stuff !!
        }
        catch( ServiceInitializationFailedException x ) {
            if( logger.isLoggable(Level.WARNING) )
                logger.warning( "Store "+getName()+
                    " could not be initialized. Reason: "+x.getMessage() );
            
            //reset resourceManagers
            resourceManagers = new Service[0];
            
            //set unavailable childstore
            XUnavailableStore unavailChildStore = new XUnavailableStore( x.getMessage() );
            setNodeStore( unavailChildStore );
            setSecurityStore( unavailChildStore );
            setLockStore( unavailChildStore );
            setRevisionDescriptorsStore( unavailChildStore );
            setRevisionDescriptorStore( unavailChildStore );
            setContentStore( unavailChildStore );
            
            // init the new childstore
            unavailChildStore.initialize( token );
            unavailChildStore.setScope( scope );
            unavailChildStore.setNamespace( namespace );
            try {
                unavailChildStore.setParameters( parameters );
            }
            catch( ServiceParameterErrorException x1 ) {}
            catch( ServiceParameterMissingException x1 ) {}
        }
        
        if( logger.isLoggable(Level.FINE) )
            logger.exiting( CLASSNAME, "initialize" );
    }
    
    /**
     * Initializes this child store with a set of parameters.
     * These are:
     * <li>taminoBase (the base of the Tamino URL, e.g. http://myhost/tamino
     * <li>database (the Tamino database, i.e. mydb)
     * <li>collection (the Tamino collection, i.e. mycoll)
     *
     * @param parameters Hashtable containing the parameters' name
     * and associated value
     *
     * @exception   ServiceParameterErrorException
     * @exception   ServiceParameterMissingException
     */
    public void setParameters(Hashtable parameters)
    throws ServiceParameterErrorException, ServiceParameterMissingException {
        if( logger.isLoggable(Level.FINE) )
            logger.entering( CLASSNAME, "setParameters", new Object[] {parameters} );

        super.setParameters( parameters );
        this.parameters = parameters;
            
        if( logger.isLoggable(Level.FINE) ) logger.exiting( CLASSNAME, "setParameters" );
    }
    
    /**
     * Get the associated namespace access token.
     * @return     the namespace access token
     */
    public NamespaceAccessToken getNamespaceAccessToken() {
        return nsaToken;
    }

    // Inherited from org.apache.slide.store.StandardStore
    public String toString () {
        return getName();
    }
}


