/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/store/monitoring/MonitoredRatio.java,v 1.3 2004/07/30 06:52:04 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:04 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.store.monitoring;


/**
 ** This class represent a ratio between 2 monitored items, 'w' and 'p' (each of which must be
 ** either of type timer of counter). The ratio can be represented either as percentage
 ** or as ratio. It also can be complemented.
 ** <p>The value of a ratio is:
 ** <li>as ratio: p/w
 ** <li>as percentage: p/w*100
 ** <li>as ratio, complemented: 1 - p/w
 ** <li>as percentage, complemented: (1 - p/w)*100
 **
 ** @author    peter.nevermann@softwareag.com
 ** @version   0.1
 **/
public class MonitoredRatio {

    private MonitoredItem whole  = null;
    private MonitoredItem part   = null;
    private boolean asPercentage = false;
    private boolean complemented = false;
    
    
    /**
     ** Default constructor.
     **
     ** @pre ((whole instanceof MonitoredCounter) || (whole instanceof MonitoredTimer)) &&
     **      ((part instanceof MonitoredCounter) || (part instanceof MonitoredTimer))
     **
     ** @param w            the whole of the ratio
     ** @param p            the part of the ratio
     ** @param asPercentage show as percentage yes or true
     ** @param complemented yes or true
     **/
    public MonitoredRatio( MonitoredItem w, MonitoredItem p,
    boolean asPercentage, boolean complemented ) {
        this.whole = w;
        this.part = p;
        this.asPercentage = asPercentage;
        this.complemented = complemented;
    }
    
    /**
     ** Get the value of the ratio.
     **
     ** @return     the value of the ratio
     **/
    public float getValue() {
        long w = ((Long)whole.getValue()).longValue();
        long p = ((Long)part.getValue()).longValue();
        float result = (w != 0 ? ((float)p/(float)w) : 0);
        
        if( complemented )
            result = 1 - result;
        
        if( asPercentage )
            result = 100 * result;
        
        return result;
    }
    
    /**
     ** Return string representation of this ratio.
     **
     ** @pre        true
     ** @post       true
     **
     ** @return string representation of this ratio.
     **/
    public String toString() {
        String result = null;
        if( asPercentage ) {
            result = String.valueOf( Math.round(getValue()) );
            result = result+" %";
        }
        else {
            result = String.valueOf( getValue() );
        }
        return result;
    }
    
    
}

