/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/tools/config/Apache.java,v 1.1 2004/03/25 16:18:04 juergen Exp $
 * $Revision: 1.1 $
 * $Date: 2004/03/25 16:18:04 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.tools.config;

import java.io.File;
import java.io.IOException;
import org.apache.slide.util.Files;
import org.apache.slide.util.IO;
import org.apache.slide.util.Properties;
import org.apache.slide.util.Strings;
import org.apache.slide.util.XException;
import org.apache.slide.util.cli.Abort;
import org.apache.slide.util.os.Catalina;
import org.apache.slide.util.os.Executable;
import org.apache.slide.util.os.Platform;

/**
 ** @version   $Revision: 1.1 $
 **/
public class Apache {
    private final IO io;
    private final File home;
    
    public Apache(File home) throws Abort {
        this.io = new IO();
        this.home = home;
        if (!home.isDirectory()) {
            throw new Abort("apache home not found: " + home);
        }
    }
    
    public boolean install(Catalina catalina)
        throws XException, IOException, Abort {
        File catalinaConfDir;
        HttpdConf httpdConf;
        boolean overwrite;
        
        // note: we don't need libjkjni.so because we're not using jni
        
        httpdConf = httpdConf(catalina.home, home);
        io.backup(httpdConf.getFile());
        overwrite = httpdConf.add();
        catalinaConfDir = catalina.getConfDir();
        copyTemplate("workers2.properties", catalinaConfDir, httpdConf.getFile().getParentFile(), catalina);
        copyTemplate("jk2.properties", catalinaConfDir, catalinaConfDir, catalina);
        return overwrite;
    }
    
    public String getVersion() throws Abort, IOException {
        final String PREFIX = "Apache/";
        final String SUFFIX = "\n";
        File dir;
        String name;
        File file;
        Executable exec;
        String str;
        int first;
        int last;
        String version;
        
        dir = (!Platform.isWindows() && home.getPath().startsWith("/etc"))? new File("/usr/sbin") : new File(home, "bin");
        name = Platform.isWindows()? "Apache.exe" : "httpd";
        file = Files.join(dir, name);
        if (!file.isFile()) {
            throw new Abort("httpd not found: " + file);
        }
        exec = new Executable(file.getPath(), new String[] { "-v" }, null);
        str = exec.run(home);
        first = str.indexOf(PREFIX);
        if (first == -1) {
            throw new Abort("invalid version string: " + str);
        }
        last = str.indexOf(SUFFIX, first + 1); // ok for first == -1
        if (last == -1) {
            throw new Abort("invalid version string: " + str);
        }
        version = str.substring(first + PREFIX.length(), last).trim();
        version = Strings.replace(version, ".", "");
        if (version.length() == 0) {
            throw new Abort("invalid version string: " + str);
        }
        return version;
    }
    
    public void remove(Catalina catalina) throws IOException, Abort {
        httpdConf(catalina.home, home).remove();
    }
    
    //--
    
    private HttpdConf httpdConf(File catalinaHome, File apacheHome) throws Abort, IOException {
        File dir;
        File module;
        String name;
        String version;
        
        dir = new File(apacheHome, "conf");
        version = getVersion();
        name = Platform.isWindows()? "mod_jk2_win32_" + version + ".dll" : "mod_jk2_linux_" + version + ".so";
        module = Files.join(catalinaHome, "native", "lib", name);
        if (!module.isFile()) {
            throw new Abort("no mod_jk2 available for apache version " + version);
        }
        return new HttpdConf(new File(dir, "httpd.conf"), module);
    }
    
    private void copyTemplate(String name, File srcDir, File destDir, Catalina catalina) throws Abort, XException {
        File src;
        File dest;
        String str;
        Properties props;
        
        src = new File(srcDir, name + ".raw");
        dest = new File(destDir, name);
        if (dest.exists()) {
            try {
                io.backup(dest);
            } catch (IOException e) {
                throw new Abort(dest + ": cannot create backup file: " + e);
            }
        }
        props = new Properties();
        props.add("port", "" + catalina.getAjpPort());
        props.add("context", catalina.getContext());
        try {
            str = io.readString(src);
            str = props.replace(str);
            Files.clean(dest);
            io.write(dest, str);
        } catch (IOException e) {
            throw new Abort(dest.getPath(), e);
        }
    }
}

