/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/tools/repairer/AbstractCheck.java,v 1.4 2004/12/15 10:38:27 pnever Exp $
 * $Revision: 1.4 $
 * $Date: 2004/12/15 10:38:27 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */



package org.apache.slide.store.tamino.tools.repairer;

import java.util.ArrayList;
import java.util.List;
import org.apache.slide.store.tamino.datastore.XDescriptorsAccessor;
import org.apache.slide.store.tamino.jdomobjects.XDescriptors;
import org.apache.slide.store.tamino.jdomobjects.XDescriptorsUtil;
import org.apache.slide.util.XException;


/**
 ** Check and Repair the consistency of WebDAV information.
 ** This class contains methods to test the consistency of WebDAV information
 ** and to repair inconsistencies.
 **
 ** @author    Hardy.Kiefer@softwareag.com
 ** @author    Peter.Nevermann@softwareag.com
 ** @version   $Revision: 1.4 $
 **
 **/
public abstract class AbstractCheck {
    
    /** holds the current db session for a collection */
    protected XTaminoClient taminoClient;
    
    /** the current taminoCollection */
    protected String taminoCollection;

    /** number of inconsistent metadata documents repaired. */
    private List patches;

    private final String startMsg;
    private final String description;
    
    /**
     ** Constructor of AbstractWebDAVtest.
     **
     ** @param      taminoClient         current database session
     **
     **/
    public AbstractCheck(XTaminoClient taminoClient, String startMsg, String desciption) throws XException {
        this.taminoClient        = taminoClient;
        this.taminoCollection = taminoClient.getContentCollection();
        this.startMsg = startMsg;
        this.description = desciption;
        this.patches = new ArrayList();
    }

    public void logStart(XRepairLogWriter logWriter) {
        logWriter.println(RepairConstants.TC_CHECK, startMsg);
        logWriter.println(RepairConstants.TC_CHECK_DESCRIPTION, description);
    }
        
    public List getPatches() {
        return patches;
    }
    
    /** only defined during repair **/
    private XRepairLogWriter logWriter = null;
    
    /** only defined during repair **/
    private boolean checkOnly = false;
    
    public final void repair(XRepairLogWriter logWriter, boolean checkOnly) throws XException {
        this.checkOnly = checkOnly;
        this.logWriter = logWriter;
        
        check();
        
        this.checkOnly = false;
        this.logWriter = null;
    }

    /**
     * This method finds checks and eventually repairs the WebDAV consistency.
     */
    public abstract void check() throws XException;
    
    public void logAndApplyPatch(Patch patch) throws XException {
        patches.add(patch);
        if (patches.size() < RepairConstants.TC_CHECK_DETAILS_MAX) {
            patch.log(logWriter);
        }
        if (!checkOnly) {
            patch.apply(taminoClient);
        }
    }
    
    public void revitalize(XDescriptorsAccessor accessor, String parentUuri, String childUuri) throws XException {
        String segment;
        XDescriptors parent;
        XDescriptors child;
        

        parent = accessor.readOneRwDescriptorsByUuri(parentUuri);
        child = accessor.readOneRwDescriptorsByUuri(childUuri);
        segment = XDescriptorsUtil.newSegmentName(parent, child, "revitalized");
        logAndApplyPatch(new NamespaceConsistencyCheck.AddReference(parent, child, segment, false));
        logAndApplyPatch(new NamespaceConsistencyCheck.AddReference(child, parent, segment, true));
    }
}


