/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/tools/repairer/XRepairModeToken.java,v 1.4 2004/07/30 06:52:05 ozeigermann Exp $
 * $Revision: 1.4 $
 * $Date: 2004/07/30 06:52:05 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.tools.repairer;

import com.softwareag.common.instrumentation.logging.Logger;
import com.softwareag.common.instrumentation.logging.LoggerFactory;
import com.softwareag.common.instrumentation.logging.LoggerUtil;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.util.ClassName;
import org.apache.slide.util.JDom;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;


/**
 ** Handler for the RepairModeToken xml file.
 **
 ** @author    peter.nevermann@softwareag.com
 ** @version   $Revision: 1.4 $
 **/
public class XRepairModeToken implements XGlobals {
    
    private static final String LOGNAME = LoggerUtil.getThisClassName();
    private static final String CLASSNAME = new ClassName(LOGNAME).getPlainName();
    private static Logger logger = LoggerFactory.getLogger(LOGNAME);
    
    // Constants
    private final static String REPAIRMODETOKEN = "repairmodetoken";
    private final static String PARAMETER       = "parameter";
    private final static String NAME            = "name";
    private final static String ON_OPEN_TA      = "onOpenTA";
    private final static String WAIT_TIMEOUT    = "waitTimeout";
    private final static String WAIT_ON_TA      = "wait";
    private final static String ERROR_ON_TA     = "error";
    private final static String ROLLBACK_ON_TA  = "rollback";
    
    private final static boolean PERSISTENT     = true;
    
    /**
     * name of the repair mode token
     */
    public  final static String REPAIR_MODE_TOKEN = "RepairModeToken.xml";
    
    // Fields
    private Document doc = null;
    private int onOpenTA = 0;
    private long waitTimeout = ON_OPEN_TA_WAIT_TIMEOUT; //default: 60 sec
    
    
    /**
     * Test for persistance.
     * @return true, if the repair-mode tokens are persistent
     */
    static public boolean isPersistent() {
        return PERSISTENT;
    }
    
    /**
     * Stream constructor
     * @param     in the input stream
     * @exception JDOMException
     *
     */
    public XRepairModeToken( InputStream in ) throws JDOMException, IOException {
        SAXBuilder sax = new SAXBuilder();
        doc = sax.build( in );
        Element root = doc.getRootElement();
        Iterator i = root.getChildren(PARAMETER).iterator();
        while( i.hasNext() ) {
            Element p = (Element)i.next();
            if( ON_OPEN_TA.equals(p.getAttributeValue(NAME)) ) {
                if( WAIT_ON_TA.equalsIgnoreCase(p.getText()) )
                    onOpenTA = ON_OPEN_TA_WAIT;
                else if( ERROR_ON_TA.equalsIgnoreCase(p.getText()) )
                    onOpenTA = ON_OPEN_TA_ERROR;
                else if( ROLLBACK_ON_TA.equalsIgnoreCase(p.getText()) )
                    onOpenTA = ON_OPEN_TA_ROLLBACK;
            }
            else if( WAIT_TIMEOUT.equals(p.getAttributeValue(NAME)) ) {
                try {
                    long wt = (new Long(p.getText())).longValue();
                    waitTimeout = wt;
                }
                catch( NumberFormatException x ) {}
            }
        }
    }
    
    public XRepairModeToken( OnOpenTaToken token ) {
        this.onOpenTA = token.action;
        this.waitTimeout = token.waitTimeout;
        doc = new Document( (Element)null );
        Element rootElm = new Element( REPAIRMODETOKEN );
        doc.setRootElement( rootElm );
        Element pElm;
        pElm = new Element( PARAMETER );
        rootElm.addContent( pElm );
        pElm.setAttribute( NAME, ON_OPEN_TA );
        switch( onOpenTA ) {
            case ON_OPEN_TA_WAIT:
                pElm.addContent( WAIT_ON_TA );
                break;
            case ON_OPEN_TA_ERROR:
                pElm.addContent( ERROR_ON_TA );
                break;
            case ON_OPEN_TA_ROLLBACK:
                pElm.addContent( ROLLBACK_ON_TA );
                break;
            default:
        }
        pElm = new Element( PARAMETER );
        rootElm.addContent( pElm );
        pElm.setAttribute( NAME, WAIT_TIMEOUT );
        pElm.addContent( String.valueOf(waitTimeout) );
    }
    
    /**
     * Accessor for onOpenTA.
     * @return open transaction handling
     */
    public int getOnOpenTA() {
        return onOpenTA;
    }
    
    /**
     * Accessor for waitTimeout.
     * @return  number of ms to wait
     */
    public long getWaitTimeout() {
        return waitTimeout;
    }
    
    /**
     * Document accessor.
     * @return  document
     */
    public Document getDocument() {
        return doc;
    }
    
    /**
     * Document outputter.
     * @exception IOException
     * @return document as a string
     */
    public String outputDocument() throws IOException {
        XMLOutputter outputter = JDom.outputter();
        return outputter.outputString( doc );
    }
    
    /**
     * Document outputter uses giving output stream.
     * @exception IOException
     * @param  out  outputstream
     */
    public void outputDocument( OutputStream out ) throws IOException {
        XMLOutputter outputter = JDom.outputter();
        outputter.output( doc, out );
    }
    
    /**
     * Returns a string representation of the object.
     * @return  object as a string
     */
    public String toString() {
        try {
            return outputDocument();
        }
        catch( IOException x ) {
            x.printStackTrace();
            return x.getMessage();
        }
    }
}

