/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/tools/stores/Parameters.java,v 1.3 2004/07/30 06:52:06 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:06 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.tools.stores;


import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.slide.util.JDom;
import org.apache.slide.util.Strings;
import org.apache.slide.util.XAssertionFailed;
import org.apache.slide.util.XException;
import org.jdom.Element;

/**
 ** Kind of a hashmap that preserves the order and has to/from xml methods. Stores strings only,
 ** does not know the Parameter class.
 **
 ** @author michael.hartmeier@softwareag.com
 ** @version $Revision: 1.3 $
 */
public class Parameters {
    private static final Parameters EMPTY_PARMS = new Parameters();

    /** no HashMap to preserve order **/
    private final List names;
    
    /** inv: never contains null, same size as names. **/
    private final List values;
    
    public Parameters() {
        names = new ArrayList();
        values = new ArrayList();
    }
    
    public Parameters(Parameters orig) {
        names = new ArrayList(orig.names);
        values = new ArrayList(orig.values);
    }

    public void substitute(String variable, String value) {
        int i;
        int max;
        
        max = values.size();
        for (i = 0; i < max; i++) {
            values.set(i, Strings.replace((String) values.get(i), variable, value));
        }
    }

    public int size() {
        return names.size();
    }
    
    public void clear() {
        names.clear();
        values.clear();
    }

    public String get(String name) {
        return get(name, null);
    }
    
    public String get(String name, String dflt) {
        int i;
        
        i = lookup(name);
        if (i == -1) {
            return dflt;
        } else {
            return (String) values.get(i);
        }
    }

    /** @return this to allow expressions like p.set().set() **/
    public Parameters set(String name, String value) {
        if (name == null) {
            throw new XAssertionFailed();
        }
        if (value == null) {
            throw new XAssertionFailed();
        }
        
        int i;
        i = lookup(name);
        if (i == -1) {
            names.add(name);
            values.add(value);
        } else {
            values.set(i, value);
        }
        return this;
    }
    
    /**
     ** Caution: returns false if only the ordering of names/values differs
     **/
    public boolean equals(Object obj) {
        Parameters parms;
        
        if (obj instanceof Parameters) {
            parms = (Parameters) obj;
            return names.equals(parms.names) && values.equals(parms.values);
        } else {
            return false;
        }
    }
    
    private int lookup( String name) {
        int i;
        int max;
        
        max = names.size();
        for (i = 0; i < max; i++) {
            if (name.equals(names.get(i))) {
                return i;
            }
        }
        return -1;
    }
    /**
     ** Copy from src to this
     **/
    public void setAll(Parameters src) {
        setAll(src, EMPTY_PARMS);
    }

    /**
     ** @param src name/values to be assigned
     ** @param excludes don't set name/value from src if it's contained here
     **/
    public void setAll(Parameters src, Parameters excludes) {
        int i;
        int max;
        String name;
        String value;
        
        max = src.names.size();
        for (i = 0; i < max; i++) {
            name = (String) src.names.get(i);
            value = (String) src.values.get(i);
            if (value == null) {
                throw new XAssertionFailed();
            }
            if( !value.equals(excludes.get(name, null)) ) {
                set(name, value);
            }
        }
    }
    
    //-- xml
    
    public static final String PARAMETER = "parameter";

    public static Parameters fromXml(Element root) throws XException {
        List lst;
        Iterator i;
        String name;
        Element p;
        Parameters parms;
        
        parms = new Parameters();
        lst = root.getChildren(PARAMETER);
        i = lst.iterator();
        while( i.hasNext() ) {
            p = (Element) i.next();
            name = JDom.getAttribute(p, XDomainConstants.NAME);
            parms.set(name, p.getText());
        }
        return parms;
    }
    
    public static void removeFromXml(Element root) {
        root.getChildren(PARAMETER).clear();
    }
    
    public void addXml(Element root) {
        int i;
        int max;
        Element parameter;
        
        max = names.size();
        for (i = 0; i < max; i++) {
            parameter = new Element( PARAMETER );
            parameter.setAttribute( XDomainConstants.NAME, (String) names.get(i) );
            parameter.setText( (String) values.get(i) );
            root.addContent( parameter );
        }
    }
}

