/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/tools/stores/XDomainConfig.java,v 1.3 2004/07/30 06:52:06 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:06 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.tools.stores;


import org.apache.slide.util.Pattern;
import org.apache.slide.util.XAssertionFailed;
import org.apache.slide.util.XException;
import org.jdom.Element;

/**
 ** <p>Represents global arguments, namely the DeltaV configuration. Stupid, immutable data container.
 ** No synchronization because an immutable object is always thread-safe. Does not derive from
 ** Parameters because I want to expose immutable methods only.</p>
 **
 ** <p>Technically, there are two more global properties: "standardLivePropertiesClass", and
 ** "uriRedirectorClass". They are not exposed to the user (i.e. shown in neither Domain.xml nor
 ** deltav.xml nor represented here) because the user cannot change them to meaningful values.
 ** If not specified, the system assumes the only valid values.</p>
 **
 ** @author michael.hartmeier@softwareag.com
 ** @version $Revision: 1.3 $
 **/
public class XDomainConfig {
    public static final Parameter NORMAL_SCOPE_PATTERN = new Parameter("normalpath", "/${store}");
    public static final Parameter HISTORY_SCOPE_PATTERN = new Parameter("historypath", "/history/${store}");
    public static final Parameter WORKSPACE_SCOPE_PATTERN = new Parameter("workspacepath", "/workspace/${store}");
    public static final Parameter WORKINGRESOURCE_SCOPE_PATTERN = new Parameter("workingresourcepath", "/workingresource/${store}");
    
    public static final Parameter AUTO_VERSION = new Parameter("auto-version", "checkout-checkin");
    public static final Parameter AUTO_VERSION_CONTROL = new Parameter("auto-version-control", "false");
    public static final Parameter VERSIONCONTROL_EXCLUDE = new Parameter("versioncontrol-exclude", "/administration");
    public static final Parameter CHECKOUT_FORK = new Parameter("checkout-fork", "forbidden");
    public static final Parameter CHECKIN_FORK = new Parameter("checkin-fork", "forbidden");
    
    public static final Parameter[] PATHS = {
        HISTORY_SCOPE_PATTERN, WORKSPACE_SCOPE_PATTERN, WORKINGRESOURCE_SCOPE_PATTERN
    };
    
    public static final Parameter[] ALL = {
        HISTORY_SCOPE_PATTERN, WORKSPACE_SCOPE_PATTERN, WORKINGRESOURCE_SCOPE_PATTERN,
            AUTO_VERSION, AUTO_VERSION_CONTROL,
            VERSIONCONTROL_EXCLUDE, CHECKOUT_FORK, CHECKIN_FORK
    };
    
    public static XDomainConfig createEmpty() {
        try {
            return new XDomainConfig(new Parameters());
        }
        catch (XException e) {
            throw new XAssertionFailed(e);
        }
    }
    
    public static XDomainConfig createDefault() {
        try {
            return new XDomainConfig(createDefaultParameter());
        }
        catch (XException e) {
            throw new XAssertionFailed(e);
        }
    }
    
    private static Parameters createDefaultParameter() {
        Parameters parms;
        
        parms = new Parameters();
        Parameter.initialize(parms, ALL);
        return parms;
    }
    
    
    //--
    
    private final Parameters parms;
    
    /**
     * Default constructor
     */
    public XDomainConfig(Parameters parms) throws XException {
        this.parms = parms;
        if( getScopePattern(XDomainConfig.WORKINGRESOURCE_SCOPE_PATTERN).isParameterized() && !getScopePattern(XDomainConfig.HISTORY_SCOPE_PATTERN).isParameterized() )
            throw new XException( "Invalid global parameters: workingresource path is parameterized but history path is not" );
    }
    
    public XStoreGroupTypeList createTypes() {
        return new XStoreGroupTypeList(new XStoreTypeList(this));
    }
    
    public Pattern getScopePattern(Parameter parameter) {
        return new Pattern(parameter.getString(parms));
    }
    
    public String toString() {
        return parms.toString();
    }
    
    public boolean equals(Object obj) {
        if (obj instanceof XDomainConfig) {
            return parms.equals(((XDomainConfig) obj).parms);
        } else {
            return false;
        }
    }
    
    
    public boolean isCompatible(XDomainConfig g) {
        int i;
        
        for (i = 0; i < PATHS.length; i++) {
            if (!getScopePattern(PATHS[i]).equals(g.getScopePattern(PATHS[i]))) {
                return false;
            }
        }
        return true;
    }
    
    //-- xml
    
    public static XDomainConfig fromXml(Element root) throws XException {
        return new XDomainConfig(Parameters.fromXml(root));
    }
    
    public Element toConfigXml() {
        Element config;
        
        config = new Element("configuration");
        parms.addXml(config);
        return config;
    }
}

