/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/accesscontroler/URMAclAdministrator.java,v 1.4 2005/03/02 10:53:32 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:32 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.accesscontroler;

import java.util.Set;

import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.common.URMDeleteException;
import org.apache.slide.urm.common.URMInsertException;
import org.apache.slide.urm.common.URMInternalServerException;
import java.util.List;
import org.apache.slide.urm.common.URMPrincipal;
import java.util.Map;
import org.apache.slide.urm.authenticator.URMDBTransactionException;
import org.apache.slide.urm.authenticator.rolemanager.URMRoleManagerSpi;
import org.apache.slide.urm.authenticator.userdb.URMMetadataPropsSpi;

/**
 * It provides an interface to manage the resources.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMAclAdministrator {

    /**
     * Returns the RoleManageSpi
     *
     * @return URMRoleManagerSpi
     */
    public URMRoleManagerSpi getRoleManager();

    /**
     * Returns the MetaDataPropsSpi
     *
     * @return URMMetaDatapropsSpi
     */
    public URMMetadataPropsSpi getMetaDataprops();

    /**
     * Returns the getAccessControler
     *
     * @return URMAccessControlerSpi
     */
    public URMAccessControlerSpi getAccessControler();
     
    /** Returns the {@link URMAcl } object representation
     * of a URI.
     *
     * @param id
     * @return URMResource
     */
    public URMAcl getAcl(String id) throws URMInternalServerException;
    
    /** Returns a List of {@link URMAcl } objects in the order of the URI. The
     * resourceIds in the uri areseparated by a "/".
     *
     * @param aces in the order the ACL objects should be retuned
     * @return URMResource
     */
    public List getAclPath(List aces) throws URMInternalServerException;
    
    /** Creates a {@link URMAcl } object representation
     * of a URI. URM will check if this value can
     * be used as the unique ACL - id.
     *
     * @param id
     * @param owner
     * @param inheritable
     * @return the unique id of the ACL
     * @exception URMAclIdNotUniqueException if an URMAcl with the passed id still exists.
     */
    public String createAcl(String id, URMSubject owner, boolean inheritable)
            throws URMAclIdNotUniqueException, URMInternalServerException, URMInsertException;
    
    /** Deletes a resource.
     *
     * @param id of the acl that should be deleted.
     */
    public void deleteAcl(String id)
            throws URMInternalServerException, URMDeleteException;
    
    /** Creates a {@link URMAction } object
     *
     * @param name of new action
     * @return new created URMAction
     * @exception URMActionIdNotUniqueException thrown if the action name still exists.
     */
    public URMAction createAction(String name)
        throws URMActionIdNotUniqueException, URMInternalServerException, URMInsertException;
    
    /** Deletes anaction
     *
     * @param name of action that should be deleted
     * @exception URMActionInUseException thrown if an ACE exists, using the action.
     */
    public void deleteAction (String name)
        throws URMActionInUseException, URMInternalServerException, URMDeleteException;
    
    /** Returns an action, null if the action does not exist.
     *
     * @param name action name
     * @return URMAction, searched action or null if the action does not exist.
     */
    public URMAction getAction(String name)
        throws URMInternalServerException;
    
    /** Returns all actions (basic and aggregated)
     *
     * @return Set of URMActions
     */
    public Set getAllActions()
        throws URMInternalServerException;
    
    /**
     * Close the connection to the meta store.
     *
     * @throws URMCloseConnectionException if any error occures during tho close operation.
     */
    //public void close() throws URMCloseConnectionException;
    
    /** Processes an access check of a URMSubject.
     *
     * @param principal the principal of the authenticated user.
     * @param subject whose access right should be checked
     * @param resource the resourceid for that the validation should be done.
     * @param action the required action.
     * @return One of the predefined values URMPermit, URMDeny,
     *              URMIndeterminable, URMNotApplicable.
     *
     * @throws URMAccessCheckException if the access check fails.
     *
     * @see URMAclValidator
     */
    public int subjectPermissionReport(URMPrincipal principal, URMSubject subject, List resource, String action)
        throws URMAccessCheckException;
    
    /**
     * Returns a set of URMUsers that are allowed to execute the specified action
     * at the passed resource.
     *
     * @param principal authenticated user who invokes this report method
     * @param action the action against the validation should be done.
     * @param resources List of ACLs in the order they should be validated
     *
     * @return the Set of URMUsers that are allowed to execute the passed action
     *         at the specified resource. If the action is not allowed for any user,
     *         the Set will be empty
     *
     * @throws URMInternalServerException if any error occures
     **/
    public Set userWithPermissionReport(URMPrincipal principal, String action, List resources)
        throws URMInternalServerException;
    
    /**
     * Returns a map of names (key) and Boolean (value) containing the actions
     * that are specified for the passed subject at the resource. If the value is
     * 'true' the action is granted, if false the action is denied. Inherited
     * actions are considered as well.
     *
     * @param principal authenticated user who invokes this report method
     * @param subject whose rights should be returned
     * @param resource the resourceid for that the validation should be done.
     *
     * @return names and value of the URMActions that are defined for the user at
     *         the resource.
     *
     * @throws URMInternalServerException if any error occures
     **/
    public Map subjectPermissionSetReport(URMPrincipal principal, URMSubject subject, List resource)
        throws URMAccessCheckException;
        
    /** start a database transaction
     *
     * @exception URMDBTransactionException
     */
    public void beginTransaction() throws URMDBTransactionException, URMInternalServerException;
    
    /** ends a database transaction
     *
     * @param manner true=commit false=rollback
     * @exception URMDBTransactionException
     */
    public void endTransaction(boolean manner) throws URMDBTransactionException, URMInternalServerException;
        
    /** Removes the needless entries from the user-role store and the metaproperties.
     *  It checks which users and groups were removed from the user database and
     *  removes the belonging entries from the user-role store and the metaproperties.
     *
     * @throws URMNotImplementedException
     * @throws URMForbiddenException
     * @throws URMInternalServerException
     * @throws URMDeleteException
     */
    
    public List getAclsWithSpecificAction(List resources,
                                             List inActions)
        throws URMInternalServerException;
    
    public void loadActions(Set actions) throws URMInternalServerException;

    /** Returns the current Principal Owner Handle
     *
     */
    public Object getOwner();
     
    /**
     * Signals the cache that an ACL does not exist
     */
    public void addTransientEmptyAcl (String aclId);
}

