/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/accesscontroler/URMSemantic.java,v 1.4 2005/03/02 10:53:32 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:32 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.accesscontroler;

import org.apache.slide.urm.common.URMPrincipal;
import java.util.Set;
import org.apache.slide.urm.common.URMInternalServerException;
import java.util.Map;
import java.util.List;
import org.apache.slide.urm.authenticator.URMSubject;

/**
 * This interface represents a semantic of an access check.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 *
 * @see URMAclValidator
 */
public interface URMSemantic {
    
    /** Processes an access check.
     *
     * @param principal the principal of the authenticated user.
     * @param resource the resourceid for that the validation should be done.
     * @param action the required action.
     * @return One of the predefined values URMPermit, URMDeny,
     *              URMIndeterminable, URMNotApplicable.
     *
     * @throws URMAccessCheckException if the access check fails.
     *
     * @see URMAclValidator
     */
    public int checkPermission(URMPrincipal principal, List resource, String action)
        throws URMAccessCheckException;

    /** Processes an access check of a URMSubject.
     *
     * @param principal the principal of the authenticated user.
     * @param subject whose access right should be checked
     * @param resource the resourceid for that the validation should be done.
     * @param action the required action.
     * @return One of the predefined values URMPermit, URMDeny,
     *              URMIndeterminable, URMNotApplicable.
     *
     * @throws URMAccessCheckException if the access check fails.
     *
     * @see URMAclValidator
     */
        public int subjectPermissionReport(URMPrincipal principal, URMSubject subject, List resource, String action)
        throws URMAccessCheckException;
    
    /**
     * Returns a set of URMUsers that are allowed to execute the specified action
     * at the passed resource.
     *
     * @param principal authenticated user who invokes this report method
     * @param action the action against the validation should be done.
     * @param resources List of ACLs in the order they should be validated
     *
     * @return the Set of URMUsers that are allowed to execute the passed action
     *         at the specified resource. If the action is not allowed for any user,
     *         the Set will be empty
     *
     * @throws URMInternalServerException if any error occures
     **/
    public Set userWithPermissionReport(URMPrincipal principal, String action, List resources)
        throws URMInternalServerException;
    
    /**
     * Returns a map of names (key) and Boolean (value) containing the actions
     * that are specified for the passed subject at the resource. If the value is
     * 'true' the action is granted, if false the action is denied. Inherited
     * actions are considered as well.
     *
     * @param principal authenticated user who invokes this report method
     * @param subject whose rights should be returned
     * @param resource the resourceid for that the validation should be done.
     *
     * @return names and value of the URMActions that are defined for the user at
     *         the resource.
     *
     * @throws URMInternalServerException if any error occures
     **/
    public Map subjectPermissionSetReport(URMPrincipal principal, URMSubject subject, List resource)
        throws URMAccessCheckException;
}
