/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/accesscontroler/URMSemanticFactory.java,v 1.4 2005/03/02 10:53:32 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:32 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.accesscontroler;


import org.apache.slide.urm.URMException;
import org.apache.slide.urm.common.URMConfigurator;
import org.apache.slide.urm.common.URMConfigurationException;
import org.apache.slide.urm.common.URMConstants;
import org.apache.slide.urm.utils.URMClassInstantiator;
import org.apache.slide.urm.utils.messagelogger.MessageLogger;

/**
 * Factory class of {@link URMSemantic }.
 * It can instantiate a URMSemanticFactory implementation using a
 * URMConfigurator which can be created e.g. out of a xml file.
 * The URMConfigurator conatains the name of the implementing factory class
 * and further attributes which are required for this specific implementation.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public abstract class URMSemanticFactory {
    
    private static org.apache.log4j.Logger msLogger =
        org.apache.log4j.Logger.getLogger(URMSemanticFactory.class.getName());
        
    /**
     * Creates a new factory using the specified configuration.
     *
     * @param semanticConf that should be used to create the factory and
     *          will be used to create further object using the this factory
     *
     * @return URMSemanticFactory object
     *
     * @throws URMConfigurationException if a wrong configuration was passed
     *
     * @throws URMException if an unpredictable internal error occurs
     */
    public static URMSemanticFactory newInstance(URMConfigurator semanticConf)
            throws URMConfigurationException, URMException {
        
        URMSemanticFactory factory = null;
        
        if (semanticConf == null)
            throw new URMConfigurationException(MessageLogger.getAndLogMessage(
                                msLogger, "URMCOC0017", "AclSemantic"));
                                
        String class_name = semanticConf.getProperties().getProperty(URMConstants.FACTORY_CLASS_ATTRIBUTE_NAME);
        
        if (class_name == null)
            throw new URMConfigurationException(
                MessageLogger.getAndLogMessage(msLogger, "URMCOC0002",
                    URMConstants.FACTORY_CLASS_ATTRIBUTE_NAME, "AclSemantic"));

        Object s_params[] = { semanticConf };
        try {
            factory = (URMSemanticFactory)URMClassInstantiator.newInstance(class_name, s_params);
        } catch (Exception e) {
            throw new URMConfigurationException("Cannot instantiate defined URMSemanticFactory class");
        }
        return factory;
    }
    
    public abstract URMSemantic newSemantic() throws URMException;
}
