/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/authenticator/rolemanager/URMRole.java,v 1.4 2005/03/02 10:53:34 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:34 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.authenticator.rolemanager;

import java.util.Properties;
import java.util.Set;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.URMNotImplementedException;
import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.common.URMDeleteException;
import org.apache.slide.urm.common.URMInsertException;
import org.apache.slide.urm.common.URMInternalServerException;
import org.apache.slide.urm.common.URMUpdateException;
import java.util.Map;

/**
 * The Role is represented by a String and 0..n ancestors
 * that define the enherited roles.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMRole extends URMSubject{

    /**
     * Returns the name of the role.
     *
     * @return Name of the role.
     */
    public String getName();
    
    /**
     * Returns all users, mapped to the current role.
     *
     * @return Set of URMUser's.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMQueryException if an unpredictable internal error occurs.
     */
    public Set getUserMapping()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /**
     * Returns all groups, mapped to the current role.
     *
     * @return Set of URMGroup's.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMQueryException if an unpredictable internal error occurs.
     */
    public Set getGroupMapping()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /**
     * Returns all ancestors direct of the current role.
     * I.e. all roles that the current one has inhereted.
     *
     * @return Set of URMRole's.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMQueryException if an unpredictable internal error occurs.
     */
    public Set getAncestors()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /**
     * Returns all (direct and indirect inherited) inherited ancestors of the current role.
     * I.e. all roles that the current one has inhereted.
     *
     * @return Set of URMRole's.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMQueryException if an unpredictable internal error occurs.
     */
    public Set getAllAncestors()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /**
     * Returns all roles where the current instance is an ancestor.
     * I.e. in which roles appears the current instance as an ancestor.
     *
     * @return Set of URMRole's.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMQueryException if an unpredictable internal error occurs.
     */
    public Set getRolesOfAncestor()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Add a predefined action for a role.
     * During access check, all predefined actions are automatically granted or denied
     * to the role.
     *
     * @param action new predefined action
     * @param positive positive or negative action
     * @throws URMNotImplementedException
     * @throws URMForbiddenException
     * @throws URMUpdateException
     */
    public void setPredefinedAction(String action, Boolean positive)
        throws URMNotImplementedException, URMForbiddenException,
                URMInsertException, URMInternalServerException;
    
    /** Retrieves the predefined actions for a role.
     * During access check, all predefined actions are automatically checked
     *
     * @return Map of action String names (key) and Boolean positive/negative (value)
     * @throws URMNotImplementedException
     * @throws URMForbiddenException
     * @throws URMUpdateException
     */
    public Map getPredefinedActions()
        throws URMNotImplementedException, URMForbiddenException,
                URMInternalServerException;
    
    /** Deletes the predefined actions for a role.
     *
     * @param  action action that should be deleted
     * @throws URMNotImplementedException
     * @throws URMForbiddenException
     * @throws URMUpdateException
     */
    public void deletePredefinedAction(String action)
        throws URMNotImplementedException, URMForbiddenException,
                URMDeleteException, URMInternalServerException;
    
    /** Adds a new user mapping to a role
     *
     * @param user that should be mapped
     * @param domain the users domain if defined, null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addUserLink(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Adds a new group mapping to a role
     *
     * @param group that should be mapped
     * @param domain the users domain if defined, null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addGroupLink(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Adds a new role mapping to a role. Role loops ar not allowed.
     *
     * @param ancestor name of the ancestor role.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addAncestor(String ancestor)
        throws URMNotImplementedException, URMForbiddenException,
               URMInsertException, URMInternalServerException;


    /** Delets a user to role mapping
     *
     * @param user the user that should be removed from the role
     * @param domain the users domain if defined, null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteUserLink(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /** Delets a group to role mapping
     *
     * @param group the group that should be removed from the role
     * @param domain the users domain if defined, null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteGroupLink(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;

    /** Delets a role to role mapping
     *
     * @param ancestor the ancestor that should be removed from the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteAncestor(String ancestor)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;

    /**
     * Retrieves subject specific properties
     *
     * @return Properties
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Properties getProperties()
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException;

    /**
     * Sets new subject specific properties
     *
     * @param props
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setProperties(Properties props)
        throws URMNotImplementedException, URMForbiddenException,
            URMUpdateException, URMDeleteException, URMInternalServerException;
            
    /**
     * Retrieves a specific property
     *
     * @param key property key
     * @return value, property value or null if the property has not been found
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Object getProperty(String key)
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException, URMInternalServerException;
            
    /**
     * Retrieves a aggregated set of values of a specific property. The result set
     * contains the values of the property defined for the ancestors of the subject.
     *
     * @param key property key
     * @return value, aggregated property value
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAggregatedProperty(String key)
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException;
                            

    /**
     * Sets new subject specific properties
     *
     * @param key property key
     * @param prop property value
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setProperty(String key, Object prop)
        throws URMNotImplementedException, URMForbiddenException,
            URMUpdateException, URMDeleteException, URMInternalServerException;
    
    /** Returns key, value pairs where the key is the name of available
     * propery and the value is a java.lang.Boolean object which is true
     * if the property can be modified (writeable flag).
     *
     * @return a Map containing property name and writeable flag pairs
     */
    public Map getAvailablePropertyNames() throws URMForbiddenException;
}
