/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/authenticator/rolemanager/URMRoleManagerSpi.java,v 1.4 2005/03/02 10:53:34 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:34 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.authenticator.rolemanager;

import java.util.Map;
import java.util.Set;
import java.util.Properties;

import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.URMNotImplementedException;
import org.apache.slide.urm.authenticator.URMDBTransactionException;
import org.apache.slide.urm.common.URMCloseConnectionException;
import org.apache.slide.urm.common.URMConnectionException;
import org.apache.slide.urm.common.URMDeleteException;
import org.apache.slide.urm.common.URMInsertException;
import org.apache.slide.urm.common.URMInternalServerException;
import org.apache.slide.urm.common.URMUpdateException;

/**
 *
 * Service Provider Interface definition for the native MetaDataAccess of the role manager package.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMRoleManagerSpi {

    /** Checks whether the role of a specific name exists.
     *
     * @param role name of a role that should be checked.
     * @return true if this is the name of an existing role.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs.
     */
    public boolean isRole(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /** Creates a new Role.
     * All role must have unique names.
     *
     * @param role name.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void createRole(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Deletes the role of the given name.
     *
     * @param role name.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteRole(String role)
        throws URMNotImplementedException, URMInternalServerException,
               URMForbiddenException, URMDeleteException;
    
    /** Returns a set of all existing roles.
     *
     * @return set of roles.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllRoles()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Sets the profile of a role.
     *
     * @param roleName object
     * @param profileProperties profile of the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMUpdateException if an unpredictable internal error occurs
     */
    public void setProfileOfRole(String roleName, Properties profileProperties)
        throws URMNotImplementedException, URMForbiddenException,
               URMUpdateException, URMInternalServerException;
    
    /** Deletes the profile of a role.
     *
     * @param roleName object
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMUpdateException if an unpredictable internal error occurs
     */
    public void deleteProfileOfRole(String roleName)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /** Sets the predefined actions of a role.
     *
     * @param roleName object
     * @param action that should be defined
     * @param type true if action should be denied, flase else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setPredefinedAction(String roleName, String action, Boolean type)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Retrieves the predefined actions of a role.
     *
     * @param roleName object
     * @return predefined actions of the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Map getPredefinedActions(String roleName)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Deletes the predefined actions of a role.
     *
     * @param roleName object
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deletePredefinedAction(String roleName, String action)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /** Gets the profile of a role.
     *
     * @param roleName object
     * @return profile of the role.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Properties getProfileOfRole(String roleName)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Adds a new user mapping to a role
     *
     * @param role name of a role.
     * @param user that should be mapped
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addUserLink(String role, String domain, String user)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Adds a new group mapping to a role
     *
     * @param role name of a role.
     * @param group that should be mapped
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addGroupLink(String role, String domain, String group)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;
    
    /** Adds a new role mapping to a role
     *
     * @param ancestor name of the ancestor role.
     * @param role name of a role that should be mapped
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void addAncestor(String role, String ancestor)
        throws URMNotImplementedException, URMForbiddenException,
               URMInsertException, URMInternalServerException;

    /** Delets a user to role mapping
     *
     * @param role name of the role.
     * @param user name the user that should be removed from the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteUserLink(String role, String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /** Delets a group to role mapping
     *
     * @param role name of the role.
     * @param group name the group that should be removed from the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteGroupLink(String role, String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;

    /** Delets a role to role mapping
     *
     * @param role name of the role.
     * @param ancestor name the ancestor that should be removed from the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteAncestor(String role, String ancestor)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /** Returns all the roles, mapped to a specific user
     *
     * @param user name of the user.
     * @return Set of URMSubject's (roles), associated with the user
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getRoleMappingOfUser(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the roles (inclusive all inherited roles), mapped to a specific user
     *
     * @param user name of the user.
     * @return Set of URMSubject's (roles), associated with the user
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllRoleMappingOfUser(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the roles (inclusive all inherited roles), mapped to a specific group
     *
     * @param group name of the group.
     * @return Set of URMSubject's (roles), associated with the group
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getRoleMappingOfGroup(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the roles, mapped to a specific group
     *
     * @param group name of the group.
     * @return Set of URMSubject's (roles), associated with the group
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllRoleMappingOfGroup(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the users, associated with a specific role
     *
     * @param role where the user mapping is requested.
     * @return Set of URMSubject's (user), associated with the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getUserMappingOfRole(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the groups, associated with a specific role
     *
     * @param role where the group mapping is requested.
     * @return Set of URMSubject's (group), associated with the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getGroupMappingOfRole(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the (first level) ancestors of a specific role
     *
     * @param role name of the Role.
     * @return Set of all ancestor URMRole's of the specific role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAncestors(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the ancestors (recursive) of a specific role
     *
     * @param role name of the Role.
     * @return Set of all ancestor URMRole's of the specific role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllAncestors(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Returns all the ancesters of a specific role
     *
     * @param role name of the Role.
     * @return Set of URMRole's that have the parameter "role" as ancestor
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getRolesOfAncestor(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /** Close the connection to the meta store.
     *
     * @throws URMCloseConnectionException if any error occures during tho close operation.
     */
    public void close() throws URMCloseConnectionException;
    
    /**
     * Start a new database transcation (i.e. to access the MetaDataStore
     *
     * @throws URMDBTransactionException
     */
    public void beginTransaction() throws URMDBTransactionException, URMInternalServerException;
    
    /**
     * Terminate a running DB transaction.
     * @param manner true=commit false=rollback
     * @throws URMDBTransactionException
     */
    public void endTransaction(boolean manner) throws URMDBTransactionException, URMInternalServerException;

    /** Removes the needless entries from the user-role store.
     *  It checks which users and groups were removed from the user database and
     *  removes the belonging entries from the user-role store.
     *
     * @throws URMInternalServerException
     * @throws URMDeleteException
     */
    public void synchronizeUsersGroupsRoles(String domain, Set existingUsers, Set existingGroups)
            throws URMInternalServerException, URMDeleteException;

    /** Returns the writeable metaproperties of the roles.
     *
     * @return Set
     */
    public Set getWriteableRolePropKeys();

    /** Returns all the available metaproperties of the roles.
     *
     * @return Set
     */
    public Set getAvailableRolePropKeys();
    
    
    /**
     * Deletes the RoleMapping of a group or user
     *
     * @param    name user name
     * @param    domain user domain
     * @param    isUser a  if 'true', delete role mapping of user, if 'false' delete role mapping of group
     *
     * @throws   URMDeleteException
     * @throws   URMInternalServerException
     *
     */
    public void deleteRoleMappingAndACEsOfUserGroupSubject(String name, String domain, boolean isUser) throws URMDeleteException, URMInternalServerException;

    /** Checks weather the given user, group or role has some ACEs defined.
     *
     * @param name
     * @param domain
     * @param subjType USER, GROUP or ROLE
     * @throws URMDeleteException
     * @throws URMInternalServerException
     */
    public boolean hasACEsOfSubject(String name, String domain, int subjType) throws URMInternalServerException;

    public Set getAvailableDomains() throws URMInternalServerException;
        
    public URMRoleManagerSpi duplicateObject(Object owner) throws URMConnectionException;
    
}

