/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/authenticator/userdb/URMUser.java,v 1.4 2005/03/02 10:53:34 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:34 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.authenticator.userdb;

import java.util.Properties;
import java.util.Set;
import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.URMNotImplementedException;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.common.URMDeleteException;
import org.apache.slide.urm.common.URMInsertException;
import org.apache.slide.urm.common.URMInternalServerException;
import org.apache.slide.urm.common.URMUpdateException;
import java.util.Map;

/**
 * Represents a userdb user.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMUser extends URMSubject{
    
    /**
     * Returns an Set over all groups where the user is a member of
     *
     * @return Set over all groups where the user is a member of
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getGroups()
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException;
    
    /**
     * Returns an Set over all groups where the user is a member of
     * and the groups where these groups are membgers recursively.
     *
     * @return Set over all groups where the user is a member of
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllGroups()
            throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
                
    /**
     * Changes the password of a user in the user database
     *
     * @param oldpassword the user's current password
     *
     * @param newpassword the desired password of the userd
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setPassword(char[] oldpassword, char[] newpassword)
        throws URMNotImplementedException, URMForbiddenException, URMUpdateException;

    /** Mapps this subject to a role, so that the subject can take this role
     *
     * @param role name of the role.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void linkToRole(String role)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException, URMInternalServerException;

    /** Delets a role link to the subject. The subject is no longer able to take this role
     *
     * @param role name the ancestor that should be removed from the role
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteRoleLink(String role)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;

    /** Returns all the roles, mapped to the subject and where the subject can take this roles
     *
     * @return Set of URMRole Object, associated with the group
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getRoleMapping()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /** Returns all the roles(inclusive all inherited roles), mapped to the subject and where the subject can take this roles
     *
     * @return Set of URMRole Object, associated with the group
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllRoleMapping()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /**
     * Retrieves subject specific properties
     *
     * @return Properties
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Properties getProperties()
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException;

    /**
     * Sets new subject specific properties
     *
     * @param props new properties set
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setProperties(Properties props)
        throws URMNotImplementedException, URMForbiddenException,
            URMUpdateException, URMDeleteException, URMInternalServerException;
            
    /**
     * Retrieves a specific property
     *
     * @param key property key
     * @return value, property value or null if the property has not been found
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Object getProperty(String key)
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException, URMInternalServerException;
            
    /**
     * Retrieves a aggregated set of values of a specific property. The result set
     * contains the values of the property defined for the ancestors (groups) of the subject.
     *
     * @param key property key
     * @return value, aggregated property value
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAggregatedProperty(String key)
        throws URMNotImplementedException, URMForbiddenException,
            URMInternalServerException;
                            

    /**
     * Sets new subject specific properties
     *
     * @param key property key
     * @param prop property value
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void setProperty(String key, Object prop)
        throws URMNotImplementedException, URMForbiddenException,
            URMUpdateException, URMDeleteException, URMInternalServerException;
    
    /** Returns key, value pairs where the key is the name of available
     * propery and the value is a java.lang.Boolean object which is true
     * if the property can be modified (writeable flag).
     *
     * @return a Map containing property name and writeable flag pairs
     */
    public Map getAvailablePropertyNames() throws URMForbiddenException;
}
