/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/authenticator/userdb/URMUserDBManager.java,v 1.4 2005/03/02 10:53:34 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:34 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.authenticator.userdb;

import java.util.Set;

import org.apache.slide.urm.URMNotImplementedException;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.common.URMDeleteException;
import org.apache.slide.urm.common.URMInsertException;
import org.apache.slide.urm.common.URMInternalServerException;

/**
 * It provides an interface to administrate the user database.
 * There are methods to create, remove users and groups, or
 * get the members of a group or the groups of a user or
 * the properties of a specific user etc.
 * The properties can have two parts. A base set of properties usually takes
 * place in the native user database (against the user will be authenticated
 * e.g. LDAP) and additional user properties can be stored in an other
 * database (e.g. Tamino) as metadata.
 * A mapping must be defined between a property names (that should be
 * independent from the underlying native user database) and the database field
 * names (e.g. in Tamino or LDAP) in the initializer URMConfigurator.
 * You can check whether the property is stored in the native user
 * database or as metadata in an other database with the getOrigin() method of the
 * URMUserProperty class.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMUserDBManager {
    /**
     * Returns an Set over all known groups of the user database
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @return Set over all known groups of the user database
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllGroups(String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /**
     * Returns an Set over all known users of the user database
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @return Set over all known users of the user database
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public Set getAllUsers(String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
    
    /**
     * Creates a new user in the user database
     *
     * @param user name of the user that should be added
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void createUser(String user, char[] password, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException;
    
    /**
     * Creates a new group in the user database
     *
     * @param group name of the group that should be added
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void createGroup(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInsertException;
    
    /**
     * Removes a user from the user database
     *
     * @param user name of the user that should be removed
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteUser(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;
    
    /**
     * Removes a group from the user database
     *
     * @param group name of the group that should be removed
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     */
    public void deleteGroup(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMDeleteException, URMInternalServerException;

    /**
     * Verifies if the user exists in the userDB.
     *
     * @param user name of the user whose properties are searched.
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @return true is user exists, false else.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs.
     */
    public boolean isUser(String user, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
            
    /**
     * Verifies if the group exists in the userDB.
     *
     * @param group name of the group whose properties are searched.
     *
     * @param domain the domain against the operation is executed if a domain is defined,
     * null else
     *
     * @return true is group exists, false else.
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs.
     */
    public boolean isGroup(String group, String domain)
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;
}
