/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/urm/org/apache/slide/urm/common/URMPrincipal.java,v 1.4 2005/03/02 10:53:35 eckehard Exp $
 * $Revision: 1.4 $
 * $Date: 2005/03/02 10:53:35 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.urm.common;

import java.security.Principal;
import java.util.Properties;
import java.util.Set;

import org.apache.slide.urm.URMException;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.URMNotImplementedException;
import org.apache.slide.urm.accesscontroler.URMAclAdministrator;
import org.apache.slide.urm.authenticator.URMAdministrator;
import org.apache.slide.urm.authenticator.rolemanager.URMRole;

/**
 * The URMPrincipal is the main object that represents an
 * authenticated user.
 *
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public interface URMPrincipal extends Principal {

    /**
     * Compares two URMPrincipal objects.
     *
     *  @return true if this instances equals the given input parameter.
     */
    boolean equals(Object parm);
    
    /**
     * Returns the user name concatenated with the domain name of the current object.
     *
     *  @return Name of object.
     */
    public String getName();
    
    /**
     * Returns the domain name of the current object.
     *
     *  @return Domain name of object.
     */
    public String getDomain();
    
    /**
     * Returns only the user name of the current object.
     *
     *  @return User name of object.
     */
    public String getUserName();
    
    /**
     * Returns a hash code for this principal.
     *
     *  @return a hash code value for this object.
     */
    public int hashCode();
    
    /**
     * Returns a URMAdministrator object.
     *
     * @return URMAdministrator
     */
    public URMAdministrator getURMAdministrator() throws URMException;
    
    /**
     * Returns a URMAclAdministrator object.
     *
     * @return URMAclAdministrator
     */
    public URMAclAdministrator getURMAclAdministrator() throws URMException;

    /**
     * Returns a hash set of all user properties of the current principal
     *
     * @return HashSet
     */
    public Properties getProperties() throws URMNotImplementedException, URMException;
    
    /**
     * Returns all groups where the current principal belongs to.
     *
     * @return Set of groups.
     */
    public Set getGroups() throws URMNotImplementedException, URMException;
    
    /**
     * Returns all possible roles that the current principal can choose from.
     *
     * @return Returns all Role objects within a Set.
     */
    public Set getPossibleRoles() throws URMNotImplementedException, URMException;
    
    /**
     * Returns a specific role out of the the of the users possible roles.
     *
     * @return Returns all Role objects within a Set.
     */
    public URMRole getSpecificPossibleRoles(String name) throws URMNotImplementedException, URMException;
    
    /**
     * Returns all roles that are linked directly to the current principal.
     *
     * @return Returns all Role objects within a Set.
     */
    public Set getPrimaryRoles() throws URMNotImplementedException, URMException;

    /**
     * Sets a specific role into the active state.
     * This method must be called before the URMPrincipal object is functional.
     * Any calls that use an instance of this class without a particular role
     * will result in exception.
     * Note: 'null' as role is possible. In this case, all possible roles will
     * be considered when roles are compared. Carefull: Having null as the active
     * role is more performance intensive during these check operations.
     *
     */
    public void setActiveRole(String role)
        throws URMSetRoleException, URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /**
     * Returns the active role.
     * An exception will be thrown, if no active role has been assigned yet.
     *
     * @return URMRole (can be null!)
     */
    public URMRole getActiveRole()
        throws URMNotImplementedException, URMForbiddenException, URMInternalServerException;

    /**
     * Copy the current instance, but supply a new (different) active role.
     * The role could be null, in which case all available roles will
     * be checked. (Carefull: performance during access check will be better when
     * a non-null role is specified).
     * The resulting URMPrincipal object can be used directly
     *
     */
    public URMPrincipal copyPrincipal(String activeRole)
        throws URMSetRoleException, URMNotImplementedException, URMForbiddenException,
            URMInternalServerException, URMConnectionException;
    
    /**
     * Returns the creation time of the Principal in milliseconds.
     *
     * @return the creation time in milliseconds.
     */
    public long getCreationTime();
     
    /**
     * Changes the password of this principal in the user database
     *
     * @param oldpassword the user's current password
     *
     * @param newpassword the desired password of the userd
     *
     * @exception URMNotImplementedException if the method is not
     *            implemented by the user database.
     * @exception URMForbiddenException if the user has not the permission
     *            to invoke the method.
     * @exception URMException if an unpredictable internal error occurs
     * @exception URMException if an unpredictable error occurs
     */
    public void setPassword(char[] oldpassword, char[] newpassword)
            throws URMInternalServerException, URMNotImplementedException, URMForbiddenException, URMUpdateException;

    /**
     *  Delivers the an authentication ID for Single Sign On prozedure.
     * 
     * @return Authentication ID
     * @throws URMInternalServerException
     * @throws URMNotImplementedException
     * @throws URMForbiddenException
     */
    public char[] getAuthenticationID()
    		throws URMInternalServerException, URMNotImplementedException, URMForbiddenException;
    /**
     * Close the connection to the meta store.
     *
     * @throws URMCloseConnectionException if any error occures during tho close operation.
     */
    //public void close() throws URMCloseConnectionException;
}
