/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/util/org/apache/slide/util/MimeMapping.java,v 1.3 2004/07/30 06:52:17 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:17 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.jdom.Document;
import org.jdom.Element;

/**
 ** Handler for the mime types defined in %TAMINOWEBDAVSERVER_HOME%\web-inf\web.xml.
 **
 ** @author    peter.nevermann@softwareag.com
 ** @version   $Revision: 1.3 $
 **/
public class MimeMapping {
    // text constants
    private static final String
        MIME_MAPPING       = "mime-mapping",
        EXTENSION          = "extension",
        MIME_TYPE          = "mime-type",
        WEB_APP            = "web-app",
        DUMMY              = "@";
    
    //--
    
    /** mime-type -> list(extension) relation */
    private final Map m2e;
    
    /** extension -> list(mime-type) relation */
    private final Map e2m;
    
    
    /**
     ** Default constructor.
     **
     ** @pre        true
     ** @post       true
     **
     ** @exception  XException
     **/
    public MimeMapping() {
        m2e = new HashMap();
        e2m = new HashMap();
    }
    
    public void addAll(Document webXmlDoc) throws XException {
        Iterator i = getMimeMappingElements(webXmlDoc).iterator();
        while( i.hasNext() ) {
            Element mmElm = (Element)i.next();
            add(getMimeType(mmElm), getExtension(mmElm));
        }
    }

    public void add(String mimeType, String extension) {
        addPair( m2e, mimeType, extension );
        addPair( e2m, extension, mimeType );
    }
    
    /**
     ** Get the set of mime-types associated to the given extension.
     **
     ** @pre        true
     ** @post       true
     **
     ** @param      extension the extension
     ** @return     the mime types (empty list if none)
     **/
    public List getMimeTypes( String extension ) {
        List result = (List)e2m.get( extension );
        if( result == null )
            result = Collections.EMPTY_LIST;
        return result;
    }
    
    /**
     ** Get the set of extensions associated to the given mime-type.
     **
     ** @pre        true
     ** @post       true
     **
     ** @param      mimeType the mime-type
     ** @return     the extensions (empty list if none)
     **/
    public List getExtensions( String mimeType ) {
        List result = (List)m2e.get( mimeType );
        if( result == null )
            result = Collections.EMPTY_LIST;
        return result;
    }
    
    
    // ----------------------------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------------------------
    
    /**
     ** Add a m->e or e->m pair.
     **
     ** @pre        m != null && key != null && value != null
     ** @post       true
     **
     ** @param      m the map
     ** @param      key the key
     ** @param      value the value
     **/
    private static void addPair( Map m, String key, String value ) {
        List vs = null; //set of values
        if( m.containsKey(key) ) {
            vs = (List)m.get( key );
        }
        else {
            vs = new ArrayList();
            m.put( key, vs );
        }
        vs.add( value );
    }
    
    /**
     ** Get the root element.
     **/
    private static Element getRootElement(Document webXmlDoc ) throws XException {
        // get the root
        Element result = webXmlDoc.getRootElement();
        if( result == null )
            throw new XException("missing root element "+WEB_APP );
        return result;
    }
    
    /**
     ** Get all mime-mapping elements.
     **/
    private static List getMimeMappingElements( Document webXmlDoc ) throws XException {
        List result = getRootElement( webXmlDoc ).getChildren( MIME_MAPPING );
        if( result == null )
            result = Collections.EMPTY_LIST;
        return result;
    }
    
    /**
     ** Get the extension of a mime-mapping element.
     **/
    private static String getExtension( Element mmElm ) throws XException {
        Element eElm = mmElm.getChild( EXTENSION );
        if( eElm == null )
            throw new XException( "The mime-mapping element "+mmElm+" has no extension." );
        return eElm.getText();
    }
    
    /**
     ** Get the mime-type of a mime-mapping element.
     **/
    private static String getMimeType( Element mmElm ) throws XException {
        Element mElm = mmElm.getChild( MIME_TYPE );
        if( mElm == null )
            throw new XException( "The mime-mapping element "+mmElm+" has no mime-type." );
        return mElm.getText();
    }
}


