/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/store/org/apache/slide/store/tamino/datastore/schema/XDoctypeCacheTest.java,v 1.1 2004/03/25 16:18:14 juergen Exp $
 * $Revision: 1.1 $
 * $Date: 2004/03/25 16:18:14 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.store.tamino.datastore.schema;

import java.util.HashSet;
import java.util.Set;
import junit.framework.TestCase;
import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.store.tamino.datastore.IContentCollection;
import org.easymock.EasyMock;
import org.easymock.MockControl;
import org.jdom.Document;
import org.jdom.input.SAXBuilder;
import org.xml.sax.InputSource;

public class XDoctypeCacheTest extends TestCase implements XGlobals {
    
    
    Document tsd2Schema;
    Document tsd3Schema;
    String tsd3Name = "x_mydoc.xml";
    IContentCollection mockCollection;
    
    public XDoctypeCacheTest (String name) {
        super (name);
    }
    
    protected void setUp () throws Exception {
        super.setUp();
        
        // read schema
        InputSource in = new InputSource (getClass().getResourceAsStream (tsd3Name));
        tsd3Schema = new SAXBuilder ().build (in);
        
        
        
        MockControl ctrlCollection = EasyMock.controlFor (IContentCollection.class);
        mockCollection = (IContentCollection)ctrlCollection.getMock();
        
        mockCollection.getSchemaVersion();
        ctrlCollection.setReturnValue (TSD3_SCHEMA_VERSION);
        
        mockCollection.getCollectionName ();
        ctrlCollection.setReturnValue ("collection");
        
        mockCollection.getUri ();
        ctrlCollection.setReturnValue
            ("http://localhost/tamino/mydb/collection");
        
        ctrlCollection.activate();
        
        
    }
    
    public void testCache () throws Exception{
        
        XSchema schema = XSchema.newInstance (tsd3Schema.getRootElement());
        Set docTypes = schema.getDoctypes();
        
        
        XDoctypeCache cache = XDoctypeCache.getSingleInstance();
        cache.addDoctype (mockCollection, docTypes);
        
        XDoctype doc = new XDoctype ("x:mydoc", "http://my.doc");
        XDoctype strongDoctype = cache.getDoctype (mockCollection, doc.getFullQualifiedName());
        
        assertEquals ("http://my.doc", strongDoctype.getNamespaceUri ());
        assertEquals ("x:mydoc", strongDoctype.getPrefixedName ());
        
        doc = new XDoctype ("y:mydoc", "http://my.doc");
        strongDoctype = cache.getDoctype (mockCollection, doc.getFullQualifiedName());
        assertEquals ("http://my.doc", strongDoctype.getNamespaceUri ());
        assertEquals ("x:mydoc", strongDoctype.getPrefixedName ());
        
        doc = new XDoctype ("mydoc", "http://my.doc");
        strongDoctype = cache.getDoctype (mockCollection, doc.getFullQualifiedName());
        assertEquals ("http://my.doc", strongDoctype.getNamespaceUri ());
        assertEquals ("x:mydoc", strongDoctype.getPrefixedName ());
        
        doc = new XDoctype ("z:mydoc", null, false);
        cache.addDoctype (mockCollection, doc);
        strongDoctype = cache.getDoctype (mockCollection, doc.getFullQualifiedName());
        assertEquals (null, strongDoctype.getNamespaceUri ());
        assertEquals ("z:mydoc", strongDoctype.getPrefixedName ());
        
        doc = new XDoctype ("mydoc", "http://your.doc");
        strongDoctype = cache.getDoctype (mockCollection, doc.getFullQualifiedName());
        assertTrue (strongDoctype == null);
    }
    
    public void testDelete () {
        
        XDoctypeCache cache = XDoctypeCache.getSingleInstance();
        
        Set set = new HashSet ();
        set.add (new XDoctype ("x:mydoc", "http://my.doc", false));
        set.add (new XDoctype ("y:mydoc1", "http://my.doc", false));
        set.add (new XDoctype ("z:mydoc3", "http://my.doc", false));
        
        cache.addDoctype (mockCollection, set);
        
        XDoctype weakDoctype = new XDoctype ("hugo", null);
        cache.addWeakDoctype (mockCollection, weakDoctype);
        
        XDoctype strongDoctype = cache.getDoctype (mockCollection, "mydoc@http://my.doc");
        assertEquals ("x:mydoc", strongDoctype.getPrefixedName ());
        
        
        cache.deleteDoctype (mockCollection, set);
        strongDoctype = cache.getDoctype (mockCollection, "mydoc@http://my.doc");
        assertTrue (strongDoctype == null);
        
        strongDoctype = cache.getDoctype (mockCollection, "hugo");
        assertEquals ("hugo", strongDoctype.getPrefixedName ());
        
        cache.deleteDoctype (mockCollection, strongDoctype);
        strongDoctype = cache.getDoctype (mockCollection, "mydoc@http://my.doc");
        assertTrue (strongDoctype == null);
        
    }
    
    
    
    public void testPrefixGenerator () throws Exception {
        
        XDoctypeCache cache = XDoctypeCache.getSingleInstance();
        
        XDoctype weak1 = new XDoctype ("a:mydoc", "http://uri1");
        try {
            cache.addDoctype (mockCollection, weak1);
            fail ("IllegalStateException should have been thrown");
            
        }
        catch (IllegalStateException e) {}
        
        cache.addWeakDoctype (mockCollection, weak1);
        
        XDoctype weak2 = new XDoctype ("a:mydoc", "http://uri2");
        cache.addWeakDoctype (mockCollection, weak2);
        
        
        XDoctype strong1 = cache.getDoctype(mockCollection, weak1.getFullQualifiedName());
        XDoctype strong2 = cache.getDoctype(mockCollection, weak2.getFullQualifiedName());
        
        assertTrue (strong1.getPrefixedName() != strong2.getPrefixedName());
    }
}

