/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/store/org/apache/slide/store/tamino/jdomobjects/XNodePropertyTest.java,v 1.1 2004/03/25 16:18:14 juergen Exp $
 * $Revision: 1.1 $
 * $Date: 2004/03/25 16:18:14 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.jdomobjects;

import java.io.IOException;
import java.text.ParseException;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;
import org.apache.slide.common.Domain;
import org.apache.slide.content.NodeProperty;
import org.apache.slide.store.tamino.common.XGlobals;
import org.apache.slide.util.JDom;
import org.jdom.Element;
import org.jdom.JDOMException;

public class XNodePropertyTest extends TestCase {
    public XNodePropertyTest(String name) {
        super( name );
    }

    public void setUp () throws Exception {
        super.setUp();
        Domain.setInitialized(true);
    }


    public void testName() throws JDOMException {
        NodeProperty slide;

        slide = check("<nodeProperty type='foo' name='someName' datatype='string' protected='true'/>");
        assertEquals("someName", slide.getName());
    }

    public void testNamespace() throws JDOMException {
        NodeProperty slide;

        slide = check("<nodeProperty type='foo' namespace='someNamespace' datatype='string' protected='true'/>");
        assertEquals("someNamespace", slide.getNamespace());
    }

    public void testType() throws JDOMException {
        NodeProperty slide;

        slide = check("<nodeProperty type='someType' datatype='string' protected='true'/>");
        assertEquals("someType", slide.getType());
    }

    public void testProtected() throws JDOMException {
        NodeProperty slide;

        slide = check("<nodeProperty type='someType' datatype='string' protected='true'/>");
        assertTrue(slide.isProtected());
        slide = check("<nodeProperty type='someType' datatype='string' protected='false'/>");
        assertFalse(slide.isProtected());
    }


    public void testIndexedProperty () throws JDOMException, IOException {
        NodeProperty slide;

        String s = "<xdavContentId datatype='string' type='' protected='true' xmlns='http://namespaces.softwareag.com/tamino/webdav'>1@schubidu<shadow xmlns=''>1@schubidu</shadow></xdavContentId>";

        slide = checkIndexed (s, XGlobals.TSD3_SCHEMA_VERSION);
        assertEquals ("xdavContentId", slide.getName());

        slide = checkIndexed (s, XGlobals.TSD4_SCHEMA_VERSION);
        assertEquals ("xdavContentId", slide.getName());
    }

    public void testStringBody() throws JDOMException {
        NodeProperty slide;

        slide = check(
            "<nodeProperty type='foo' datatype='string' protected='true'>text</nodeProperty>\n"
        );
        assertEquals("text", slide.getValue());
    }

    public void testIntBody() throws JDOMException {
        NodeProperty slide;

        slide = check(
            "<nodeProperty type='foo' datatype='int' protected='true'>42</nodeProperty>\n"
        );
        assertEquals(new Integer(42), slide.getValue());
    }

    public void testLongBody() throws JDOMException {
        NodeProperty slide;

        slide = check(
            "<nodeProperty type='foo' datatype='long' protected='true'>7</nodeProperty>\n"
        );
        assertEquals(new Long(7), slide.getValue());
    }

    public void testBooleanBody() throws JDOMException {
        NodeProperty slide;

        slide = check(
            "<nodeProperty type='foo' datatype='boolean' protected='true'>false</nodeProperty>\n"
        );
        assertEquals(new Boolean(false), slide.getValue());
    }

    public void testDateBody() throws JDOMException, ParseException {
        final String date = "20030506174911";
        NodeProperty slide;

        slide = check(
            "<nodeProperty type='foo' datatype='date' protected='true'>" + date + "</nodeProperty>\n"
        );
        assertEquals(XJdom.DATE_FORMATTER.parse(date), slide.getValue());
    }

    public void testUnkownDatatype() throws JDOMException, ParseException {
        try {
            check(
                "<nodeProperty type='foo' datatype='nosuchtype' protected='true'></nodeProperty>\n"
            );
            fail();
        } catch (IllegalStateException e) {
            // ok
        }
    }

    public NodeProperty check(String str) throws JDOMException {
        Element xml;
        NodeProperty slide;

        try {
            xml = JDom.forString(str).getRootElement();
        } catch (IOException e) {
            fail(e.getMessage());
            return null; // dummy
        }
        // do toSlide/getElement roundtrip to test getElement as well
        slide = XNodeProperty.toSlide(xml);
        xml = XNodeProperty.getElement (slide, XGlobals.TSD3_SCHEMA_VERSION);
        return XNodeProperty.toSlide(xml);
    }

    public NodeProperty checkIndexed (String str, String schemaLanguage) throws JDOMException, IOException {
        Element xml;
        NodeProperty slide;

        xml = JDom.forString(str).getRootElement();
        // do toSlide/getElement roundtrip to test getElement as well
        slide = XNodeProperty.toSlide(xml);
        xml = XNodeProperty.getElement (slide, schemaLanguage);
        return XNodeProperty.toSlide(xml);
    }

    /**
     ** Start precisely this test suite with the commandline interface.
     **
     ** @param args   commandline options (not used)
     **/
    public static void main( String[] args ) {
        TestRunner.run( suite() );
    }

    public static Test suite() {
        return new TestSuite(XNodePropertyTest.class);
    }
}
