/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/store/org/apache/slide/store/tamino/jdomobjects/XObjectNodeTest.java,v 1.1 2004/03/25 16:18:14 juergen Exp $
 * $Revision: 1.1 $
 * $Date: 2004/03/25 16:18:14 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.jdomobjects;

import java.io.IOException;
import java.util.Enumeration;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;
import org.apache.slide.structure.ActionNode;
import org.apache.slide.structure.LinkNode;
import org.apache.slide.structure.ObjectNode;
import org.apache.slide.structure.SubjectNode;
import org.apache.slide.util.JDom;
import org.apache.slide.util.XAssertionFailed;
import org.jdom.Element;
import org.jdom.JDOMException;

public class XObjectNodeTest extends TestCase {
    public XObjectNodeTest(String name) {
        super( name );
    }

    public void testKindSubject() throws JDOMException {
        checkKind("subject", SubjectNode.class);
    }

    public void testKindAction() throws JDOMException {
        checkUnkownKind("action", ActionNode.class);
    }

    public void testKindLink() throws JDOMException {
        checkUnkownKind("link", LinkNode.class);
    }

    public void testKindUnkown() throws JDOMException {
        checkUnkownKind("noSuchKind", null);
    }

    public void checkUnkownKind(String kind, Class clazz) throws JDOMException {
        try {
            checkKind(kind, clazz);
            fail();
        } catch (XAssertionFailed e) {
            assertEquals(kind, e.getMessage());
        }
    }



    public void testChildren() throws JDOMException {
        ObjectNode slide;
        Enumeration en;

        slide = check("<objectNode kind='subject'>\n" +
                          "  <parents/>\n" +
                          "  <children>\n" +
                          "    <childNode name='1' uriRef='first'/>\n" +
                          "    <childNode name='2' uriRef='second'/>\n" +
                          "  </children>\n" +
                          "  <links/>\n" +
                          "</objectNode>"
                     );
        en = slide.enumerateBindings();
        checkBinding("1", "first", en.nextElement());
        checkBinding("2", "second", en.nextElement());
        assertFalse(en.hasMoreElements());
    }

    public void testParents() throws JDOMException {
        ObjectNode slide;
        Enumeration en;

        slide = check("<objectNode kind='subject'>\n" +
                          "  <parents>\n" +
                          "    <parentNode name='2' uriRef='second'/>\n" +
                          "    <parentNode name='1' uriRef='first'/>\n" +
                          "  </parents>\n" +
                          "  <children/>\n" +
                          "  <links/>\n" +
                          "</objectNode>"
                     );
        en = slide.enumerateParentBindings();
        checkBinding("2", "second", en.nextElement());
        checkBinding("1", "first", en.nextElement());
        assertFalse(en.hasMoreElements());
    }


    public void testLinks() throws JDOMException {
        ObjectNode slide;
        Enumeration en;

        slide = check(
            "<objectNode kind='subject'>\n" +
                "  <parents/>\n" +
                "  <children/>\n" +
                "  <links>\n" +
                "    <linkNode uriRef='a'/>\n" +
                "    <linkNode uriRef='b'/>\n" +
                "  </links>\n" +
                "</objectNode>"
        );
        en = slide.enumerateLinks();
        assertEquals("a", en.nextElement());
        assertEquals("b", en.nextElement());
        assertFalse(en.hasMoreElements());
    }

    //--

    private void checkKind(String kindStr, Class kindClass) throws JDOMException {
        ObjectNode slide;

        slide = check(
            "<objectNode kind='" + kindStr + "'>\n" +
                "  <parents/>\n" +
                "  <children/>\n" +
                "  <links/>\n" +
                "</objectNode>"
        );
        assertTrue(kindClass.isInstance(slide));
    }

    private void checkBinding(String name, String uriRef, Object obj) {
        ObjectNode.Binding binding;

        assertTrue(obj instanceof ObjectNode.Binding);
        binding = (ObjectNode.Binding) obj;
        assertEquals(name, binding.getName());
        assertEquals(uriRef, binding.getUuri());
    }

    private ObjectNode check(String str) throws JDOMException {
        final String URL = "";
        Element xml;
        ObjectNode slide;

        try {
            xml = JDom.forString(str).getRootElement();
        } catch (IOException e) {
            fail(e.getMessage());
            return null; // dummy
        }
        // do toSlide/getElement roundtrip to test getElement as well
        slide = XObjectNode.toSlide(xml, URL);
        xml = XObjectNode.toXml(slide);
        return XObjectNode.toSlide(xml, URL);
    }

    /**
     ** Start precisely this test suite with the commandline interface.
     **
     ** @param args   commandline options (not used)
     **/
    public static void main( String[] args ) {
        TestRunner.run( suite() );
    }


    public static Test suite() {
        return new TestSuite(XObjectNodeTest.class);
    }
}
