/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/urm/org/apache/slide/urm/common/TestURMPrincipal.java,v 1.3 2004/07/30 06:52:20 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:20 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.urm.common;

import java.io.FileInputStream;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.apache.slide.urm.URMException;
import org.apache.slide.urm.accesscontroler.URMAclAdministrator;
import org.apache.slide.urm.authenticator.URMAdministrator;
import org.apache.slide.urm.authenticator.URMAuthenticator;
import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.authenticator.rolemanager.URMRole;
import org.apache.slide.urm.authenticator.userdb.URMGroup;
import org.apache.slide.urm.authenticator.userdb.URMUser;
import org.apache.slide.urm.common.URMConfigurator;
import org.apache.slide.urm.common.URMConstants;
import org.apache.slide.urm.common.URMPrincipal;
import org.apache.slide.urm.common.impl.URMConfiguratorXML;

/**
 * @author eckehard.hermann@softwareag.com
 * @author dieter.kessler@softwareag.com
 * @author zsolt.sasvarie@softwareag.com
 */
public class TestURMPrincipal extends TestCase {

    private static String TEST_USER1 = "URM_junit_testp_usr1";
    private static String TEST_GROUP1 = "URM_junit_testp_grp1";
    private static String TEST_GROUP2 = "URM_junit_testp_grp2";
    private static String TEST_ROLE1 = "URM_junit_testp_role1";
    private static String TEST_ROLE2 = "URM_junit_testp_role2";
    private static String TEST_ROLE3 = "URM_junit_testp_role3";
    private static String TEST_ROLE4 = "URM_junit_testp_role4";
    private static String TEST_ACTION1 = "URM_junit_testp_action1";
    private static String TEST_RESOURCE_ID1 = "URM_junit_testp_acl_resource_id1";

    private static URMAuthenticator     mAuthenticator = null;
    private static URMPrincipal         mPrincipal = null;
    private static URMAdministrator     mAdministrator = null;
    private static URMAclAdministrator  mAclAdministrator = null;
    private static URMConfigurator      mTestConfigurator = null;
    
    private static String mDomain = null;
    
    private URMUser     mUser1  = null;
    private URMGroup    mGroup1 = null;
    private URMGroup    mGroup2 = null;
    private URMRole     mRole1  = null;
    private URMRole     mRole2  = null;
    private URMRole     mRole3  = null;

    public TestURMPrincipal(String name) throws Exception {
        super(name);
        initUrmTest();
    }

    private static synchronized void initUrmTest() throws Exception {
        if (mTestConfigurator != null)
            return;
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationTamino.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapDK.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapSunOne.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapActiveDirectory.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationOS.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapXtlin1.xml");

        FileInputStream fis = new FileInputStream("config/ConfigurationLdapSunOne.xml");
        //FileInputStream fis = new FileInputStream("config/ConfigurationLdapOpenLdap.xml");
        //FileInputStream fis = new FileInputStream("config/ConfigurationOS.xml");

        URMConfigurator conf = new URMConfiguratorXML(fis);
        fis.close();
        mTestConfigurator = (URMConfigurator)(conf.getSubConfigurators().get("JUnitTest"));
        conf = (URMConfigurator)(conf.getSubConfigurators().get("Authenticator"));
        mAuthenticator = URMAuthenticator.newInstance(conf);
        mDomain = mTestConfigurator.getProperties().getProperty("TestDomain");
        if (mDomain.length() <= 0)
            mDomain = null;
        mPrincipal = mAuthenticator.authenticateUser(
                mTestConfigurator.getProperties().getProperty("TestAdminUser"),
                mTestConfigurator.getProperties().getProperty("TestAdminPw").toCharArray(), mDomain);
        mPrincipal.setActiveRole(URMConstants.URM_ADMIN_ROLE);
        mAdministrator = mPrincipal.getURMAdministrator();
        if (mAdministrator == null)
            throw new Exception ("CANNOT CREATE URM ADMINISTRATOR, CHECK THE TEST ADMIN USER PARAMS!");
        mAclAdministrator = mPrincipal.getURMAclAdministrator();
        assertNotNull("null URMAclAdministrator was received!!!", mAclAdministrator);
    
        mDomain = mTestConfigurator.getProperties().getProperty("TestDomain");
        if (mDomain.length() <= 0)
            mDomain = null;
    }

    public static Test suite() {
        return new TestSuite(TestURMPrincipal.class);
    }

    private boolean containsValues(Set set, String val1, String val2) {
        /*if (set == null)
            return false;*/
    
        Iterator iter = set.iterator();
        boolean role1exists = false, role2exists = false;
        while (/*iter != null && */iter.hasNext()) {
            Object obj = iter.next();
            if (((URMSubject)obj).getName().equals(val1))
                role1exists = true;
            else if (((URMSubject)obj).getName().equals(val2))
                role2exists = true;
        }
        return role1exists && role2exists;
    }

    private boolean containsValue(Set set, String val) {
        /*if (set == null)
            return false;*/
    
        Iterator iter = set.iterator();
        while (/*iter != null && */iter.hasNext())
            if (((URMSubject)iter.next()).getName().equals(val))
                return true;
        return false;
    }

    private String getMissingKeys(Enumeration keys, Properties props) {
        String missingkeys = ",";
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String gval = props.getProperty(key);
            if (gval == null)
                missingkeys += ("," + key);
        }
        return missingkeys.substring(2);
    }

    public void setUp() throws Exception {
        mAdministrator.createUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        mUser1  = mAdministrator.getUser(TEST_USER1, mDomain);
        mAdministrator.createGroup(TEST_GROUP1, mDomain);
        mAdministrator.createGroup(TEST_GROUP2, mDomain);
        mGroup1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        mGroup2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        mAdministrator.createRole(TEST_ROLE1);
        mAdministrator.createRole(TEST_ROLE2);
        mAdministrator.createRole(TEST_ROLE3);
        mAdministrator.createRole(TEST_ROLE4);
        mRole1  = mAdministrator.getRole(TEST_ROLE1);
        mRole2  = mAdministrator.getRole(TEST_ROLE2);
        mRole3  = mAdministrator.getRole(TEST_ROLE3);
    }

    public void tearDown() throws Exception {
        mAdministrator.deleteUser(TEST_USER1, mDomain);
        mAdministrator.deleteGroup(TEST_GROUP1, mDomain);
        mAdministrator.deleteGroup(TEST_GROUP2, mDomain);
        mAdministrator.deleteRole(TEST_ROLE1);
        mAdministrator.deleteRole(TEST_ROLE2);
        mAdministrator.deleteRole(TEST_ROLE3);
        mAdministrator.deleteRole(TEST_ROLE4);
    }

    public void test0_() throws Exception {
    }

    /** Implemented test cases:
     *
     * URMPrincipal_0001, URMPrincipal_0008, URMPrincipal_0011
     */
    public void test1_nonSelectableGuestRole() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting the non-selectable GUEST role did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        // add to GUEST role
        mUser1.linkToRole(URMConstants.URM_GUEST_ROLE);
        assertTrue("User have more or less than one role mapping (just one was added).", mUser1.getRoleMapping().size() == 1);
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting the non-selectable "+TEST_ROLE1+" role did not throw exception.", true);
        } catch (URMException e) { }
        // set GUEST active role
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
        } catch (URMException e) {
            assertFalse("Setting the selectable GUEST role threw an exception: "+e.getMessage(), true);
        }
        assertTrue("Setting the selectable GUEST role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_GUEST_ROLE));
        try { // set again the same
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting active role (after it was set once) did not throw exception.", true);
        } catch (URMException e) { }
        // copy principal without active role and set the active role twice
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting active role to a non-selectabe role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
        // copy principal with active role and set the active role
        princ = princ.copyPrincipal(URMConstants.URM_GUEST_ROLE);
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting active role (after copy principal with active role parameter) did not throw exception.", true);
        } catch (URMException e) { }
        // remove from GUEST role
        mUser1.deleteRoleLink(URMConstants.URM_GUEST_ROLE);
        try {
            princ.copyPrincipal(URMConstants.URM_GUEST_ROLE);
            assertFalse("Copy principal with the non-selectable GUEST role did not throw exception.", true);
        } catch (URMException e) { }
    }
    
    /** Implemented test cases:
     *
     * URMPrincipal_0002, URMPrincipal_0007, URMPrincipal_0012, URMPrincipal_0014
     */
    public void test2_nonSelectableRootRole() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting the non-selectable GUEST role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role did not throw exception.", true);
        } catch (URMException e) { }
        // add to ROOT role
        mUser1.linkToRole(URMConstants.URM_ROOT_ROLE);
        assertTrue("User have more or less than one role mapping (just one was added).", mUser1.getRoleMapping().size() == 1);
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting the non-selectable GUEST role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting the non-selectable "+TEST_ROLE1+" role did not throw exception.", true);
        } catch (URMException e) { }
        // set ROOT active role
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
        } catch (URMException e) {
            assertFalse("Setting the selectable ROOT role threw an exception: "+e.getMessage(), true);
        }
        assertTrue("Setting the selectable ROOT role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ROOT_ROLE));
        try { // set again the same
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting active role (after it was set once) did not throw exception.", true);
        } catch (URMException e) { }
        // copy principal without active role and set the active role twice
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting active role to a non-selectabe role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
        // copy principal with active role and set the active role
        princ = princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting active role (after copy principal with active role parameter) did not throw exception.", true);
        } catch (URMException e) { }
        // remove from ROOT role
        mUser1.deleteRoleLink(URMConstants.URM_ROOT_ROLE);
        try {
            princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
            assertFalse("Copy principal with the non-selectable ROOT role did not throw exception.", true);
        } catch (URMException e) { }
    }
    
    /** Implemented test cases:
     *
     * URMPrincipal_0003, URMPrincipal_0009, URMPrincipal_0010, URMPrincipal_0013
     */
    public void test3_nonSelectableUserDefinedRole() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        try {
            princ.setActiveRole("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting the non-selectable GUEST role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role (1.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting the non-selectable "+TEST_ROLE1+" role did not throw exception.", true);
        } catch (URMException e) { }
        // add to TEST_ROLE1 role
        mUser1.linkToRole(TEST_ROLE1);
        assertTrue("User have more or less than one role mapping (just one was added).", mUser1.getRoleMapping().size() == 1);
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting the non-selectable ADMIN role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting the non-selectable GUEST role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        // set TEST_ROLE1 active role
        try {
            princ.setActiveRole(TEST_ROLE1);
        } catch (URMException e) {
            assertFalse("Setting the selectable TEST_ROLE1 role threw an exception: "+e.getMessage(), true);
        }
        assertTrue("Setting the selectable TEST_ROLE1 role was unsuccessful.", princ.getActiveRole().equals(TEST_ROLE1));
        try { // set again the same
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting active role (after it was set once) did not throw exception.", true);
        } catch (URMException e) { }
        // copy principal without active role and set the active role twice
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting active role to a non-selectabe role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        princ.setActiveRole(TEST_ROLE1);
        // copy principal with active role and set the active role
        princ = princ.copyPrincipal(TEST_ROLE1);
        try {
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting active role (after copy principal with active role parameter) did not throw exception.", true);
        } catch (URMException e) { }
        // remove from TEST_ROLE1 role
        mUser1.deleteRoleLink(TEST_ROLE1);
        try {
            princ.copyPrincipal(TEST_ROLE1);
            assertFalse("Copy principal with the non-selectable TEST_ROLE1 role did not throw exception.", true);
        } catch (URMException e) { }
    }
    
    /** Implemented test cases:
     *
     * URMPrincipal_0015, URMPrincipal_0016
     */
    public void test4_guestPrincipal() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser();
        try {
            princ.copyPrincipal(TEST_ROLE1);
            assertFalse("Copy principal with the non-selectable TEST_ROLE1 role did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.copyPrincipal(null);
            assertFalse("Copy Guest principal did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_GUEST_ROLE);
            assertFalse("Setting active role to the non-selectabe GUEST role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
            assertFalse("Copy principal with the non-selectable ROOT role did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting active role to the non-selectabe ADMIN role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
    }
    
    /** Implemented test cases:
     *
     * URMPrincipal_0017, URMPrincipal_0018, URMPrincipal_0019,
     * URMPrincipal_0020, URMPrincipal_0021, URMPrincipal_0022,
     * URMPrincipal_0035, URMPrincipal_0036, URMPrincipal_0037
     */
    public void test5_copyPrincipal() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        assertNull("Get active role does not return null (it was not set).", princ.getActiveRole());
        try {
            princ.setActiveRole(null);
            assertFalse("Setting 'null' role did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.copyPrincipal("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (1.) did not throw exception.", true);
        } catch (URMException e) { }

        mUser1.linkToRole(TEST_ROLE1);
        princ = princ.copyPrincipal(TEST_ROLE1);
        assertTrue("Copy Principal with the selectable TEST_ROLE1 role was unsuccessful.", princ.getActiveRole().equals(TEST_ROLE1));
        mUser1.linkToRole(URMConstants.URM_ROOT_ROLE);
        princ = princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
        assertTrue("Copy Principal with the selectable ROOT role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ROOT_ROLE));
        mUser1.linkToRole(URMConstants.URM_ADMIN_ROLE);
        princ = princ.copyPrincipal(URMConstants.URM_ADMIN_ROLE);
        assertTrue("Copy Principal with the selectable ADMIN role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ADMIN_ROLE));

        try {
            princ.copyPrincipal("URM_junit_non_existing_role");
            assertFalse("Copy Principal with the non-existing role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        
        mUser1.deleteRoleLink(TEST_ROLE1);
        try {
            princ.copyPrincipal(TEST_ROLE1);
            assertFalse("Copy principal with the non-selectable TEST_ROLE1 role did not throw exception.", true);
        } catch (URMException e) { }
        mUser1.deleteRoleLink(URMConstants.URM_ADMIN_ROLE);
        try {
            princ.copyPrincipal(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Copy Principal with the non-selectabe ADMIN role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        princ = princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
        assertTrue("Copy Principal with the selectable ROOT role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ROOT_ROLE));
        mUser1.deleteRoleLink(URMConstants.URM_ROOT_ROLE);
        try {
            princ.copyPrincipal(URMConstants.URM_ROOT_ROLE);
            assertFalse("Copy principal with the non-selectable ROOT role did not throw exception.", true);
        } catch (URMException e) { }
        try {
            princ.copyPrincipal("URM_junit_non_existing_role");
            assertFalse("Copy Principal with the non-existing role (3.) did not throw exception.", true);
        } catch (URMException e) { }
        
        // and all once again with setActiveRole()
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (1.) did not throw exception.", true);
        } catch (URMException e) { }

        mUser1.linkToRole(TEST_ROLE1);
        princ = princ.copyPrincipal(null);
        princ.setActiveRole(TEST_ROLE1);
        assertTrue("Setting the selectable TEST_ROLE1 role was unsuccessful.", princ.getActiveRole().equals(TEST_ROLE1));
        mUser1.linkToRole(URMConstants.URM_ROOT_ROLE);
        princ = princ.copyPrincipal(null);
        princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
        assertTrue("Setting the selectable ROOT role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ROOT_ROLE));
        mUser1.linkToRole(URMConstants.URM_ADMIN_ROLE);
        princ = princ.copyPrincipal(null);
        princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
        assertTrue("Setting the selectable ADMIN role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ADMIN_ROLE));

        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (2.) did not throw exception.", true);
        } catch (URMException e) { }
        
        mUser1.deleteRoleLink(TEST_ROLE1);
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(TEST_ROLE1);
            assertFalse("Setting the non-selectable TEST_ROLE1 role did not throw exception.", true);
        } catch (URMException e) { }
        mUser1.deleteRoleLink(URMConstants.URM_ADMIN_ROLE);
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(URMConstants.URM_ADMIN_ROLE);
            assertFalse("Setting active role to the non-selectabe ADMIN role (after copy principal) did not throw exception.", true);
        } catch (URMException e) { }
        princ = princ.copyPrincipal(null);
        princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
        assertTrue("Setting the selectable ROOT role was unsuccessful.", princ.getActiveRole().equals(URMConstants.URM_ROOT_ROLE));
        mUser1.deleteRoleLink(URMConstants.URM_ROOT_ROLE);
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole(URMConstants.URM_ROOT_ROLE);
            assertFalse("Setting the non-selectable ROOT role did not throw exception.", true);
        } catch (URMException e) { }
        princ = princ.copyPrincipal(null);
        try {
            princ.setActiveRole("URM_junit_non_existing_role");
            assertFalse("Setting the non-existing role (3.) did not throw exception.", true);
        } catch (URMException e) { }
    }
    
    /** Implemented test cases:
     *
     * URMPrincipal_0029, URMPrincipal_0053(URMUser_0023), URMPrincipal_0054(URMUser_0024),
     * URMPrincipal_0055(URMUser_0027), URMPrincipal_0056(URMUser_0028), URMPrincipal_0057(URMUser_0029)
     */
    public void test6_possibleRoles() throws Exception {
        URMPrincipal princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        assertFalse("User has possible roles (nothing was added).", princ.getPossibleRoles().size() > 0);
        mUser1.linkToRole(TEST_ROLE1);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (one was added).", princ.getPossibleRoles().size() == 1);
        mUser1.deleteRoleLink(TEST_ROLE1);
        assertFalse("User has possible roles (everything was removed 1.).", princ.getPossibleRoles().size() > 0);

        mGroup1.addUserToGroup(TEST_USER1, mDomain);
        assertFalse("User has possible roles (everything was removed 2.).", princ.getPossibleRoles().size() > 0);
        mGroup1.linkToRole(TEST_ROLE1);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (one was added to a container group).", princ.getPossibleRoles().size() == 1);
        mUser1.linkToRole(TEST_ROLE2);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 1.).", princ.getPossibleRoles().size() == 2);

        mGroup2.addUserToGroup(TEST_USER1, mDomain);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 2.).", princ.getPossibleRoles().size() == 2);
        mGroup2.linkToRole(TEST_ROLE2);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 3.).", princ.getPossibleRoles().size() == 2);
        
        mUser1.deleteRoleLink(TEST_ROLE2);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 4.).", princ.getPossibleRoles().size() == 2);
        try {
            mUser1.deleteRoleLink(TEST_ROLE2);
        } catch (URMException e) { }
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 5.).", princ.getPossibleRoles().size() == 2);

        mRole1.addAncestor(TEST_ROLE2);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 6.).", princ.getPossibleRoles().size() == 2);
        mGroup2.deleteRoleLink(TEST_ROLE2);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (two was added 7.).", princ.getPossibleRoles().size() == 2);
        mRole2.addAncestor(TEST_ROLE3);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (three was added 1.).", princ.getPossibleRoles().size() == 3);
        mRole1.addAncestor(TEST_ROLE4);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (four was added 1.).", princ.getPossibleRoles().size() == 4);
        mRole1.deleteAncestor(TEST_ROLE4);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (three was added 2.).", princ.getPossibleRoles().size() == 3);
        mRole2.addAncestor(TEST_ROLE4);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (four was added 2.).", princ.getPossibleRoles().size() == 4);

        mGroup1.deleteRoleLink(TEST_ROLE1);
        assertFalse("User has possible roles (everything was removed 3.).", princ.getPossibleRoles().size() > 0);
        mUser1.linkToRole(TEST_ROLE1);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (four was added 2.).", princ.getPossibleRoles().size() == 4);
        mRole2.deleteAncestor(TEST_ROLE4);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (three was added 3.).", princ.getPossibleRoles().size() == 3);
        mRole3.addAncestor(TEST_ROLE4);
        assertTrue("User has "+princ.getPossibleRoles().size()+" possible roles (four was added 3.).", princ.getPossibleRoles().size() == 4);
    }

    public static void main(String[] args) throws Exception {
        String[] testCaseName = { "-noloading", TestURMPrincipal.class.getName() };
        
        org.apache.log4j.PropertyConfigurator.configure("testsuite/junit/log4j.properties");
        
        junit.textui.TestRunner.main(testCaseName);
    }
}
