/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/urm/org/apache/slide/urm/test/authenticator/userdb/TestURMUserDBManager.java,v 1.3 2004/07/30 06:52:21 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:21 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.urm.test.authenticator.userdb;

import java.io.FileInputStream;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.apache.slide.urm.URMException;
import org.apache.slide.urm.URMForbiddenException;
import org.apache.slide.urm.accesscontroler.URMAce;
import org.apache.slide.urm.accesscontroler.URMAcl;
import org.apache.slide.urm.accesscontroler.URMAclAdministrator;
import org.apache.slide.urm.authenticator.URMAdministrator;
import org.apache.slide.urm.authenticator.URMAuthenticator;
import org.apache.slide.urm.authenticator.URMAuthenticatorException;
import org.apache.slide.urm.authenticator.URMSubject;
import org.apache.slide.urm.authenticator.rolemanager.URMRole;
import org.apache.slide.urm.authenticator.rolemanager.impl.URMRoleGuestImpl;
import org.apache.slide.urm.authenticator.rolemanager.impl.URMRoleImpl;
import org.apache.slide.urm.authenticator.userdb.URMGroup;
import org.apache.slide.urm.authenticator.userdb.URMUser;
import org.apache.slide.urm.authenticator.userdb.impl.tamino.URMMetadataPropsSpiTamino;
import org.apache.slide.urm.common.URMConfigurator;
import org.apache.slide.urm.common.URMConstants;
import org.apache.slide.urm.common.URMPrincipal;
import org.apache.slide.urm.common.impl.URMConfiguratorUtil;
import org.apache.slide.urm.common.impl.URMConfiguratorXML;

/**
 * @author zsa
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class TestURMUserDBManager extends TestCase
{
    private static final int TEST_METHOD_NUMBER = 28;
    
    private static String TEST_USER1 = "URM_junit_test_user1";
    private static String TEST_USER2 = "URM_junit_test_user2";
    private static String TEST_GROUP1 = "URM_junit_test_group1";
    private static String TEST_GROUP2 = "URM_junit_test_group2";
    private static String TEST_GROUP3 = "URM_junit_test_group3";
    private static String TEST_ACTION1 = "URM_junit_test_action1";
    private static String TEST_ACTION2 = "URM_junit_test_action2";
    private static String TEST_RESOURCE_ID1 = "URM_junit_test_acl_resource_id1";
    private static String TEST_RESOURCE_ID2 = "URM_junit_test_acl_resource_id2";

    private static URMAuthenticator mAuthenticator = null;
    private static URMPrincipal mPrincipal = null;
    private static URMAdministrator mAdministrator = null;
    private static URMAclAdministrator mAclAdministrator = null;
    private static URMConfigurator mTestConfigurator = null;
    private static String mDomain = null;
    private static int mSetUpTearDownCounter = 0;
    private static int mSetUpTearDownCounterCheck = 0;
    private static LinkedList mCreatedUserNames    = new LinkedList();
    private static LinkedList mCreatedUserDomains  = new LinkedList();
    private static LinkedList mCreatedGroupNames   = new LinkedList();
    private static LinkedList mCreatedGroupDomains = new LinkedList();
    private static LinkedList mCreatedRoleNames    = new LinkedList();
    private static LinkedList mCreatedAclIds       = new LinkedList();
    
    public TestURMUserDBManager(String name) throws Exception {
        super(name);
        initUrmTest();
    }
    
    private static synchronized void initUrmTest() throws Exception {
        if (mTestConfigurator != null)
            return;
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationTamino.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapDK.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapSunOne.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapActiveDirectory.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationOS.xml");
        //FileInputStream fis = new FileInputStream("testsuite/junit/ConfigurationLdapXtlin1.xml");

        FileInputStream fis = new FileInputStream("config/ConfigurationLdapSunOne.xml");
        //FileInputStream fis = new FileInputStream("config/ConfigurationLdapOpenLdap.xml");
        //FileInputStream fis = new FileInputStream("config/ConfigurationOS.xml");

        URMConfigurator conf = new URMConfiguratorXML(fis);
        fis.close();
        mTestConfigurator = (URMConfigurator)(conf.getSubConfigurators().get("JUnitTest"));
        conf = (URMConfigurator)(conf.getSubConfigurators().get("Authenticator"));
        mAuthenticator = URMAuthenticator.newInstance(conf);
        mDomain = mTestConfigurator.getProperties().getProperty("TestDomain");
        if (mDomain.length() <= 0)
            mDomain = null;
        mPrincipal = mAuthenticator.authenticateUser(
                mTestConfigurator.getProperties().getProperty("TestAdminUser"),
                mTestConfigurator.getProperties().getProperty("TestAdminPw").toCharArray(), mDomain);
        mPrincipal.setActiveRole(URMConstants.URM_ADMIN_ROLE);
        mAdministrator = mPrincipal.getURMAdministrator();
        if (mAdministrator == null)
            throw new Exception ("CANNOT CREATE URM ADMINISTRATOR, CHECK THE TEST ADMIN USER PARAMS!");
        mAclAdministrator = mPrincipal.getURMAclAdministrator();
        assertNotNull("null URMAclAdministrator was received!!!", mAclAdministrator);
    }
    
    public static Test suite() {
        return new TestSuite(TestURMUserDBManager.class);
    }
    
    private boolean containsValues(Set set, String val1, String val2) {
        /*if (set == null)
            return false;*/
        
        Iterator iter = set.iterator();
        boolean role1exists = false, role2exists = false;
        while (/*iter != null && */iter.hasNext()) {
            Object obj = iter.next();
            if (((URMSubject)obj).getName().equals(val1))
                role1exists = true;
            else if (((URMSubject)obj).getName().equals(val2))
                role2exists = true;
        }
        return role1exists && role2exists;
    }
    
    private boolean containsValue(Set set, String val) {
        /*if (set == null)
            return false;*/
        
        Iterator iter = set.iterator();
        while (/*iter != null && */iter.hasNext())
            if (((URMSubject)iter.next()).getName().equals(val))
                return true;
        return false;
    }
    
    private String getMissingKeys(Enumeration keys, Properties props) {
        String missingkeys = ",";
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String gval = props.getProperty(key);
            if (gval == null)
                missingkeys += ("," + key);
        }
        return missingkeys.substring(2);
    }
    
    public void setUp() throws Exception {
        /*
        URMGroup grp = mAdministrator.getGroup("group1", null);
        Set members = grp.getGroupMembers();
        mAdministrator = null;
        */
        synchronized (mAdministrator) {
            assertFalse("Please, increment the 'TEST_METHOD_NUMBER' variable.",
                            mSetUpTearDownCounterCheck >= TEST_METHOD_NUMBER);
            if (mSetUpTearDownCounterCheck++ > 0) {
                return;
            }
            //System.out.println("Setup met zero counter.");
            mSetUpTearDownCounter = TEST_METHOD_NUMBER;
            String err;
            // create users
            try { err = null;
                mAdministrator.createUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_USER1+"' back("+err+").",
                            mAdministrator.getUser(TEST_USER1, mDomain));
            URMUser usr = mAdministrator.getUser(TEST_USER1, mDomain);
            if (!URMRoleImpl.contains(usr.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
                usr.linkToRole(URMConstants.URM_GUEST_ROLE);
            try { err = null;
                mAdministrator.createUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_USER2+"' back("+err+").",
                            mAdministrator.getUser(TEST_USER2, mDomain));
            // create groups
            try { err = null;
                mAdministrator.createGroup(TEST_GROUP1, mDomain);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_GROUP1+"' back("+err+").",
                            mAdministrator.getGroup(TEST_GROUP1, mDomain));
            try { err = null;
                mAdministrator.createGroup(TEST_GROUP2, mDomain);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_GROUP2+"' back("+err+").",
                            mAdministrator.getGroup(TEST_GROUP2, mDomain));
            try { err = null;
                mAdministrator.createGroup(TEST_GROUP3, mDomain);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_GROUP3+"' back("+err+").",
                            mAdministrator.getGroup(TEST_GROUP3, mDomain));
            // create actions
            try { err = null;
                Object obj = null;
                try { obj = mAclAdministrator.getAction(TEST_ACTION1); } catch (URMException e) {}
                if (obj == null)
                    mAclAdministrator.createAction(TEST_ACTION1);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_ACTION1+"' back("+err+").",
                            mAclAdministrator.getAction(TEST_ACTION1));
            try { err = null;
                Object obj = null;
                try { obj = mAclAdministrator.getAction(TEST_ACTION2); } catch (URMException e) {}
                if (obj == null)
                    mAclAdministrator.createAction(TEST_ACTION2);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+TEST_ACTION2+"' back("+err+").",
                            mAclAdministrator.getAction(TEST_ACTION2));
        }
    }

    public void tearDown() throws Exception {
        synchronized (mAdministrator) {
            if (--mSetUpTearDownCounter > 0)
                return;
            //System.out.println("Tear down met zero counter.");
        }
                        
        // delete users
        mAdministrator.deleteUser(TEST_USER1, mDomain);
        assertTrue("The delete '"+TEST_USER1+"' failed (created in setUp).",
                        !mAdministrator.isUser(TEST_USER1, mDomain));
        assertTrue("The '"+TEST_USER2+"' cannot be found (created in setUp).",
                        mAdministrator.isUser(TEST_USER2, mDomain));
        mAdministrator.deleteUser(TEST_USER2, mDomain);
        assertTrue("The delete '"+TEST_USER2+"' failed (created in setUp).",
                        !mAdministrator.isUser(TEST_USER2, mDomain));
        // delete groups
        mAdministrator.deleteGroup(TEST_GROUP1, mDomain);
        assertTrue("The delete '"+TEST_GROUP1+"' failed (created in setUp).",
                        !mAdministrator.isGroup(TEST_GROUP1, mDomain));
        assertTrue("The '"+TEST_GROUP2+"' cannot be found (created in setUp).",
                        mAdministrator.isGroup(TEST_GROUP2, mDomain));
        mAdministrator.deleteGroup(TEST_GROUP2, mDomain);
        assertTrue("The delete '"+TEST_GROUP2+"' failed (created in setUp).",
                        !mAdministrator.isGroup(TEST_GROUP2, mDomain));
        mAdministrator.deleteGroup(TEST_GROUP3, mDomain);
        assertTrue("The delete '"+TEST_GROUP3+"' failed (created in setUp).",
                        !mAdministrator.isGroup(TEST_GROUP3, mDomain));

        for (int i = 0; i < mCreatedUserNames.size(); ++i) {
            String usr = (String)mCreatedUserNames.get(i), dmn = (String)mCreatedUserDomains.get(i);
            mAdministrator.deleteUser(usr, dmn);
            assertTrue("The delete '"+usr+"'("+dmn+") failed (created during the tests).",
                            !mAdministrator.isUser(usr, dmn));
        }
        for (int i = 0; i < mCreatedGroupNames.size(); ++i) {
            String usr = (String)mCreatedGroupNames.get(i), dmn = (String)mCreatedGroupDomains.get(i);
            mAdministrator.deleteGroup(usr, dmn);
            assertTrue("The delete '"+usr+"'("+dmn+") failed (created during the tests).",
                            !mAdministrator.isGroup(usr, dmn));
        }
        for (int i = 0; i < mCreatedRoleNames.size(); ++i) {
            String role = (String)mCreatedRoleNames.get(i);
            mAdministrator.deleteRole(role);
            assertTrue("The delete '"+role+"' failed (created during the tests).",
                            !mAdministrator.isRole(role));
        }
        for (int i = 0; i < mCreatedAclIds.size(); ++i) {
            String resourceid = (String)mCreatedAclIds.get(i);
            mAclAdministrator.deleteAcl(resourceid);
            assertNull("The delete ACL '"+resourceid+"' failed (created during the tests).",
                            mAclAdministrator.getAcl(resourceid));
        }
        
        // delete actions
        if (mAclAdministrator.getAction(TEST_ACTION1) != null) {
            mAclAdministrator.deleteAction(TEST_ACTION1);
        }
        assertNull("The delete '"+TEST_ACTION1+"' failed (created in setUp).",
                        mAclAdministrator.getAction(TEST_ACTION1));
        if (mAclAdministrator.getAction(TEST_ACTION2) != null) {
            mAclAdministrator.deleteAction(TEST_ACTION2);
        }
        assertNull("The delete '"+TEST_ACTION2+"' failed (created in setUp).",
                        mAclAdministrator.getAction(TEST_ACTION2));

        //mPrincipal.close();
    }

    public void test0_() throws Exception {
    }
    
    
    /** Implemented test cases:
     *
     * URMAdministrator_0002, URMAdministrator_0005, URMAdministrator_0006,
     * URMAdministrator_0007, URMAdministrator_0010, URMAdministrator_0013,
     * URMAdministrator_0022, URMAdministrator_0026, URMAdministrator_0027
     */
    public void test1_getAllUsers() throws Exception {
        assertTrue("Cannot get the added users back.",
                containsValues(mAdministrator.getAllUsers(mDomain), TEST_USER1, TEST_USER2));
        // prepare GUEST user
        URMPrincipal princ1 = null;
        String errmsg = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+errmsg+").", princ1);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        // create user
        String err, usr_name = "URM_junit_test_allus";
            // trying create with GUEST role
        try { err = null;
            princ1.getURMAdministrator().createUser(usr_name, usr_name.toCharArray(), mDomain);
            mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
        } catch (URMForbiddenException e) { err = e.getMessage(); }
        assertNotNull("Creating user with GUEST role did not throw an exception.", err);
        assertFalse("Creating user with GUEST role was successful (but threw an exception).",
                        mAdministrator.isUser(usr_name, mDomain));
            // create with ADMIN
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertTrue("Creating user '"+usr_name+"' failed ("+err+").",
                        mAdministrator.isUser(usr_name, mDomain));
        mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
            // delete user immediately ...
        try {
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain); }
        assertFalse("The delete '"+usr_name+"' failed ("+err+").",
                        mAdministrator.isUser(usr_name, mDomain));
            // ... and create user again immidiately
        mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+usr_name+"' (2.) back("+err+").",
                        mAdministrator.getUser(usr_name, mDomain));
        mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Creating a just created user did not throw an exception.", err);
        
        assertTrue("Cannot get the added users ("+TEST_USER1+" and "+TEST_USER2+") back (1.).",
                containsValues(mAdministrator.getAllUsers(mDomain), TEST_USER1, TEST_USER2));
        assertTrue("Cannot get the added user '"+usr_name+"' back.",
                containsValue(mAdministrator.getAllUsers(mDomain), usr_name));
        // delete user
            // trying delete with GUEST role
        try { err = null;
            princ1.getURMAdministrator().deleteUser(usr_name, mDomain);
            mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
        } catch (URMForbiddenException e) { err = e.getMessage(); }
        assertNotNull("Deleting user with GUEST role did not throw an exception.", err);
        assertTrue("Deleting user with GUEST role was successful (but threw an exception).",
                        mAdministrator.isUser(usr_name, mDomain));
            // delete with ADMIN
        try {
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertFalse("The delete '"+usr_name+"' failed ("+err+").",
                        mAdministrator.isUser(usr_name, mDomain));
        mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
        try { err = null;
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Deleting a just deleted user did not throw an exception.", err);
        
        assertNull("The removed user '"+usr_name+"' still exists.",
                        mAdministrator.getUser(usr_name, mDomain));
        assertTrue("Cannot get the added users ("+TEST_USER1+" and "+TEST_USER2+") back (2.).",
                containsValues(mAdministrator.getAllUsers(mDomain), TEST_USER1, TEST_USER2));
        assertFalse("The removed user '"+usr_name+"' is still among all users.",
                containsValue(mAdministrator.getAllUsers(mDomain), usr_name));
    }

    /** Implemented test cases:
     *
     * URMAdministrator_0006 (with group), URMAdministrator_0013 (with group),
     * URMAdministrator_0014, URMAdministrator_0015, URMAdministrator_0017,
     * URMAdministrator_0018, URMAdministrator_0021, URMAdministrator_0023,
     * URMAdministrator_0024
     */
    public void test2_getAllGroups() throws Exception {
        String err, grp_name = "URM_junit_test_allgrps";
        try { // delete for sure
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { }
        assertTrue("Cannot get the added groups (1.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP1, TEST_GROUP3));
        assertTrue("Cannot get the added groups (2.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP2, TEST_GROUP3));
        // prepare GUEST user
        URMUser usr = mAdministrator.getUser(TEST_USER1, mDomain);
        if (!URMRoleImpl.contains(usr.getRoleMapping(), URMConstants.URM_ADMIN_ROLE))
            usr.linkToRole(URMConstants.URM_ADMIN_ROLE);
        URMPrincipal princ1 = null;
        try { err = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+err+").", princ1);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        // create group
            // trying create with GUEST role
        try { err = null;
            princ1.getURMAdministrator().createGroup(grp_name, mDomain);
            mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);
        } catch (URMForbiddenException e) { err = e.getMessage(); }
        assertNotNull("Creating group with GUEST role did not throw an exception.", err);
        assertFalse("Creating group with GUEST role was successful (but threw an exception).",
                        mAdministrator.isGroup(grp_name, mDomain));
            // create with ADMIN
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertTrue("Creating group '"+grp_name+"' failed ("+err+").",
                        mAdministrator.isGroup(grp_name, mDomain));
        mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);
            // delete group immediately ...
        try {
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain); }
        assertFalse("The delete '"+grp_name+"' failed.",
                        mAdministrator.isGroup(grp_name, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
            // ... and create group again immidiately
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+grp_name+"' (2.) back("+err+").",
                        mAdministrator.getGroup(grp_name, mDomain));
        mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Create a just created group did not throw an exception.", err);
        
        assertTrue("Cannot get the added groups (3.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP1, TEST_GROUP3));
        assertTrue("Cannot get the added groups (4.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP2, TEST_GROUP3));
        assertTrue("Cannot get the added group '"+grp_name+"' back.",
                containsValue(mAdministrator.getAllGroups(mDomain), grp_name));
        // delete group
            // trying delete with GUEST role
        try { err = null;
            princ1.getURMAdministrator().deleteGroup(grp_name, mDomain);
            mCreatedGroupNames.removeLast(); mCreatedGroupNames.removeLast();
        } catch (URMForbiddenException e) { err = e.getMessage(); }
        assertNotNull("Deleting group with GUEST role did not throw an exception.", err);
        assertTrue("Deleting group with GUEST role was successful (but threw an exception).",
                        mAdministrator.isGroup(grp_name, mDomain));
            // delete with ADMIN
        try {
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertFalse("The delete '"+grp_name+"' failed ("+err+").",
                        mAdministrator.isGroup(grp_name, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
        try { err = null;
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Delete a just deleted group did not throw an exception.", err);
        
        assertNull("The removed group '"+grp_name+"' still exists.",
                        mAdministrator.getGroup(grp_name, mDomain));
        assertTrue("Cannot get the added groups (5.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP1, TEST_GROUP3));
        assertTrue("Cannot get the added groups (6.) back.",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP2, TEST_GROUP3));
        assertFalse("The removed group '"+grp_name+"' is still among all groups.",
                containsValue(mAdministrator.getAllGroups(mDomain), grp_name));
    }
    
    /** Implemented test cases:
     *
     * URMUser_0001, URMUser_0002, URMUser_0003
     * URMGroup_0009, URMGroup_0010, URMGroup_0013, URMGroup_0014,
     */
    public void test3_addRemoveUserToFromGroup() throws Exception {
        String errmsg = null;
        // ADDING
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        assertTrue("The '"+TEST_GROUP1+"' has already a member (nothing was added).",
                        /*grp1.getGroupMembers() == null || */grp1.getGroupMembers().size() <= 0);
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr2);
/*zsa*/        assertTrue("The '"+TEST_USER2+"' is already a member in a group (nothing was added).",
/*zsa*/                        /*usr2.getGroups() == null || */usr2.getGroups().size() <= 0);
        try { errmsg = null;
            grp1.addUserToGroup("URM_junit_non_exist", mDomain);
            assertFalse("Adding a non existing user to group did not throw an exception.", true);
        } catch (URMException e) { errmsg = e.getMessage(); }

            // check with GUEST role
        URMPrincipal princ1 = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+errmsg+").", princ1);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        try { errmsg = null;
            URMUser usr2p1 = princ1.getURMAdministrator().getUser(TEST_USER2, mDomain);
            assertFalse("Creating URMUser object of '"+TEST_USER2+"' did not throw an exception (using GUEST role).", true);
        } catch (URMForbiddenException e) { errmsg = e.getMessage(); }

        grp1.addUserToGroup(TEST_USER2, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' contains '"+TEST_USER1+"' (not added).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (not added).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));

        URMGroup grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
        assertTrue("The group '"+TEST_GROUP2+"' has already a member.",
                        /*grp2.getGroupMembers() == null || */grp2.getGroupMembers().size() <= 0);
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
/*zsa*/        assertTrue("The '"+TEST_USER1+"' is already a member in a group (nothing was added) (using user obj).",
/*zsa*/                        /*usr1.getGroups() == null || */usr1.getGroups().size() <= 0);
/*zsa*/        assertTrue("The '"+TEST_USER1+"' is already a member in a group (nothing was added) (using principal obj).",
/*zsa*/                        /*princ1.getGroups() == null || */princ1.getGroups().size() <= 0);

        grp2.addUserToGroup(TEST_USER1, mDomain);
        assertTrue("The '"+TEST_GROUP2+"' does not contain '"+TEST_USER1+"'.",
                    containsValue(grp2.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP2+"' (using user obj).",
                    containsValue(usr1.getGroups(), TEST_GROUP2));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP2+"' (using principal obj).",
                    containsValue(princ1.getGroups(), TEST_GROUP2));
        assertTrue("The '"+TEST_USER1+"' is member in '"+TEST_GROUP1+"'.",
                    !containsValue(usr1.getGroups(), TEST_GROUP1));
        // and again
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' contains '"+TEST_USER1+"' (not added).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (not added).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));
                    
        grp1.addUserToGroup(TEST_USER1, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER1+"' or "+TEST_USER2+"'.",
                    containsValues(grp1.getGroupMembers(), TEST_USER1, TEST_USER2));
        assertTrue("The '"+TEST_GROUP2+"' does not contain '"+TEST_USER1+"'.",
                    containsValue(grp2.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_GROUP2+"' contains '"+TEST_USER2+"' (not added).",
                    !containsValue(grp2.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP1+"' or '"+TEST_GROUP1+"'.",
                    containsValues(usr1.getGroups(), TEST_GROUP1, TEST_GROUP2));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP2+"'.",
                    containsValue(usr1.getGroups(), TEST_GROUP2));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (not added).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));
        
        Set set = grp1.getGroupMembers();
        assertFalse("'"+TEST_GROUP1+"' contains more than two member.", set.size() > 2);
        set = grp2.getGroupMembers();
        assertFalse("'"+TEST_GROUP2+"' contains more than one member.", set.size() > 1);
        set = usr1.getGroups();
/*zsa*/        assertFalse("'"+TEST_USER1+"' is a member more than two group.", set.size() > 2);
        set = usr2.getGroups();
/*zsa*/        assertFalse("'"+TEST_USER2+"' is a member more than one group.", set.size() > 1);

        try { errmsg = null;
            grp1.addUserToGroup(TEST_USER2, mDomain);
            assertFalse("Adding a user member to group, where the user is already a member, did not throw an exception.", true);
        } catch (URMException e) { errmsg = e.getMessage(); }
                
        // REMOVING
        try { // removing a not existing member
            grp2.removeUserFromGroup(TEST_USER2, mDomain);
            assertFalse("Removing a non existing user member did not throw an exception.", true);
        } catch (URMException e) { }
        // should not have been changed anything
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER1+"' or "+TEST_USER2+"'.",
                    containsValues(grp1.getGroupMembers(), TEST_USER1, TEST_USER2));
        assertTrue("The '"+TEST_GROUP2+"' does not contain '"+TEST_USER1+"'.",
                    containsValue(grp2.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_GROUP2+"' contains '"+TEST_USER2+"' (removed).",
                    !containsValue(grp2.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP1+"' or '"+TEST_GROUP1+"'.",
                    containsValues(usr1.getGroups(), TEST_GROUP1, TEST_GROUP2));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP2+"'.",
                    containsValue(usr1.getGroups(), TEST_GROUP2));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (removed).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));
        
        grp1.removeUserFromGroup(TEST_USER1, mDomain);
        assertTrue("The '"+TEST_GROUP2+"' does not contain '"+TEST_USER1+"'.",
                    containsValue(grp2.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER1+"' is not member in '"+TEST_GROUP2+"'.",
                    containsValue(usr1.getGroups(), TEST_GROUP2));
        assertTrue("The '"+TEST_USER1+"' is still member in '"+TEST_GROUP1+"'.",
                    !containsValue(usr1.getGroups(), TEST_GROUP1));
        // and again
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' contains '"+TEST_USER1+"' (removed).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (removed).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));

        grp2.removeUserFromGroup(TEST_USER1, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' contains '"+TEST_USER1+"' (removed).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER1+"' is member in '"+TEST_GROUP1+"' (removed).",
                    !containsValue(usr1.getGroups(), TEST_GROUP1));

        grp1.removeUserFromGroup(TEST_USER2, mDomain);
        // empty groups checks
        grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        assertTrue("The '"+TEST_GROUP1+"' has already a member (everything is removed).",
                        /*grp1.getGroupMembers() == null || */grp1.getGroupMembers().size() <= 0);
        usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr2);
/*zsa*/        assertTrue("The '"+TEST_USER2+"' is already a member in a group (everything is removed).",
/*zsa*/                        /*usr2.getGroups() == null || */usr2.getGroups().size() <= 0);
        grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
        assertTrue("The group '"+TEST_GROUP2+"' has already a member.",
                        /*grp2.getGroupMembers() == null || */grp2.getGroupMembers().size() <= 0);
        usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
/*zsa*/        assertTrue("The '"+TEST_USER1+"' is already a member in a group (everything is removed).",
/*zsa*/                        /*usr1.getGroups() == null || */usr1.getGroups().size() <= 0);

        //test1_getAllUsers();
        //test2_getAllGroups();
    }

    /** Implemented test cases:
     *
     *  URMUser_0030, URMUser_0031, URMUser_0036, URMUser_0038
     */
    public void test4_setGetUserProperties() throws Exception {
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        // check get with GUEST role
        URMPrincipal princ1 = null;
        String errmsg = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+errmsg+").", princ1);
        if (!URMRoleImpl.contains(usr1.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr1.linkToRole(URMConstants.URM_GUEST_ROLE);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        try { errmsg = null;
            princ1.getURMAdministrator().getUser(TEST_USER1, mDomain).getProperties();
            assertTrue("Getting user properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        if (URMRoleImpl.contains(usr1.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr1.deleteRoleLink(URMConstants.URM_GUEST_ROLE);
        
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
        // set the properties from the 'User1' node
        URMConfigurator userprops = propconf.getSubConfigurator("User1");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User1' test properties from the config file.", sprops);
        
        // check set without ADMIN role
        princ1 = princ1.copyPrincipal(null);
        try { errmsg = null;
            princ1.getURMAdministrator().getUser(TEST_USER1, mDomain).setProperties(sprops);
            assertTrue("Setting user properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        // check set with ADMIN role
        usr1.setProperties(sprops);
        
        Properties gprops = usr1.getProperties();
        gprops = usr1.getProperties();
        assertNotNull("Cannot get the 'User1' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'User1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // set the properties from the 'User2' node
        userprops = propconf.getSubConfigurator("User2");
        sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User2' test properties from the config file.", sprops);
        
        usr1.setProperties(sprops);
        
        gprops = usr1.getProperties();
        assertNotNull("Cannot get the 'User2' test properties back after setting them.", gprops);
        counter = 0;
        keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'User2' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' from the 'User2' test properties, getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
                        
        // clear metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableUserPropKeys();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();)
            clrprops.setProperty((String)iter.next(), "");
        usr1.setProperties(clrprops);
        gprops = usr1.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+TEST_USER1+" still exists (it had been deleted)", gprops.containsKey(propkey));
        }
    }
    
    public void test5_setGetGroupProperties() throws Exception {
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);

        // check with GUEST role
        URMPrincipal princ1 = null;
        String errmsg = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+errmsg+").", princ1);
        if (!URMRoleImpl.contains(usr1.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr1.linkToRole(URMConstants.URM_GUEST_ROLE);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        try { errmsg = null;
            princ1.getURMAdministrator().getGroup(TEST_GROUP1, mDomain).getProperties();
            assertTrue("Getting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        if (URMRoleImpl.contains(grp1.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            grp1.deleteRoleLink(URMConstants.URM_GUEST_ROLE);
        
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
        // set the properties from the 'Group1' node
        URMConfigurator groupprops = propconf.getSubConfigurator("Group1");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group1' test properties from the config file.", sprops);
        
        // check set without ADMIN role
        princ1 = princ1.copyPrincipal(null);
        try { errmsg = null;
            princ1.getURMAdministrator().getGroup(TEST_GROUP1, mDomain).setProperties(sprops);
            assertTrue("Setting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        // check set with ADMIN role
        grp1.setProperties(sprops);
        
        Properties gprops = grp1.getProperties();
        assertNotNull("Cannot get the 'Group1' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'Group1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // set the properties from the 'Group2' node
        groupprops = propconf.getSubConfigurator("Group2");
        sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group2' test properties from the config file.", sprops);
        
        grp1.setProperties(sprops);
        
        gprops = grp1.getProperties();
        assertNotNull("Cannot get the 'Group2' test properties back after setting them.", gprops);
        counter = 0;
        keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'Group2' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' from the 'Group2' test properties, getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
                        
        // clear metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableGroupPropKeys();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();)
            clrprops.setProperty((String)iter.next(), "");
        grp1.setProperties(clrprops);
        gprops = grp1.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+TEST_GROUP1+" still exists (it had been deleted)", gprops.containsKey(propkey));
        }
    }
    
    /** Implemented test cases:
     *
     * URMUser_0004,
     * URMUser_0010, URMUser_0011
     */
    public void test6_setUserPassword() throws Exception {
        char[] newpwd = "newpassword".toCharArray();
        String errmsg = null;
        
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        
        // changing password with admin
        try {
            usr1.setPassword(TEST_USER1.toCharArray(), newpwd);
        } catch (URMException e) {
            assertFalse("Cannot set a new password with admin (giving the old one): "+e.getMessage(), true);
        }
            // check with the old (wrong) password
        URMPrincipal princ1 = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user still can be authenticate (1) with the old password after it had been changed.", princ1);
            // check with the new (correct) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, newpwd, mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user (1) with the new password("+errmsg+").", princ1);
        // check the received principal
        URMAdministrator admin = princ1.getURMAdministrator();
        assertNotNull("Cannot get the administrator object of the '"+TEST_USER1+"' user principal back.", admin);
        usr1 = admin.getUser(TEST_USER1, mDomain);
        Properties uprops = usr1.getProperties();
        assertNotNull("Cannot get the user properties back after modifying the password.", uprops);
        // changing back the password with admin
        usr1 = mAdministrator.getUser(TEST_USER1, mDomain);// with admin again
/*zsa-->*/
        try {
            usr1.setPassword(null, TEST_USER1.toCharArray());
        } catch (URMException e) {
            assertFalse("Cannot set a new password with admin (without giving the old one): "+e.getMessage(), true);
        }
            // check with the old (wrong) password
        URMPrincipal princ2 = null;
        try { errmsg = null;
            princ2 = mAuthenticator.authenticateUser(TEST_USER1, newpwd, mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user still can be authenticate (2) with the old password after it had been changed.", princ2);
            // check with the new (correct) password
        try { errmsg = null;
            princ2 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user (2) with the new password("+errmsg+").", princ2);
        
        // changing password with principal
        try {
            princ2.setPassword(TEST_USER1.toCharArray(), newpwd);
        } catch (URMException e) {
            assertFalse("Cannot set a new password with principal (giving the new one): "+e.getMessage(), true);
        }
/*<--zsa*/
            // check with the old (wrong) password
        URMPrincipal princ3 = null;
        try { errmsg = null;
            princ3 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user still can be authenticate (3) with the old password after it had been changed.", princ3);
            // check with the new (correct) password
        try { errmsg = null;
            princ3 = mAuthenticator.authenticateUser(TEST_USER1, newpwd, mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user (3) with the new password("+errmsg+").", princ3);

        // just to check if it throws an exception
        Map map = princ3.getURMAdministrator().getUser(TEST_USER1, null).getAvailablePropertyNames();
        //System.out.print("\nAvailable user properties: "+map);
        map = princ3.getURMAdministrator().getGroup(TEST_GROUP1, null).getAvailablePropertyNames();
        //System.out.println("\nAvailable group properties: "+map);
        
        // changing back the password with principal
        try {
            princ3.setPassword(newpwd, TEST_USER1.toCharArray());
        } catch (URMException e) {
            assertFalse("Cannot set a new password with principal (giving the old one): "+e.getMessage(), true);
        }
            // check with the old (wrong) password
        URMPrincipal princ4 = null;
        try { errmsg = null;
            princ4 = mAuthenticator.authenticateUser(TEST_USER1, newpwd, mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user still can be authenticate (4) with the old password after it had been changed.", princ4);
            // check with the new (correct) password
        try { errmsg = null;
            princ4 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user (4) with the new password("+errmsg+").", princ4);

        // checks the old principal whether it works
        Set grps4 = null;
        try {
            grps4 = princ4.getGroups();
            //System.out.println("Groups of user '"+TEST_USER1+"': "+grps4);
        } catch (URMException e) {
            assertFalse("Cannot read the groups after changing password: "+e.getMessage(), true);
        }

        Set grps1 = null;
        try {
            grps1 = princ1.getGroups();
            //System.out.println("Groups of user '"+TEST_USER1+"': "+grps1);
        } catch (URMException e) {
            grps1 = null;
        }
        if (grps1 != null) {
            System.out.println("\nAfter changing the password (and caching is enabled) \n"+
                                "you can only query user information from the cache using \n"+
                                "an old principal (created with the old password).");
        }
    }
    
    public void test7_isUserIsGroup() throws Exception {
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        assertTrue("The '"+TEST_GROUP1+"' is not a group.",
                mAdministrator.isGroup(grp1.getName(), grp1.getDomain()));
        assertFalse("The group '"+TEST_GROUP1+"' is a user.",
                mAdministrator.isUser(grp1.getName(), grp1.getDomain()));
        URMGroup grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
        assertTrue("The '"+TEST_GROUP2+"' is not a group.",
                mAdministrator.isGroup(TEST_GROUP2, mDomain));
        assertFalse("The group '"+TEST_GROUP2+"' is a user.",
                mAdministrator.isUser(TEST_GROUP2, mDomain));
        
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        assertTrue("The '"+TEST_USER1+"' is not a user.",
                mAdministrator.isUser(usr1.getName(), usr1.getDomain()));
        assertFalse("The user '"+TEST_USER1+"' is a group.",
                mAdministrator.isGroup(usr1.getName(), usr1.getDomain()));
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr1);
        assertTrue("The '"+TEST_USER2+"' is not a user.",
                mAdministrator.isUser(TEST_USER2, mDomain));
        assertFalse("The user '"+TEST_USER2+"' is a group.",
                mAdministrator.isGroup(TEST_USER2, mDomain));
    }
    
    /** Implemented test cases:
     *
     * URMGroup_0001, URMGroup_0004, URMGroup_0005, URMGroup_0007,
     */
    public void test8_addRemoveGroupToFromGroup() throws Exception {
        String errmsg = null;
        // ADDING
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
/*zsa*/        assertTrue("The '"+TEST_GROUP1+"' has already a member (nothing was added).",
/*zsa*/                        /*grp1.getGroupMembers() == null || */grp1.getGroupMembers().size() <= 0);
        URMGroup grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
/*zsa*/        assertTrue("The '"+TEST_GROUP2+"' has already a member (nothing was added).",
/*zsa*/                        /*grp2.getGroupMembers() == null || */grp2.getGroupMembers().size() <= 0);
        URMGroup grp3 = mAdministrator.getGroup(TEST_GROUP3, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP3+"' back.", grp3);
/*zsa*/        assertTrue("The '"+TEST_GROUP2+"' has already a member (nothing was added).",
/*zsa*/                        /*grp3.getGroupMembers() == null || */grp3.getGroupMembers().size() <= 0);
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr2);
/*zsa*/        assertTrue("The '"+TEST_USER2+"' is already a member in a group (nothing was added).",
/*zsa*/                        /*usr2.getGroups() == null || */usr2.getGroups().size() <= 0);

        try { errmsg = null;
            grp1.addGroupToGroup("URM_junit_non_exist", mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Adding a non existing group to group did not throw an exception.", errmsg);

            // check with GUEST role
        URMPrincipal princ1 = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user ("+errmsg+").", princ1);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        try { errmsg = null;
            URMGroup grp2p1 = princ1.getURMAdministrator().getGroup(TEST_GROUP2, mDomain);
            assertFalse("Creating URMUser object of '"+TEST_GROUP2+"' did not throw an exception (using GUEST role).", true);
        } catch (URMForbiddenException e) { errmsg = e.getMessage(); }

        grp1.addUserToGroup(TEST_USER2, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_USER2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' contains '"+TEST_USER1+"' (not added).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertTrue("The '"+TEST_USER2+"' is not member in '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        assertTrue("The '"+TEST_USER2+"' is member in '"+TEST_GROUP2+"' (not added).",
                    !containsValue(usr2.getGroups(), TEST_GROUP2));

        Set grps = grp1.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP1+"' is a member already in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(just a user was added).", /*grps == null || */grps.size() <= 0);
        grps = grp2.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP2+"' is a member already in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(just a user was added).", /*grps == null || */grps.size() <= 0);
        grps = grp3.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP3+"' is a member already in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(nothing was added).", /*grps == null || */grps.size() <= 0);
                        
        grp1.addGroupToGroup(TEST_GROUP2, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_GROUP2+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_GROUP2));
        assertTrue("The '"+TEST_GROUP2+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp2.getGroupsMemberOf(), TEST_GROUP1));

        grp1.addGroupToGroup(TEST_GROUP3, mDomain);
        grps = grp1.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP1+"' is a member already in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(just a user was added and two other groups to this one).", /*grps == null || */grps.size() <= 0);
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_GROUP2+"' or '"+TEST_GROUP3+"'.",
                    containsValues(grp1.getGroupMembers(), TEST_GROUP2, TEST_GROUP3));
        assertTrue("The '"+TEST_GROUP2+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp2.getGroupsMemberOf(), TEST_GROUP1));
        assertTrue("The '"+TEST_GROUP3+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp3.getGroupsMemberOf(), TEST_GROUP1));
        
        try { errmsg = null;
            grp1.addGroupToGroup(TEST_GROUP2, mDomain);
            assertFalse("Adding a group member to group, where the group is already a member, did not throw an exception.", true);
        } catch (URMException e) { errmsg = e.getMessage(); }
                
        // REMOVING
        grp1.removeGroupFromGroup(TEST_GROUP2, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' still contains '"+TEST_GROUP2+"' (removed).",
                    !containsValue(grp1.getGroupMembers(), TEST_GROUP2));
        assertTrue("The '"+TEST_GROUP2+"' is still a member of '"+TEST_GROUP1+"' (removed).",
                    !containsValue(grp2.getGroupsMemberOf(), TEST_GROUP1));
        try { // removing a not existing member
            grp1.removeGroupFromGroup(TEST_GROUP2, mDomain);
            assertFalse("Removing a non existing group member did not throw an exception.", true);
        } catch (URMException e) { }

        grp1.removeUserFromGroup(TEST_USER2, mDomain);
        assertTrue("The '"+TEST_GROUP1+"' still contains '"+TEST_USER2+"' (removed).",
                    !containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertTrue("The '"+TEST_GROUP1+"' does not contain '"+TEST_GROUP3+"'.",
                    containsValue(grp1.getGroupMembers(), TEST_GROUP3));
        assertTrue("The '"+TEST_GROUP3+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp3.getGroupsMemberOf(), TEST_GROUP1));

        grp1.removeGroupFromGroup(TEST_GROUP3, mDomain);
        grps = grp1.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP1+"' is a member in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(everything was removed).", /*grps == null || */grps.size() <= 0);
        grps = grp2.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP2+"' is still a member in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(everything was removed).", /*grps == null || */grps.size() <= 0);
        grps = grp3.getGroupsMemberOf();
        assertTrue("The '"+TEST_GROUP3+"' is still a member in '" + (grps.size() > 0 ? grps.toArray()[0] : "") +
                    "'(everything was removed).", /*grps == null || */grps.size() <= 0);
    }
    
    public void test9_isUserIsGroupAgainBecauseOfPossibleCachingProblems() throws Exception {
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        assertTrue("The '"+TEST_GROUP1+"' is not a group.",
                mAdministrator.isGroup(grp1.getName(), grp1.getDomain()));
        assertFalse("The group '"+TEST_GROUP1+"' is a user.",
                mAdministrator.isUser(grp1.getName(), grp1.getDomain()));
        URMGroup grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
        assertTrue("The '"+TEST_GROUP2+"' is not a group.",
                mAdministrator.isGroup(TEST_GROUP2, mDomain));
        assertFalse("The group '"+TEST_GROUP2+"' is a user.",
                mAdministrator.isUser(TEST_GROUP2, mDomain));
        
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        assertTrue("The '"+TEST_USER1+"' is not a user.",
                mAdministrator.isUser(usr1.getName(), usr1.getDomain()));
        assertFalse("The user '"+TEST_USER1+"' is a group.",
                mAdministrator.isGroup(usr1.getName(), usr1.getDomain()));
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr1);
        assertTrue("The '"+TEST_USER2+"' is not a user.",
                mAdministrator.isUser(TEST_USER2, mDomain));
        assertFalse("The user '"+TEST_USER2+"' is a group.",
                mAdministrator.isGroup(TEST_USER2, mDomain));
    }
    
    public void test10_isUserMember() throws Exception {
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        if (!containsValue(grp1.getGroupMembers(), TEST_USER2))
            grp1.addUserToGroup(TEST_USER2, mDomain);
        assertTrue("The added '"+TEST_USER2+"' is not a groupmember in '"+TEST_GROUP1+"'.",
                grp1.isUserMember(TEST_USER2, mDomain));
        assertFalse("The added '"+TEST_USER2+"' is not a groupmember in '"+TEST_GROUP1+"'.",
                grp1.isGroupMember(TEST_USER2, mDomain));
        assertTrue("The '"+TEST_USER2+"' cannot be found among the groupmember of '"+
                TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), TEST_USER2));

        URMUser usr2 = null;
        for (Iterator iter = grp1.getGroupMembers().iterator(); iter.hasNext();) {
            URMSubject sbj = (URMSubject)iter.next();
            if (mAdministrator.isUser(sbj.getName(), sbj.getDomain())
                    && (usr2 = (URMUser)sbj).getName().equals(TEST_USER2))
                break;
            else
                usr2 = null;
        }
        assertTrue("The '"+TEST_USER2+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
        // remove from group
        grp1.removeUserFromGroup(TEST_USER2, mDomain);
        assertFalse("The removed '"+TEST_USER2+"' is still a groupmember in '"+TEST_GROUP1+"'.",
                    grp1.isGroupMember(TEST_USER2, mDomain));
        assertFalse("The '"+TEST_USER2+"' can be still found among the groupmember of '"+
                    TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), TEST_USER2));
        assertFalse("The '"+TEST_USER2+"' is still a member of '"+TEST_GROUP1+"'.",
                    containsValue(usr2.getGroups(), TEST_GROUP1));
                     
        // check whether the delete user operation removes the user from the groups as well
        String user_del_member1 = "URM_junit_del_user1";
        if (!mAdministrator.isUser(user_del_member1, mDomain))
            mAdministrator.createUser(user_del_member1, user_del_member1.toCharArray(), mDomain);
        assertTrue("The '"+user_del_member1+"' cannot be found among the all users (after create).",
                                containsValue(mAdministrator.getAllUsers(mDomain), user_del_member1));
        assertNotNull("getUser() does not return the just created '"+user_del_member1+"' user.",
                                mAdministrator.getUser(user_del_member1, mDomain));
        mCreatedUserNames.add(user_del_member1); mCreatedUserDomains.add(mDomain);
        
        grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        if (!containsValue(grp1.getGroupMembers(), user_del_member1))
            grp1.addUserToGroup(user_del_member1, mDomain);
        assertTrue("The added '"+user_del_member1+"' is not a user member in '"+user_del_member1+"'.",
                grp1.isUserMember(user_del_member1, mDomain));
        assertFalse("The added '"+user_del_member1+"' user is a group member in '"+TEST_GROUP1+"'.",
                grp1.isGroupMember(user_del_member1, mDomain));
        assertTrue("The '"+user_del_member1+"' cannot be found among the groupmember of '"+
                TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), user_del_member1));

        URMUser usr1 = null;
        for (Iterator iter = grp1.getGroupMembers().iterator(); iter.hasNext();) {
            URMSubject sbj = (URMSubject)iter.next();
            if (mAdministrator.isUser(sbj.getName(), sbj.getDomain())
                    && (usr1 = (URMUser)sbj).getName().equals(user_del_member1))
                break;
            else
                usr1 = null;
        }
        assertTrue("The '"+user_del_member1+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(usr1.getGroups(), TEST_GROUP1));

        mAdministrator.deleteUser(user_del_member1, mDomain);
        assertFalse("The '"+user_del_member1+"' can be still found among the all users (after delete).",
                                containsValue(mAdministrator.getAllUsers(mDomain), user_del_member1));
        assertNull("getUser() still returns the deleted '"+user_del_member1+"' user.",
                                mAdministrator.getUser(user_del_member1, mDomain));
        mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
                                
        mAdministrator.createUser(user_del_member1, user_del_member1.toCharArray(), mDomain);
        assertTrue("The '"+user_del_member1+"' cannot be found among the all users (after create).",
                                containsValue(mAdministrator.getAllUsers(mDomain), user_del_member1));
        assertNotNull("getUser() did not return the just created '"+user_del_member1+"' user.",
                                mAdministrator.getUser(user_del_member1, mDomain));
        mCreatedUserNames.add(user_del_member1); mCreatedUserDomains.add(mDomain);
        
        assertFalse("The removed '"+user_del_member1+"' is still a groupmember in '"+TEST_GROUP1+"'.",
                    grp1.isGroupMember(user_del_member1, mDomain));
        assertFalse("The '"+user_del_member1+"' can be  sill found among the groupmember of '"+
                    TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), user_del_member1));
        assertFalse("The '"+user_del_member1+"' is still a member of '"+TEST_GROUP1+"'.",
                    containsValue(usr1.getGroups(), TEST_GROUP1));

        mAdministrator.deleteUser(user_del_member1, mDomain);
        assertFalse("The '"+user_del_member1+"' can be still found among the all users (after delete).",
                                containsValue(mAdministrator.getAllUsers(mDomain), user_del_member1));
        assertNull("getUser() still returns the deleted '"+user_del_member1+"' user.",
                                mAdministrator.getUser(user_del_member1, mDomain));
        mCreatedUserNames.remove(user_del_member1); mCreatedUserDomains.removeLast();
    }
    
    public void test11_isGroupMember() throws Exception {
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        grp1.addUserToGroup(TEST_USER1, mDomain);
        grp1.addGroupToGroup(TEST_GROUP2, mDomain);
        assertTrue("The added '"+TEST_GROUP2+"' is not a groupmember in '"+TEST_GROUP1+"'.",
                grp1.isGroupMember(TEST_GROUP2, mDomain));
        assertFalse("The added '"+TEST_GROUP2+"' is not a groupmember in '"+TEST_GROUP1+"'.",
                grp1.isUserMember(TEST_GROUP2, mDomain));
        assertTrue("The '"+TEST_GROUP2+"' cannot be found among the groupmember of '"+
                TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), TEST_GROUP2));

        URMGroup grp2 = null;
        for (Iterator iter = grp1.getGroupMembers().iterator(); iter.hasNext();) {
            URMSubject sbj = (URMSubject)iter.next();
            if (mAdministrator.isGroup(sbj.getName(), sbj.getDomain())
                    && (grp2 = (URMGroup)sbj).getName().equals(TEST_GROUP2))
                break;
            else
                grp2 = null;
        }
        assertTrue("The '"+TEST_GROUP2+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp2.getGroupsMemberOf(), TEST_GROUP1));
        // remove group
        grp1.removeGroupFromGroup(TEST_GROUP2, mDomain);
        assertFalse("The removed '"+TEST_GROUP2+"' is still a groupmember in '"+TEST_GROUP1+"'.",
                    grp1.isGroupMember(TEST_GROUP2, mDomain));
        assertFalse("The '"+TEST_GROUP2+"' cannot be found among the groupmember of '"+
                    TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), TEST_GROUP2));
        assertFalse("The '"+TEST_GROUP2+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(grp2.getGroupsMemberOf(), TEST_GROUP1));
        // remove user
        grp1.removeUserFromGroup(TEST_USER1, mDomain);
        assertFalse("The removed '"+TEST_USER1+"' is still a groupmember in '"+TEST_GROUP1+"'.",
                    grp1.isGroupMember(TEST_USER1, mDomain));
        assertFalse("The '"+TEST_USER1+"' cannot be found among the groupmember of '"+
                    TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), TEST_USER1));
        assertFalse("The '"+TEST_USER1+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(mAdministrator.getUser(TEST_USER1, mDomain).getGroups(), TEST_GROUP1));
                     
        // check whether the delete user operation removes the user from the groups as well
        String group_del_member1 = "URM_junit_del_group1";
        if (!mAdministrator.isGroup(group_del_member1, mDomain))
            mAdministrator.createGroup(group_del_member1, mDomain);
        assertTrue("The '"+group_del_member1+"' cannot be found among the all groups (after create).",
                                containsValue(mAdministrator.getAllGroups(mDomain), group_del_member1));
        assertNotNull("getGroup() does not return the just created '"+group_del_member1+"' group.",
                                mAdministrator.getGroup(group_del_member1, mDomain));
        mCreatedGroupNames.add(group_del_member1); mCreatedGroupDomains.add(mDomain);
        
        grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);
        if (!containsValue(grp1.getGroupMembers(), group_del_member1))
            grp1.addGroupToGroup(group_del_member1, mDomain);
        assertTrue("The added '"+group_del_member1+"' is not a group member in '"+group_del_member1+"'.",
                grp1.isGroupMember(group_del_member1, mDomain));
        assertFalse("The added '"+group_del_member1+"' group is a user member in '"+TEST_GROUP1+"'.",
                grp1.isUserMember(group_del_member1, mDomain));
        assertTrue("The '"+group_del_member1+"' cannot be found among the groupmember of '"+
                TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), group_del_member1));

        URMGroup mgrp1 = null;
        for (Iterator iter = grp1.getGroupMembers().iterator(); iter.hasNext();) {
            URMSubject sbj = (URMSubject)iter.next();
            if (mAdministrator.isGroup(sbj.getName(), sbj.getDomain())
                    && (mgrp1 = (URMGroup)sbj).getName().equals(group_del_member1))
                break;
            else
                mgrp1 = null;
        }
        assertTrue("The '"+group_del_member1+"' is not a member of '"+TEST_GROUP1+"'.",
                    containsValue(mgrp1.getGroupsMemberOf(), TEST_GROUP1));

        mAdministrator.deleteGroup(group_del_member1, mDomain);
        assertFalse("The '"+group_del_member1+"' can be still found among the all groups (after delete).",
                                containsValue(mAdministrator.getAllGroups(mDomain), group_del_member1));
        assertNull("getUser() still returns the deleted '"+group_del_member1+"' group.",
                                mAdministrator.getGroup(group_del_member1, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
        
        mAdministrator.createGroup(group_del_member1, mDomain);
        assertTrue("The '"+group_del_member1+"' cannot be found among the all groups (after create).",
                                containsValue(mAdministrator.getAllGroups(mDomain), group_del_member1));
        assertNotNull("getUser() did not return the just created '"+group_del_member1+"' group.",
                                mAdministrator.getGroup(group_del_member1, mDomain));
        mCreatedGroupNames.add(group_del_member1); mCreatedGroupDomains.add(mDomain);
        
        assertFalse("The removed '"+group_del_member1+"' is still a groupmember in '"+TEST_GROUP1+"'.",
                    grp1.isGroupMember(group_del_member1, mDomain));
        assertFalse("The '"+group_del_member1+"' can be still found among the groupmember of '"+
                    TEST_GROUP1+"'.", containsValue(grp1.getGroupMembers(), group_del_member1));
        assertFalse("The '"+group_del_member1+"' is still a member of '"+TEST_GROUP1+"'.",
                    containsValue(mgrp1.getGroupsMemberOf(), TEST_GROUP1));

        mAdministrator.deleteGroup(group_del_member1, mDomain);
        assertFalse("The '"+group_del_member1+"' can be still found among the all groups (after delete).",
                                containsValue(mAdministrator.getAllGroups(mDomain), group_del_member1));
        assertNull("getUser() still returns the deleted '"+group_del_member1+"' group.",
                                mAdministrator.getGroup(group_del_member1, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
    }

    public void test12_synchronizeUsers() throws Exception {
        String err = null;
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/Synchronization/Users");
        Hashtable users_properies = new Hashtable();
        // create users and set the properties of them
        Map usrs_subconf = propconf.getSubConfigurators();
        Object ousr = usrs_subconf.get("User");
        if (ousr instanceof URMConfigurator) {
            LinkedList ousrll = new LinkedList();
            ousrll.add(ousr); ousr = ousrll;
        }
        String resourceid1 = null, resourceid2 = null;
        for (Iterator iter = ((List)ousr).iterator(); iter.hasNext();) {
            ousr = iter.next();
            Properties usrprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                ((URMConfigurator)ousr).getSubConfigurator("MetaProperties"), "Property", "name", "value");
            if (usrprops == null || usrprops.size() <= 0)
                continue; // it is no test without properties

            Properties uinf = ((URMConfigurator)ousr).getProperties();
            String usrname = uinf.getProperty("name");
            String dmnname = (uinf.getProperty("domain") != null && uinf.getProperty("domain").length() > 0) ? uinf.getProperty("domain") : null;
            // create a user
            try { err = null;
                mAdministrator.createUser(usrname, uinf.getProperty("password").toCharArray(), dmnname);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+usrname+"'("+dmnname+") back("+err+").",
                            mAdministrator.getUser(usrname, dmnname));
            mCreatedUserNames.add(usrname);
            mCreatedUserDomains.add(dmnname);
            users_properies.put(usrname + "/" + dmnname, usrprops);
        
            URMUser usr = mAdministrator.getUser(usrname, dmnname);
            assertNotNull("Cannot get the '"+usrname+"'("+dmnname+") back.", usr);
            if (usrprops != null && usrprops.size() > 0) {
                usr.setProperties(usrprops);
                assertNotNull("Cannot get the '"+usrname+"'("+dmnname+") test properties back after setting them.", usr.getProperties());
            }
            // create roles, role links
            URMConfigurator roleconf = ((URMConfigurator)ousr).getSubConfigurator("RoleLinks");
            Object orole = roleconf != null ? roleconf.getSubConfigurators().get("Role") : null;
            if (orole == null)
                continue;
            if (orole instanceof URMConfigurator) {
                LinkedList orolell = new LinkedList();
                orolell.add(orole); orole = orolell;
            }
            for (Iterator riter = ((List)orole).iterator(); riter.hasNext();) {
                roleconf = (URMConfigurator)riter.next();
                String rolename = roleconf.getProperties().getProperty("name");
                // create role
                try { err = null;
                    if (mAdministrator.getRole(rolename) == null)
                        mAdministrator.createRole(rolename);
                } catch (URMException e) { err = e.getMessage(); }
                URMRole urm_role = mAdministrator.getRole(rolename);
                assertNotNull("Cannot get the just created '"+rolename+"' back("+err+").", urm_role);
                if (!mCreatedRoleNames.contains(rolename))
                    mCreatedRoleNames.add(rolename);
                // add user link
                try { err = null;
                    if (!URMRoleGuestImpl.contains(usr.getRoleMapping(), rolename))
                        usr.linkToRole(rolename);
                } catch (URMException e) { err = e.getMessage(); }
                assertNull("Cannot creatre the '"+rolename+"' ("+err+").", err);
                // create acl for the user
                try { err = null;
                    if (mAclAdministrator.getAcl(TEST_RESOURCE_ID1) != null)
                        resourceid1 = TEST_RESOURCE_ID1;
                    else
                        resourceid1 = mAclAdministrator.createAcl(TEST_RESOURCE_ID1, usr, false);
                } catch (URMException e) { err = e.getMessage(); }
                assertNotNull("Creating ACL for '"+usr.getName()+"' returned null resource id("+err+").",
                                resourceid1);
                assertNotNull("Cannot get the just created ACL for '"+usr.getName()+"' back("+err+").",
                                mAclAdministrator.getAcl(resourceid1));
                if (!mCreatedAclIds.contains(resourceid1))
                    mCreatedAclIds.add(resourceid1);
                // create acl for a role and ace for the user in it
                try { err = null;
                    if (mAclAdministrator.getAcl(TEST_RESOURCE_ID2) != null)
                        resourceid2 = TEST_RESOURCE_ID2;
                    else
                        resourceid2 = mAclAdministrator.createAcl(TEST_RESOURCE_ID2, urm_role, false);
                } catch (URMException e) { err = e.getMessage(); }
                assertNotNull("Creating ACL for '"+urm_role.getName()+"' returned null resource id("+err+").",
                                resourceid2);
                URMAcl urm_acl = mAclAdministrator.getAcl(resourceid2);
                assertNotNull("Cannot get the just created ACL for '"+urm_role.getName()+"' back("+err+").",
                                urm_acl);
                if (!mCreatedAclIds.contains(resourceid2))
                    mCreatedAclIds.add(resourceid2);
                urm_acl.defineAce(URMConstants.GRANT, usr, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
                assertNotNull("Cannot get the just created ACE for '"+usr.getName()+"' (in ACL of '"+
                                urm_role.getName()+"') back("+err+").", urm_acl.getAce(0));
            }
        }
        
        // delete the users
        for (Iterator iteru = mCreatedUserNames.iterator(), iterd = mCreatedUserDomains.iterator(); iteru.hasNext();) {
            Object obj = iterd.next();
            String usrnm = (String)iteru.next(), dmnnm = (obj != null) ? (String)obj : null;
            /*try { err = null;
                mAdministrator.deleteUser(usrnm, dmnnm);
                assertFalse("The delete '"+usrnm+"'("+dmnnm+") did not fail (it has ACL reference).", true);
            } catch (URMDeleteException e) { err = e.getMessage(); }
            //delete ACEs
            URMAcl acl = mAclAdministrator.getAcl(resourceid1);
            for (int idx = acl.getAllAces().size() -1; idx >= 0; --idx)
                acl.removeAce(idx);
            acl = mAclAdministrator.getAcl(resourceid2);
            for (int idx = acl.getAllAces().size() -1; idx >= 0; --idx)
                acl.removeAce(idx);*/
            mAdministrator.deleteUser(usrnm, dmnnm);
            //((URMGuestAdministratorImpl)mAdministrator).mUserDBManagerSpi.deleteUser(usrnm, dmnnm);
            assertTrue("The delete '"+usrnm+"'("+dmnnm+") failed (created in setUp).",
                            !mAdministrator.isUser(usrnm, dmnnm));
        }

        // synchronize the database
        mAdministrator.synchronizeUsersGroupsRoles();
        
        // create the same users again and check the properties
        for (Iterator iteru = mCreatedUserNames.iterator(), iterd = mCreatedUserDomains.iterator(); iteru.hasNext();) {
            Object obj = iterd.next();
            String usrnm = (String)iteru.next(), dmnnm = (obj != null) ? (String)obj : null;
            try { err = null;
                mAdministrator.createUser(usrnm, usrnm.toCharArray(), dmnnm);
            } catch (URMException e) { err = e.getMessage(); }
            URMUser urm_user = mAdministrator.getUser(usrnm, dmnnm);
            assertNotNull("Cannot get the just created '"+usrnm+"'("+dmnnm+") back("+err+").", urm_user);
            // check whether we can get back one of the properties that we earlier set
            Properties sprops = (Properties)users_properies.get(usrnm + "/" + dmnnm);
            Properties gprops = urm_user.getProperties();
            Enumeration keys = gprops.keys();
            while (keys.hasMoreElements()) {
                String key = (String)keys.nextElement();
                assertNull("The synchronization did not removed the '"+key+"' metaproperty.", sprops.getProperty(key));
            }
            // check role mapping
            assertTrue("After the synchronization there are still any role mapping of the removed user '"+usrnm+"'.",
                        urm_user.getRoleMapping() == null || urm_user.getRoleMapping().size() == 0);
        }
        // check existing acls
            // it is commented out while the acls are not removed any more
        /*HashSet acl_res_ids = new HashSet(mCreatedAclIds); //make a copy because of the remove()
        for (Iterator acliter = acl_res_ids.iterator(); acliter.hasNext();) {
            String resourceid = (String)acliter.next();
            URMAcl urm_acl = mAclAdministrator.getAcl(resourceid);
            if (urm_acl != null && mAdministrator.isRole(urm_acl.getOwner().getName())) {
                assertTrue("After the synchronization there is still an ACE in the ACL with ID '"+resourceid+"'.",
                        urm_acl.getAllAces() == null || urm_acl.getAllAces().size() == 0);
                continue;
            }
            assertNull("After the synchronization there is still an ACL with the ID '"+resourceid+"'.",
                        urm_acl);
            mCreatedAclIds.remove(resourceid);
        }*/
    }
    
    public void test13_synchronizeGroups() throws Exception {
        String err = null;
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/Synchronization/Groups");
        Hashtable groups_properies = new Hashtable();
        // create groups and set the properties of them
        Map grps_subconf = propconf.getSubConfigurators();
        Object ogrp = grps_subconf.get("Group");
        if (ogrp instanceof URMConfigurator) {
            LinkedList ogrpll = new LinkedList();
            ogrpll.add(ogrp); ogrp = ogrpll;
        }
        String resourceid1 = null, resourceid2 = null;
        for (Iterator iter = ((List)ogrp).iterator(); iter.hasNext();) {
            ogrp = iter.next();
            Properties grpprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                ((URMConfigurator)ogrp).getSubConfigurator("MetaProperties"), "Property", "name", "value");
            if (grpprops == null || grpprops.size() <= 0)
                continue; // it is no test without properties

            Properties uinf = ((URMConfigurator)ogrp).getProperties();
            String grpname = uinf.getProperty("name");
            String dmnname = (uinf.getProperty("domain") != null && uinf.getProperty("domain").length() > 0) ? uinf.getProperty("domain") : null;
            // create a group
            try { err = null;
                mAdministrator.createGroup(grpname, dmnname);
            } catch (URMException e) { err = e.getMessage(); }
            assertNotNull("Cannot get the just created '"+grpname+"'("+dmnname+") back("+err+").",
                            mAdministrator.getGroup(grpname, dmnname));
            mCreatedGroupNames.add(grpname);
            mCreatedGroupDomains.add(dmnname);
            groups_properies.put(grpname + "/" + dmnname, grpprops);
    
            URMGroup grp = mAdministrator.getGroup(grpname, dmnname);
            assertNotNull("Cannot get the '"+grpname+"'("+dmnname+") back.", grp);
            if (grpprops != null && grpprops.size() > 0) {
                grp.setProperties(grpprops);
                assertNotNull("Cannot get the '"+grpname+"'("+dmnname+") test properties back after setting them.", grp.getProperties());
            }
            // create roles, role links
            URMConfigurator roleconf = ((URMConfigurator)ogrp).getSubConfigurator("RoleLinks");
            Object orole = roleconf != null ? roleconf.getSubConfigurators().get("Role") : null;
            if (orole == null)
                continue;
            if (orole instanceof URMConfigurator) {
                LinkedList orolell = new LinkedList();
                orolell.add(orole); orole = orolell;
            }
            for (Iterator riter = ((List)orole).iterator(); riter.hasNext();) {
                roleconf = (URMConfigurator)riter.next();
                String rolename = roleconf.getProperties().getProperty("name");
                // create role
                try { err = null;
                    if (mAdministrator.getRole(rolename) == null)
                        mAdministrator.createRole(rolename);
                } catch (URMException e) { err = e.getMessage(); }
                URMRole urm_role = mAdministrator.getRole(rolename);
                assertNotNull("Cannot get the just created '"+rolename+"' back("+err+").", urm_role);
                if (!mCreatedRoleNames.contains(rolename))
                    mCreatedRoleNames.add(rolename);
                // add group link
                try { err = null;
                    if (!URMRoleGuestImpl.contains(grp.getRoleMapping(), rolename))
                        grp.linkToRole(rolename);
                } catch (URMException e) { err = e.getMessage(); }
                assertNull("Cannot creatre the '"+rolename+"' ("+err+").", err);
                // create acl for the group
                try { err = null;
                    if (mAclAdministrator.getAcl(TEST_RESOURCE_ID1) != null)
                        resourceid1 = TEST_RESOURCE_ID1;
                    else
                        resourceid1 = mAclAdministrator.createAcl(TEST_RESOURCE_ID1, grp, false);
                } catch (URMException e) { err = e.getMessage(); }
                assertNotNull("Creating ACL for '"+grp.getName()+"' returned null resource id("+err+").",
                                resourceid1);
                assertNotNull("Cannot get the just created ACL for '"+grp.getName()+"' back("+err+").",
                                mAclAdministrator.getAcl(resourceid1));
                if (!mCreatedAclIds.contains(resourceid1))
                    mCreatedAclIds.add(resourceid1);
                // create acl for a role and ace for the group in it
                try { err = null;
                    if (mAclAdministrator.getAcl(TEST_RESOURCE_ID2) != null)
                        resourceid2 = TEST_RESOURCE_ID2;
                    else
                        resourceid2 = mAclAdministrator.createAcl(TEST_RESOURCE_ID2, urm_role, false);
                } catch (URMException e) { err = e.getMessage(); }
                assertNotNull("Creating ACL for '"+urm_role.getName()+"' returned null resource id("+err+").",
                                resourceid2);
                URMAcl urm_acl = mAclAdministrator.getAcl(resourceid2);
                assertNotNull("Cannot get the just created ACL for '"+urm_role.getName()+"' back("+err+").",
                                urm_acl);
                if (!mCreatedAclIds.contains(resourceid2))
                    mCreatedAclIds.add(resourceid2);
                urm_acl.defineAce(URMConstants.GRANT, grp, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
                assertNotNull("Cannot get the just created ACE for '"+grp.getName()+"' (in ACL of '"+
                                urm_role.getName()+"') back("+err+").", urm_acl.getAce(0));
            }
        }
    
        // delete the groups
        for (Iterator iteru = mCreatedGroupNames.iterator(), iterd = mCreatedGroupDomains.iterator(); iteru.hasNext();) {
            Object obj = iterd.next();
            String grpnm = (String)iteru.next(), dmnnm = (obj != null) ? (String)obj : null;
            /*try { err = null;
                mAdministrator.deleteGroup(grpnm, dmnnm);
            } catch (URMDeleteException e) {err = e.getMessage(); }
            assertNotNull("The delete '"+grpnm+"'("+dmnnm+") did not fail (it has ACE reference).", err);
            //delete ACEs
            URMAcl acl = mAclAdministrator.getAcl(resourceid1);
            for (int idx = acl.getAllAces().size() -1; idx >= 0; --idx)
                acl.removeAce(idx);
            acl = mAclAdministrator.getAcl(resourceid2);
            for (int idx = acl.getAllAces().size()- 1; idx >= 0; --idx)
                acl.removeAce(idx);*/
            mAdministrator.deleteGroup(grpnm, dmnnm);
            //((URMGuestAdministratorImpl)mAdministrator).mUserDBManagerSpi.deleteGroup(grpnm, dmnnm);
            assertTrue("The delete '"+grpnm+"'("+dmnnm+") failed (created in setUp).",
                            !mAdministrator.isGroup(grpnm, dmnnm));
        }

        // synchronize the database
        mAdministrator.synchronizeUsersGroupsRoles();
    
        // create the same groups again and check the properties
        for (Iterator iteru = mCreatedGroupNames.iterator(), iterd = mCreatedGroupDomains.iterator(); iteru.hasNext();) {
            Object obj = iterd.next();
            String grpnm = (String)iteru.next(), dmnnm = (obj != null) ? (String)obj : null;
            try { err = null;
                mAdministrator.createGroup(grpnm, dmnnm);
            } catch (URMException e) { err = e.getMessage(); }
            URMGroup urm_group = mAdministrator.getGroup(grpnm, dmnnm);
            assertNotNull("Cannot get the just created '"+grpnm+"'("+dmnnm+") back("+err+").", urm_group);
            // check whether we can get back one of the properties that we earlier set
            Properties sprops = (Properties)groups_properies.get(grpnm + "/" + dmnnm);
            Properties gprops = urm_group.getProperties();
            Enumeration keys = gprops.keys();
            while (keys.hasMoreElements()) {
                String key = (String)keys.nextElement();
                assertNull("The synchronization did not removed the '"+key+"' metaproperty.", sprops.getProperty(key));
            }
            // check role mapping
            assertTrue("After the synchronization there are still any role mapping of the removed group '"+grpnm+"'.",
                        urm_group.getRoleMapping() == null || urm_group.getRoleMapping().size() == 0);
        }
        // check existing acls
            // it is commented out while the acls are not removed any more
        /*HashSet acl_res_ids = new HashSet(mCreatedAclIds); //make a copy because of the remove()
        for (Iterator acliter = acl_res_ids.iterator(); acliter.hasNext();) {
            String resourceid = (String)acliter.next();
            URMAcl urm_acl = mAclAdministrator.getAcl(resourceid);
            if (urm_acl != null && mAdministrator.isRole(urm_acl.getOwner().getName())) {
                assertTrue("After the synchronization there is still an ACE in the ACL with ID '"+resourceid+"'.",
                        urm_acl.getAllAces() == null || urm_acl.getAllAces().size() == 0);
                continue;
            }
            assertNull("After the synchronization there is still an ACL with the ID '"+resourceid+"'.",
                        urm_acl);
            mCreatedAclIds.remove(resourceid);
        }*/
    }

    public void test14_ssxFinalizers() throws Exception {
        URMAuthenticator a = URMAuthenticator.newInstance(mTestConfigurator.getRootConfigurator());
        URMPrincipal p = a.authenticateUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
        a = URMAuthenticator.newInstance(mTestConfigurator.getRootConfigurator());
        p = null;
        p = a.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        p = a.authenticateUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
        a = URMAuthenticator.newInstance(mTestConfigurator.getRootConfigurator());
        a = URMAuthenticator.newInstance(mTestConfigurator.getRootConfigurator());
        p = null;
        p = a.authenticateUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
        a = URMAuthenticator.newInstance(mTestConfigurator.getRootConfigurator());
        p = null;
        p = a.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
    }
    
    public void test15_deleteUserSync() throws Exception {
        String err, usr_name = "URM_junit_test_delus";
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        URMUser urm_usr = mAdministrator.getUser(usr_name, mDomain);
        assertNotNull("Cannot get the just created '"+usr_name+"' (1.) back("+err+").", urm_usr);
        // set metaproperties from config
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
            // set the properties from the 'User2' node
        URMConfigurator userprops = propconf.getSubConfigurator("User2");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User2' test properties from the config file.", sprops);
        
        urm_usr.setProperties(sprops);
        
        Properties gprops = urm_usr.getProperties();
        assertNotNull("Cannot get the 'User2' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'User1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // create acl for the user
            // resource 1 with an ACE
        String resourceid1 = null;
        try { err = null;
            if (mAclAdministrator.getAcl(TEST_RESOURCE_ID1) != null)
                resourceid1 = TEST_RESOURCE_ID1;
            else
                resourceid1 = mAclAdministrator.createAcl(TEST_RESOURCE_ID1, urm_usr, false);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Creating ACL for '"+urm_usr.getName()+"' returned null resource id("+err+").",
                        resourceid1);
        assertNotNull("Cannot get the just created ACL for '"+urm_usr.getName()+"' back("+err+").",
                        mAclAdministrator.getAcl(resourceid1));
        if (!mCreatedAclIds.contains(resourceid1))
            mCreatedAclIds.add(resourceid1);
        URMAcl urm_acl = mAclAdministrator.getAcl(resourceid1);
        assertNotNull("Cannot get the just created ACL for '"+urm_usr.getName()+"' back("+err+").",
                        urm_acl);
            
        urm_acl.defineAce(URMConstants.GRANT, urm_usr, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_usr.getName()+"' (in ACL of '"+
                        urm_usr.getName()+"') back("+err+").", urm_acl.getAce(0));
            // resource 2 with two ACEs
        String resourceid2 = null;
        try { err = null; resourceid2 = null;
            if (mAclAdministrator.getAcl(TEST_RESOURCE_ID2) != null)
                resourceid2 = TEST_RESOURCE_ID2;
            else
                resourceid2 = mAclAdministrator.createAcl(TEST_RESOURCE_ID2, urm_usr, false);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Creating ACL for '"+urm_usr.getName()+"' returned null resource id("+err+").",
                        resourceid2);
        urm_acl = mAclAdministrator.getAcl(resourceid2);
        assertNotNull("Cannot get the just created ACL for '"+urm_usr.getName()+"' back("+err+").",
                        urm_acl);
        if (!mCreatedAclIds.contains(resourceid2))
            mCreatedAclIds.add(resourceid2);
        urm_acl.defineAce(URMConstants.GRANT, urm_usr, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_usr.getName()+"' (in ACL of '"+
                        urm_usr.getName()+"') back("+err+").", urm_acl.getAce(0));
        urm_acl.defineAce(URMConstants.DENY, urm_usr, mAclAdministrator.getAction(TEST_ACTION2), 1, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_usr.getName()+"' (in ACL of '"+
                        urm_usr.getName()+"') back("+err+").", urm_acl.getAce(1));
                        
        // create role mapping
        String role_name = "URM_junit_delusr_sync_test_role1";
        try { err = null;
            if (!mAdministrator.isRole(role_name))
                mAdministrator.createRole(role_name);
            mCreatedRoleNames.add(role_name);
            if (!URMRoleGuestImpl.contains(urm_usr.getRoleMapping(), role_name))
                urm_usr.linkToRole(role_name);
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("Cannot creatre the '"+role_name+"' ("+err+").", err);


        // delete user ...
        /*try { err = null;
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMDeleteException e) {err = e.getMessage(); }
        assertNotNull("The delete '"+usr_name+"'("+mDomain+") did not fail (it has ACE reference).", err);
        //delete ACEs
        URMAcl acl = mAclAdministrator.getAcl(resourceid1);
        for (int idx = acl.getAllAces().size() - 1; idx >= 0; --idx)
            acl.removeAce(idx);
        acl = mAclAdministrator.getAcl(resourceid2);
        for (int idx = acl.getAllAces().size() - 1; idx >= 0; --idx)
            acl.removeAce(idx);*/
        try {
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain); }
        assertFalse("The delete '"+usr_name+"' failed ("+err+").",
                        mAdministrator.isUser(usr_name, mDomain));
        // ... and create user again immidiately
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+usr_name+"' (2.) back("+err+").",
                        mAdministrator.getUser(usr_name, mDomain));
        mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
        
        assertTrue("Cannot get the added users ("+TEST_USER1+" and "+TEST_USER2+") back (1.).",
                containsValues(mAdministrator.getAllUsers(mDomain), TEST_USER1, TEST_USER2));
        assertTrue("Cannot get the added user '"+usr_name+"' back.",
                containsValue(mAdministrator.getAllUsers(mDomain), usr_name));
                
        // check whether the ...
        urm_usr = mAdministrator.getUser(usr_name, mDomain);
        assertNotNull("Cannot get the just created '"+usr_name+"' (1.) back("+err+").", urm_usr);

            // check metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableUserPropKeys();
        gprops = urm_usr.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+usr_name+" still exists (user had been deleted and recreated)", gprops.containsKey(propkey));
        }
            // check role mapping
        assertTrue("A role mapping of "+usr_name+" still exists (user had been deleted and recreated)",
                urm_usr.getRoleMapping() == null || urm_usr.getRoleMapping().size() == 0);
            // check ACEs
        urm_acl = mAclAdministrator.getAcl(resourceid1);
        List aces = urm_acl.getAllAces();
        for (Iterator iter = aces.iterator(); iter.hasNext(); )
            assertFalse("'"+usr_name+"' still has an ACE reference (user had been deleted and recreated)",
                            ((URMAce)iter.next()).getSubject().getName().equals(usr_name));
        urm_acl = mAclAdministrator.getAcl(resourceid2);
        aces = urm_acl.getAllAces();
        for (Iterator iter = aces.iterator(); iter.hasNext(); )
            assertFalse("'"+usr_name+"' still has an ACE reference (user had been deleted and recreated)",
                            ((URMAce)iter.next()).getSubject().getName().equals(usr_name));

        // delete user
        try {
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertTrue("The delete '"+usr_name+"' failed.",
                        !mAdministrator.isUser(usr_name, mDomain));
        mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
        
        assertTrue("Cannot get the added users ("+TEST_USER1+" and "+TEST_USER2+") back (2.).",
                containsValues(mAdministrator.getAllUsers(mDomain), TEST_USER1, TEST_USER2));
        assertFalse("The removed user '"+usr_name+"' is still among all users.",
                containsValue(mAdministrator.getAllUsers(mDomain), usr_name));
    }
    
    public void test16_deleteGroupSync() throws Exception {
        String err, grp_name = "URM_junit_test_delgr";
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        URMGroup urm_grp = mAdministrator.getGroup(grp_name, mDomain);
        assertNotNull("Cannot get the just created '"+grp_name+"' (1.) back("+err+").", urm_grp);
        // set metaproperties from config
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
            // set the properties from the 'Group2' node
        URMConfigurator groupprops = propconf.getSubConfigurator("Group2");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group2' test properties from the config file.", sprops);
        
        urm_grp.setProperties(sprops);
        
        Properties gprops = urm_grp.getProperties();
        assertNotNull("Cannot get the 'Group2' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'Group1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // create acl for the group
            // resource 1 with an ACE
        String resourceid1 = null;
        try { err = null;
            if (mAclAdministrator.getAcl(TEST_RESOURCE_ID1) != null)
                resourceid1 = TEST_RESOURCE_ID1;
            else
                resourceid1 = mAclAdministrator.createAcl(TEST_RESOURCE_ID1, urm_grp, false);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Creating ACL for '"+urm_grp.getName()+"' returned null resource id("+err+").",
                        resourceid1);
        assertNotNull("Cannot get the just created ACL for '"+urm_grp.getName()+"' back("+err+").",
                        mAclAdministrator.getAcl(resourceid1));
        if (!mCreatedAclIds.contains(resourceid1))
            mCreatedAclIds.add(resourceid1);
        URMAcl urm_acl = mAclAdministrator.getAcl(resourceid1);
        assertNotNull("Cannot get the just created ACL for '"+urm_grp.getName()+"' back("+err+").",
                        urm_acl);
            
        urm_acl.defineAce(URMConstants.GRANT, urm_grp, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_grp.getName()+"' (in ACL of '"+
                        urm_grp.getName()+"') back("+err+").", urm_acl.getAce(0));
            // resource 2 with two ACEs
        String resourceid2 = null;
        try { err = null; resourceid2 = null;
            if (mAclAdministrator.getAcl(TEST_RESOURCE_ID2) != null)
                resourceid2 = TEST_RESOURCE_ID2;
            else
                resourceid2 = mAclAdministrator.createAcl(TEST_RESOURCE_ID2, urm_grp, false);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Creating ACL for '"+urm_grp.getName()+"' returned null resource id("+err+").",
                        resourceid2);
        urm_acl = mAclAdministrator.getAcl(resourceid2);
        assertNotNull("Cannot get the just created ACL for '"+urm_grp.getName()+"' back("+err+").",
                        urm_acl);
        if (!mCreatedAclIds.contains(resourceid2))
            mCreatedAclIds.add(resourceid2);
        urm_acl.defineAce(URMConstants.GRANT, urm_grp, mAclAdministrator.getAction(TEST_ACTION1), 0, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_grp.getName()+"' (in ACL of '"+
                        urm_grp.getName()+"') back("+err+").", urm_acl.getAce(0));
        urm_acl.defineAce(URMConstants.DENY, urm_grp, mAclAdministrator.getAction(TEST_ACTION2), 1, false);
        assertNotNull("Cannot get the just created ACE for '"+urm_grp.getName()+"' (in ACL of '"+
                        urm_grp.getName()+"') back("+err+").", urm_acl.getAce(1));
                        
        // create role mapping
        String role_name = "URM_junit_delgrp_sync_test_role1";
        try { err = null;
            if (!mAdministrator.isRole(role_name))
                mAdministrator.createRole(role_name);
            mCreatedRoleNames.add(role_name);
            if (!URMRoleGuestImpl.contains(urm_grp.getRoleMapping(), role_name))
                urm_grp.linkToRole(role_name);
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("Cannot creatre the '"+role_name+"' ("+err+").", err);


        // delete group ...
        /*try { err = null;
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMDeleteException e) {err = e.getMessage(); }
        assertNotNull("The delete '"+grp_name+"'("+mDomain+") did not fail (it has ACE reference).", err);
        //delete ACEs
        URMAcl acl = mAclAdministrator.getAcl(resourceid1);
        for (int idx = acl.getAllAces().size() - 1; idx >= 0; --idx)
            acl.removeAce(idx);
        acl = mAclAdministrator.getAcl(resourceid2);
        for (int idx = acl.getAllAces().size() - 1; idx >= 0; --idx)
            acl.removeAce(idx);*/
        try {
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain); }
        assertFalse("The delete '"+grp_name+"' failed ("+err+").",
                        mAdministrator.isGroup(grp_name, mDomain));
        // ... and create group again immidiately
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+grp_name+"' (2.) back("+err+").",
                        mAdministrator.getGroup(grp_name, mDomain));
        mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);
        
        assertTrue("Cannot get the added groups ("+TEST_GROUP1+" and "+TEST_GROUP2+") back (1.).",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP1, TEST_GROUP2));
        assertTrue("Cannot get the added group '"+grp_name+"' back.",
                containsValue(mAdministrator.getAllGroups(mDomain), grp_name));
                
        // check whether the ...
        urm_grp = mAdministrator.getGroup(grp_name, mDomain);
        assertNotNull("Cannot get the just created '"+grp_name+"' (1.) back("+err+").", urm_grp);

            // check metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableGroupPropKeys();
        gprops = urm_grp.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+grp_name+" still exists (group had been deleted and recreated)", gprops.containsKey(propkey));
        }
            // check role mapping
        assertTrue("A role mapping of "+grp_name+" still exists (group had been deleted and recreated)",
                urm_grp.getRoleMapping() == null || urm_grp.getRoleMapping().size() == 0);
            // check ACEs
        urm_acl = mAclAdministrator.getAcl(resourceid1);
        List aces = urm_acl.getAllAces();
        for (Iterator iter = aces.iterator(); iter.hasNext(); )
            assertFalse("'"+grp_name+"' still has an ACE reference (group had been deleted and recreated)",
                            ((URMAce)iter.next()).getSubject().getName().equals(grp_name));
        urm_acl = mAclAdministrator.getAcl(resourceid2);
        aces = urm_acl.getAllAces();
        for (Iterator iter = aces.iterator(); iter.hasNext(); )
            assertFalse("'"+grp_name+"' still has an ACE reference (group had been deleted and recreated)",
                            ((URMAce)iter.next()).getSubject().getName().equals(grp_name));
        
        // delete group
        try {
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertTrue("The delete '"+grp_name+"' failed.",
                        !mAdministrator.isGroup(grp_name, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
        
        assertTrue("Cannot get the added groups ("+TEST_GROUP1+" and "+TEST_GROUP2+") back (2.).",
                containsValues(mAdministrator.getAllGroups(mDomain), TEST_GROUP1, TEST_GROUP2));
        assertFalse("The removed group '"+grp_name+"' is still among all groups.",
                containsValue(mAdministrator.getAllGroups(mDomain), grp_name));
    }
    
    /** Implemented test cases:
     *
     *  URMAdministrator_0020
     */
    public void test17_i18nUserGroup() throws Exception {
        // create user, group
        String err, usr_name = "URM_junit_test_�����";
        try { err = null;
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+usr_name+"' (1.) back("+err+").",
                        mAdministrator.getUser(usr_name, mDomain));
        mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
        String grp_name = "URM_junit_test_�������";
        try { err = null;
            mAdministrator.createGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("Cannot get the just created '"+grp_name+"' back("+err+").",
                        mAdministrator.getGroup(grp_name, mDomain));
        mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);

        // add user to group
        URMGroup grp1 = mAdministrator.getGroup(grp_name, mDomain);
        assertNotNull("Cannot get the '"+grp_name+"' back.", grp1);
        assertTrue("The '"+grp_name+"' has already a member (nothing was added).",
                        grp1.getGroupMembers() == null || grp1.getGroupMembers().size() <= 0);
        URMUser usr1 = mAdministrator.getUser(usr_name, mDomain);
        assertNotNull("Cannot get the '"+usr_name+"' back.", usr1);
/*zsa*/        assertTrue("The '"+usr_name+"' is already a member in a group (nothing was added).",
/*zsa*/                        usr1.getGroups() == null || usr1.getGroups().size() <= 0);

        grp1.addUserToGroup(usr_name, mDomain);
        assertTrue("The '"+grp_name+"' does not contain '"+usr_name+"'.",
                    containsValue(grp1.getGroupMembers(), usr_name));
        assertTrue("The '"+usr_name+"' is not member in '"+grp_name+"'.",
                    containsValue(usr1.getGroups(), grp_name));

        // remove user from group
        grp1.removeUserFromGroup(usr_name, mDomain);
        grp1 = mAdministrator.getGroup(grp_name, mDomain);
        assertNotNull("Cannot get the '"+grp_name+"' back.", grp1);
        assertTrue("The '"+grp_name+"' has already a member (everything is removed).",
                        grp1.getGroupMembers() == null || grp1.getGroupMembers().size() <= 0);
        usr1 = mAdministrator.getUser(usr_name, mDomain);
        assertNotNull("Cannot get the '"+usr_name+"' back.", usr1);
/*zsa*/        assertTrue("The '"+usr_name+"' is already a member in a group (everything is removed).",
/*zsa*/                        usr1.getGroups() == null || usr1.getGroups().size() <= 0);
                        
        // delete user, group
        try {
            mAdministrator.deleteGroup(grp_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain); }
        assertTrue("The delete '"+grp_name+"' failed.",
                        !mAdministrator.isGroup(grp_name, mDomain));
        mCreatedGroupNames.removeLast(); mCreatedGroupDomains.removeLast();
        try {
            mAdministrator.deleteUser(usr_name, mDomain);
        } catch (URMException e) { err = e.getMessage(); mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain); }
        assertFalse("The delete '"+usr_name+"' failed ("+err+").",
                        mAdministrator.isUser(usr_name, mDomain));
        mCreatedUserNames.removeLast(); mCreatedUserDomains.removeLast();
    }
    
    public void test18_i18nUserProperties() throws Exception {
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);

        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
        // set the properties from the 'User1' node
        URMConfigurator userprops = propconf.getSubConfigurator("User1");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User1' test properties from the config file.", sprops);
        
        String prop_prefix = "���������?�?��?�?��_";
        for (Enumeration en = sprops.propertyNames(); en.hasMoreElements(); ) {
            String key = (String)en.nextElement();
            String val =  prop_prefix + sprops.getProperty(key);
            sprops.setProperty(key, val);
        }
        
        usr1.setProperties(sprops);
        
        Properties gprops = usr1.getProperties();
        assertNotNull("Cannot get the 'User1' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'User1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // set the properties from the 'User2' node
        userprops = propconf.getSubConfigurator("User2");
        sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User2' test properties from the config file.", sprops);
        
        usr1.setProperties(sprops);
        
        gprops = usr1.getProperties();
        assertNotNull("Cannot get the 'User2' test properties back after setting them.", gprops);
        counter = 0;
        keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'User2' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' from the 'User2' test properties, getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
                        
        // clear metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableUserPropKeys();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();)
            clrprops.setProperty((String)iter.next(), "");
        usr1.setProperties(clrprops);
        gprops = usr1.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+TEST_USER1+" still exists (it had been deleted)", gprops.containsKey(propkey));
        }
    }
    
    public void test19_i18nGroupProperties() throws Exception {
        URMGroup grp1 = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP1+"' back.", grp1);

        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
        // set the properties from the 'Group1' node
        URMConfigurator groupprops = propconf.getSubConfigurator("Group1");
        Properties sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group1' test properties from the config file.", sprops);
        
        String prop_prefix = "���������?�?��?�?��_";
        for (Enumeration en = sprops.propertyNames(); en.hasMoreElements(); ) {
            String key = (String)en.nextElement();
            String val =  prop_prefix + sprops.getProperty(key);
            sprops.setProperty(key, val);
        }
        
        grp1.setProperties(sprops);
        
        Properties gprops = grp1.getProperties();
        assertNotNull("Cannot get the 'Group1' test properties back after setting them.", gprops);
        int counter = 0;
        Enumeration keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'Group1' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
        
        // set the properties from the 'Group2' node
        groupprops = propconf.getSubConfigurator("Group2");
        sprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group2' test properties from the config file.", sprops);
        
        grp1.setProperties(sprops);
        
        gprops = grp1.getProperties();
        assertNotNull("Cannot get the 'Group2' test properties back after setting them.", gprops);
        counter = 0;
        keys = gprops.keys();
        while (keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            String val = gprops.getProperty(key);
            assertNotNull("Cannot get the '"+key+"' from the 'Group2' test properties back after setting them.", val);
            String sval = sprops.getProperty(key);
            if (sval != null) {
                assertEquals("After setting '"+sval+"' from the 'Group2' test properties, getting still '"+val+"' back for the '"
                            +key+"' property.", val, sval);
                counter++;
            }
        }
        if (counter != sprops.size())
            assertFalse("Cannot get the "+getMissingKeys(sprops.keys(), gprops)+
                        " property(ies) back(they were set).", true);
                        
        // clear metadata properties
        URMMetadataPropsSpiTamino metaspi = new URMMetadataPropsSpiTamino(
            mTestConfigurator.getSubConfigurator("/Authenticator/Administrator/UserDatabase/MetadataProperties"), this);
        Properties clrprops = new Properties();
        Set metapropnames = metaspi.getWriteableGroupPropKeys();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();)
            clrprops.setProperty((String)iter.next(), "");
        grp1.setProperties(clrprops);
        gprops = grp1.getProperties();
        for (Iterator iter = metapropnames.iterator();iter.hasNext();) {
            String propkey = (String)iter.next();
            assertFalse("The property '"+propkey+"' of "+TEST_GROUP1+" still exists (it had been deleted)", gprops.containsKey(propkey));
        }
    }
    
    /** Implemented test cases:
     *
     *  URMAdministrator_0019
     */
    public void test20_invalidUserGroupName() throws Exception {
        String err = null, usr_name = "URM_junit_test_u;\"$", grp_name = "URM_junit_test_g;\"$";
        try {
            mAdministrator.createUser(usr_name, usr_name.toCharArray(), mDomain);
            mCreatedUserNames.add(usr_name); mCreatedUserDomains.add(mDomain);
        } catch (Throwable e) { err = e.getMessage(); }
        assertNotNull("Creating '"+usr_name+"' user did not throw an exception.", err);

        try {
            mAdministrator.createGroup(grp_name, mDomain);
            mCreatedGroupNames.add(grp_name); mCreatedGroupDomains.add(mDomain);
        } catch (Throwable e) { err = e.getMessage(); }
        assertNotNull("Creating '"+grp_name+"' group did not throw an exception.", err);
    }
    
    /** Implemented test cases:
     *
     *
     */
    public void test21_invalidPassword() throws Exception {
        /*
        char[] newpwd = "newpassword".toCharArray();
        String errmsg = null;
        
        URMUser usr1 = mAdministrator.getUser(TEST_USER1, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER1+"' back.", usr1);
        
        // changing password to 'null' with admin
        try { errmsg = null;
            usr1.setPassword(TEST_USER1.toCharArray(), null);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Changing password to 'null' with admin was successful (giving the old one).", errmsg);
            // check with the old (wrong) password
        URMPrincipal princ1 = null;
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, null, mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user can be authenticate with 'null' password.", princ1);
            // check with the new (correct) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user with the correct password("+errmsg+").", princ1);
        
        // changing password to '' with admin
        try { errmsg = null;
            usr1.setPassword(TEST_USER1.toCharArray(), "".toCharArray());
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Changing password to '' with admin was successful (giving the old one).", errmsg);
            // check with the old (wrong) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, "".toCharArray(), mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user can be authenticate with '' password.", princ1);
            // check with the new (correct) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user with the correct password("+errmsg+").", princ1);
        
        // changing password to ' ' with admin
        try { errmsg = null;
            usr1.setPassword(TEST_USER1.toCharArray(), " ".toCharArray());
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Changing password to ' ' with admin was successful (giving the old one).", errmsg);
            // check with the old (wrong) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, " ".toCharArray(), mDomain);
        } catch (URMAuthenticatorException e) { errmsg = e.getMessage(); }
        assertNull("The '"+TEST_USER1+"' user can be authenticate with ' ' password.", princ1);
            // check with the new (correct) password
        try { errmsg = null;
            princ1 = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER1+"' user with the correct password("+errmsg+").", princ1);
        */
    }
    
    public void test22_denyCache() throws Exception {
        int _try_limit = 60;
        for (int i = 0; i < _try_limit; ++i) {
            try {
                mAuthenticator.authenticateUser(TEST_USER1, "wrong_pwd".toCharArray(), mDomain);
            } catch (URMException e) {}
        }
        URMPrincipal princ = null;
        try {
            princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) {}
        assertNull("After " + _try_limit + " checks with wrong password the user is still not"+
                        "in the deny cache (1.).", princ);
        String err = null;
        try {
            mAdministrator.getUser(TEST_USER1, mDomain).setPassword(TEST_USER1.toCharArray(),
                                                                "new_password".toCharArray());
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("After " + _try_limit + " checks with wrong password "+
                        "the admin change password failed ("+err+") (1.).", err);
        try { err = null;
            princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("After " + _try_limit + " checks with wrong password and a successful admin password "+
                        "change the authentication is still possible with the old password ("+err+") (1.).", err);
        try { err = null;
            princ = mAuthenticator.authenticateUser(TEST_USER1, "new_password".toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("After " + _try_limit + " checks with wrong password and a successful "+
                        "admin password change the authentication failed ("+err+") (1.).", err);
        // and again but now without giving the old password by change password
        for (int i = 0; i < _try_limit; ++i) {
            try {
                mAuthenticator.authenticateUser(TEST_USER1, "wrong_pwd".toCharArray(), mDomain);
            } catch (URMException e) {}
        }
        try { princ = null;
            princ = mAuthenticator.authenticateUser(TEST_USER1, "new_password".toCharArray(), mDomain);
        } catch (URMException e) {}
        assertNull("After " + _try_limit + " checks with wrong password the user is still not"+
                        "in the deny cache (2.).", princ);
        try { err = null;
            mAdministrator.getUser(TEST_USER1, mDomain).setPassword("new_password".toCharArray(), TEST_USER1.toCharArray());
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("After " + _try_limit + " checks with wrong password "+
                        "the admin change password failed ("+err+") (2.).", err);
        try { err = null;
            princ = mAuthenticator.authenticateUser(TEST_USER1, "new_password".toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNotNull("After " + _try_limit + " checks with wrong password and a successful admin password "+
                        "change the authentication is still possible with the old password ("+err+") (2.).", err);
        try { err = null;
            princ = mAuthenticator.authenticateUser(TEST_USER1, TEST_USER1.toCharArray(), mDomain);
        } catch (URMException e) { err = e.getMessage(); }
        assertNull("After " + _try_limit + " checks with wrong password and a successful "+
                        "admin password change the authentication failed ("+err+") (2.).", err);
        
    }
    
    /** Implemented test cases:
     *
     *  URMUser_0043, URMUser_0044, URMUser_0045
     */
    public void test23_setGetUserAggregatedProperty() throws Exception {
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr2);
        URMGroup grp2 = mAdministrator.getGroup(TEST_GROUP2, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP2+"' back.", grp2);
        URMGroup grp3 = mAdministrator.getGroup(TEST_GROUP3, mDomain);
        assertNotNull("Cannot get the '"+TEST_GROUP3+"' back.", grp3);

        // prepare GUEST role
        URMPrincipal princ1 = null;
        String errmsg = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER2+"' user ("+errmsg+").", princ1);
        if (!URMRoleImpl.contains(usr2.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr2.linkToRole(URMConstants.URM_GUEST_ROLE);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        
        // set metaproperties from config
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
            // set the properties from the 'User2' node
        URMConfigurator userprops = propconf.getSubConfigurator("User2");
        Properties usprops = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                userprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'User2' test properties from the config file.", usprops);
        URMConfigurator groupprops = propconf.getSubConfigurator("Group2");
        Properties gsprops2 = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        groupprops = propconf.getSubConfigurator("Group1");
        Properties gsprops1 = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group1' test properties from the config file.", gsprops1);
        // remove not common props
        for (Iterator iter = (new HashSet(usprops.keySet())).iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            if (gsprops2.getProperty(key) == null)
                usprops.remove(key);
            else if (gsprops1.getProperty(key) == null)
                usprops.remove(key);
        }
        for (Iterator iter = (new HashSet(gsprops2.keySet())).iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            if (usprops.getProperty(key) == null) {
                gsprops2.remove(key);
                gsprops1.remove(key);
            }
        }
        // ADDING
        // add user to group2 and check again
        grp2.addUserToGroup(TEST_USER2, mDomain);
        // set group props with GUEST role
        try { errmsg = null;
            princ1.getURMAdministrator().getGroup(TEST_GROUP2, mDomain).setProperties(usprops);
            assertTrue("Setting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        // set group props with ADMIN role
        grp2.setProperties(gsprops2);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' user property back.", uaprops.size() < 1);
            assertFalse("There are more than one aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 1);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        gsprops2.getProperty(key)+"' but it was set to '"+uaprops.toArray()[0]+"'.",
                        gsprops2.getProperty(key).equals(uaprops.toArray()[0]));
        }
        // add user to group3 and check again
        grp3.addUserToGroup(TEST_USER2, mDomain);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' user property back.", uaprops.size() < 1);
            assertFalse("There are more than one aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 1);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        gsprops2.getProperty(key)+"' but it was set to '"+uaprops.toArray()[0]+"'.",
                        gsprops2.getProperty(key).equals(uaprops.toArray()[0]));
        }
        grp3.setProperties(gsprops1);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated user property of the '"+key+"' back.", uaprops.size() < 2);
            assertFalse("There are more than two aggregated user property for '"+key+
                            "' but just two was set.", uaprops.size() > 2);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        uaprops.toArray()[0]+"' and '"+uaprops.toArray()[1]+"' but it was set to '"+
                        gsprops1.getProperty(key)+"' and '"+gsprops2.getProperty(key)+"'.",
                            (gsprops1.getProperty(key).equals(uaprops.toArray()[0]) &&
                                gsprops2.getProperty(key).equals(uaprops.toArray()[1])) ||
                            (gsprops1.getProperty(key).equals(uaprops.toArray()[1]) &&
                                gsprops2.getProperty(key).equals(uaprops.toArray()[0])));
        }
        // set user props with GUEST role
        try { errmsg = null;
            princ1.getURMAdministrator().getUser(TEST_USER2, mDomain).setProperties(usprops);
            assertTrue("Setting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        if (URMRoleImpl.contains(usr2.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr2.deleteRoleLink(URMConstants.URM_GUEST_ROLE);
        // set user props with ADMIN role
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated user property of the '"+key+"' back.", uaprops.size() < 2);
            assertFalse("There are more than two aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 2);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        uaprops.toArray()[0]+"' and '"+uaprops.toArray()[1]+"' but it was set to '"+
                        gsprops1.getProperty(key)+"' and '"+gsprops2.getProperty(key)+"'.",
                            (gsprops1.getProperty(key).equals(uaprops.toArray()[0]) &&
                                gsprops2.getProperty(key).equals(uaprops.toArray()[1])) ||
                            (gsprops1.getProperty(key).equals(uaprops.toArray()[1]) &&
                                gsprops2.getProperty(key).equals(uaprops.toArray()[0])));
        }
        usr2.setProperties(usprops);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the three aggregated user property of the '"+key+"' back.", uaprops.size() < 3);
            assertFalse("There are more than three aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 3);
            int counter = 0;
            for (int i = 0; i < 3; ++i)
                if (usprops.getProperty(key).equals(uaprops.toArray()[i])) ++counter;
                else if (gsprops2.getProperty(key).equals(uaprops.toArray()[i])) ++counter;
                else if (gsprops1.getProperty(key).equals(uaprops.toArray()[i])) ++counter;
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        uaprops.toArray()[0]+"', '"+uaprops.toArray()[1]+"' and '"+uaprops.toArray()[2]+"'" +
                        " but it was set to '"+usprops.getProperty(key)+"', '"+gsprops2.getProperty(key)+
                        "' and '"+gsprops1.getProperty(key)+"'.", counter == 3);
        }
        // REMOVING
        // removes user to group2 and check again
        grp2.removeUserFromGroup(TEST_USER2, mDomain);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated user property of the '"+key+"' back.", uaprops.size() < 2);
            assertFalse("There are more than two aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 2);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        uaprops.toArray()[0]+"' and '"+uaprops.toArray()[1]+"' but it was set to '"+
                        usprops.getProperty(key)+"' and '"+gsprops1.getProperty(key)+"'.",
                            (usprops.getProperty(key).equals(uaprops.toArray()[0]) &&
                                gsprops1.getProperty(key).equals(uaprops.toArray()[1])) ||
                            (usprops.getProperty(key).equals(uaprops.toArray()[1]) &&
                                gsprops1.getProperty(key).equals(uaprops.toArray()[0])));
        }
        // removes user to group3 and check again
        grp3.removeUserFromGroup(TEST_USER2, mDomain);
        for (Iterator iter = usprops.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set uaprops = usr2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' user property back.", uaprops.size() < 1);
            assertFalse("There are more than one aggregated user property for '"+key+
                            "' but just one was set.", uaprops.size() > 1);
            assertTrue("The aggregated user property '"+key+"' has the value '"+
                        usprops.getProperty(key)+"' but it was set to '"+uaprops.toArray()[0]+"'.",
                        usprops.getProperty(key).equals(uaprops.toArray()[0]));
        }
    }
    
    /** Implemented test cases:
     *
     *  URMGroup_0020, URMGroup_0021, URMGroup_0022
     */
    public void test24_setGetGroupAggregatedProperty() throws Exception {
        URMUser usr2 = mAdministrator.getUser(TEST_USER2, mDomain);
        assertNotNull("Cannot get the '"+TEST_USER2+"' back.", usr2);

        String aggr_grp1 = "URM_junit_test_aggr_props_grp1";
        String aggr_grp2 = "URM_junit_test_aggr_props_grp2";
        String aggr_grp3 = "URM_junit_test_aggr_props_grp3";
        mAdministrator.createGroup(aggr_grp1, mDomain); mCreatedGroupNames.add(aggr_grp1); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(aggr_grp2, mDomain); mCreatedGroupNames.add(aggr_grp2); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(aggr_grp3, mDomain); mCreatedGroupNames.add(aggr_grp3); mCreatedGroupDomains.add(mDomain);
        URMGroup grp1 = mAdministrator.getGroup(aggr_grp1, mDomain);
        assertNotNull("Cannot get the '"+aggr_grp1+"' back.", grp1);
        URMGroup grp2 = mAdministrator.getGroup(aggr_grp2, mDomain);
        assertNotNull("Cannot get the '"+aggr_grp2+"' back.", grp2);
        URMGroup grp3 = mAdministrator.getGroup(aggr_grp3, mDomain);
        assertNotNull("Cannot get the '"+aggr_grp3+"' back.", grp3);

        // prepare GUEST role
        URMPrincipal princ1 = null;
        String errmsg = null;
        try {
            princ1 = mAuthenticator.authenticateUser(TEST_USER2, TEST_USER2.toCharArray(), mDomain);
        } catch (URMException e) { errmsg = e.getMessage(); }
        assertNotNull("Cannot authenticate the '"+TEST_USER2+"' user ("+errmsg+").", princ1);
        if (!URMRoleImpl.contains(grp2.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr2.linkToRole(URMConstants.URM_GUEST_ROLE);
        princ1.setActiveRole(URMConstants.URM_GUEST_ROLE);
        
        // set metaproperties from config
        URMConfigurator propconf = mTestConfigurator.getSubConfigurator(
                                    "Administrator/UserDatabase/TestProperties");
            // set the properties from the 'Group3' node
        URMConfigurator groupprops = propconf.getSubConfigurator("Group3");
        Properties gsprops3 = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group3' test properties from the config file.", gsprops3);
        groupprops = propconf.getSubConfigurator("Group2");
        Properties gsprops2 = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        groupprops = propconf.getSubConfigurator("Group1");
        Properties gsprops1 = URMConfiguratorUtil.getNodeAttrsAsProperties(
                                groupprops, "Property", "name", "value");
        assertNotNull("Cannot read the 'Group1' test properties from the config file.", gsprops1);
        // remove not common props
        for (Iterator iter = (new HashSet(gsprops3.keySet())).iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            if (gsprops2.getProperty(key) == null)
                gsprops3.remove(key);
            else if (gsprops1.getProperty(key) == null)
                gsprops3.remove(key);
        }
        for (Iterator iter = (new HashSet(gsprops2.keySet())).iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            if (gsprops3.getProperty(key) == null) {
                gsprops2.remove(key);
                gsprops1.remove(key);
            }
        }
        // ADDING
        // add group to group2 and check again
        grp1.addGroupToGroup(aggr_grp2, mDomain);
        // set group props with GUEST role
        try { errmsg = null;
            princ1.getURMAdministrator().getGroup(aggr_grp1, mDomain).setProperties(gsprops3);
            assertTrue("Setting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        // set group props with ADMIN role
        grp1.setProperties(gsprops1);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' group property back.", rgaprops.size() < 1);
            assertFalse("There are more than one aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 1);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        gsprops1.getProperty(key)+"' but it was set to '"+rgaprops.toArray()[0]+"'.",
                        gsprops1.getProperty(key).equals(rgaprops.toArray()[0]));
        }
        // add group to group3 and check again
        grp3.addGroupToGroup(aggr_grp2, mDomain);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' group property back.", rgaprops.size() < 1);
            assertFalse("There are more than one aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 1);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        gsprops1.getProperty(key)+"' but it was set to '"+rgaprops.toArray()[0]+"'.",
                        gsprops1.getProperty(key).equals(rgaprops.toArray()[0]));
        }
        grp3.setProperties(gsprops3);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated group property of the '"+key+"' back.", rgaprops.size() < 2);
            assertFalse("There are more than two aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 2);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        rgaprops.toArray()[0]+"' and '"+rgaprops.toArray()[1]+"' but it was set to '"+
                        gsprops1.getProperty(key)+"' and '"+gsprops3.getProperty(key)+"'.",
                            (gsprops1.getProperty(key).equals(rgaprops.toArray()[0]) &&
                                gsprops3.getProperty(key).equals(rgaprops.toArray()[1])) ||
                            (gsprops1.getProperty(key).equals(rgaprops.toArray()[1]) &&
                                gsprops3.getProperty(key).equals(rgaprops.toArray()[0])));
        }
        // set group props with GUEST role
        try { errmsg = null;
            princ1.getURMAdministrator().getGroup(aggr_grp2, mDomain).setProperties(gsprops3);
            assertTrue("Setting group properties with GUEST role did not throw an exception.", false);
        } catch (URMException e) { errmsg = e.getMessage(); }
        if (URMRoleImpl.contains(usr2.getRoleMapping(), URMConstants.URM_GUEST_ROLE))
            usr2.deleteRoleLink(URMConstants.URM_GUEST_ROLE);
        // set group props with ADMIN role
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated group property of the '"+key+"' back.", rgaprops.size() < 2);
            assertFalse("There are more than two aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 2);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        rgaprops.toArray()[0]+"' and '"+rgaprops.toArray()[1]+"' but it was set to '"+
                        gsprops1.getProperty(key)+"' and '"+gsprops3.getProperty(key)+"'.",
                            (gsprops1.getProperty(key).equals(rgaprops.toArray()[0]) &&
                                gsprops3.getProperty(key).equals(rgaprops.toArray()[1])) ||
                            (gsprops1.getProperty(key).equals(rgaprops.toArray()[1]) &&
                                gsprops3.getProperty(key).equals(rgaprops.toArray()[0])));
        }
        grp2.setProperties(gsprops2);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the three aggregated group property of the '"+key+"' back.", rgaprops.size() < 3);
            assertFalse("There are more than three aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 3);
            int counter = 0;
            for (int i = 0; i < 3; ++i)
                if (gsprops3.getProperty(key).equals(rgaprops.toArray()[i])) ++counter;
                else if (gsprops2.getProperty(key).equals(rgaprops.toArray()[i])) ++counter;
                else if (gsprops1.getProperty(key).equals(rgaprops.toArray()[i])) ++counter;
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        rgaprops.toArray()[0]+"', '"+rgaprops.toArray()[1]+"' and '"+rgaprops.toArray()[2]+"'" +
                        " but it was set to '"+gsprops3.getProperty(key)+"', '"+gsprops2.getProperty(key)+
                        "' and '"+gsprops1.getProperty(key)+"'.", counter == 3);
        }
        // REMOVING
        // removes group to group2 and check again
        grp1.removeGroupFromGroup(aggr_grp2, mDomain);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the two aggregated group property of the '"+key+"' back.", rgaprops.size() < 2);
            assertFalse("There are more than two aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 2);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        rgaprops.toArray()[0]+"' and '"+rgaprops.toArray()[1]+"' but it was set to '"+
                        gsprops3.getProperty(key)+"' and '"+gsprops2.getProperty(key)+"'.",
                            (gsprops3.getProperty(key).equals(rgaprops.toArray()[0]) &&
                                gsprops2.getProperty(key).equals(rgaprops.toArray()[1])) ||
                            (gsprops3.getProperty(key).equals(rgaprops.toArray()[1]) &&
                                gsprops2.getProperty(key).equals(rgaprops.toArray()[0])));
        }
        // removes group to group3 and check again
        grp3.removeGroupFromGroup(aggr_grp2, mDomain);
        for (Iterator iter = gsprops3.keySet().iterator(); iter.hasNext(); ) {
            String key = (String)iter.next();
            Set rgaprops = grp2.getAggregatedProperty(key);
            assertFalse("Cannot get the '"+key+"' group property back.", rgaprops.size() < 1);
            assertFalse("There are more than one aggregated group property for '"+key+
                            "' but just one was set.", rgaprops.size() > 1);
            assertTrue("The aggregated group property '"+key+"' has the value '"+
                        gsprops2.getProperty(key)+"' but it was set to '"+rgaprops.toArray()[0]+"'.",
                        gsprops2.getProperty(key).equals(rgaprops.toArray()[0]));
        }
    }

    public void test25_allGroupsOfUser() throws Exception {
        String ltest_grp1 = "URM_junit_test_usrallgrps_grp1";
        String ltest_grp2 = "URM_junit_test_usrallgrps_grp2";
        String ltest_grp3 = "URM_junit_test_usrallgrps_grp3";
        String ltest_grp4 = "URM_junit_test_usrallgrps_grp4";
        String ltest_grp5 = "URM_junit_test_usrallgrps_grp5";
        mAdministrator.createGroup(ltest_grp1, mDomain); mCreatedGroupNames.add(ltest_grp1); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp2, mDomain); mCreatedGroupNames.add(ltest_grp2); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp3, mDomain); mCreatedGroupNames.add(ltest_grp3); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp4, mDomain); mCreatedGroupNames.add(ltest_grp4); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp5, mDomain); mCreatedGroupNames.add(ltest_grp5); mCreatedGroupDomains.add(mDomain);
        // adding
        URMUser usr = mAdministrator.getUser(TEST_USER1, mDomain);
        assertTrue("User "+TEST_USER1+" is already member in a group (nothing was added 1.)", usr.getAllGroups().size() == 0);
        mAdministrator.getGroup(ltest_grp1, mDomain).addUserToGroup(TEST_USER1, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in any groups (one was added 1.)", usr.getAllGroups().size() < 1);
        assertFalse("User "+TEST_USER1+" is member in more than one groups (just one was added 1.)", usr.getAllGroups().size() > 1);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" (it was added 1.)",
                        containsValue(usr.getAllGroups(), ltest_grp1));

        mAdministrator.getGroup(ltest_grp2, mDomain).addGroupToGroup(ltest_grp1, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than two groups (two was added)", usr.getAllGroups().size() < 2);
        assertFalse("User "+TEST_USER1+" is member in more than two groups (just two was added)", usr.getAllGroups().size() > 2);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp2+" (it was added)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp2));

        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp2, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than three groups (three was added)", usr.getAllGroups().size() < 3);
        assertFalse("User "+TEST_USER1+" is member in more than three groups (just three was added)", usr.getAllGroups().size() > 3);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp3+" (it was added)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp3));

        mAdministrator.getGroup(ltest_grp4, mDomain).addGroupToGroup(ltest_grp3, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than four groups (four was added 1.)", usr.getAllGroups().size() < 4);
        assertFalse("User "+TEST_USER1+" is member in more than four groups (just four was added 1.)", usr.getAllGroups().size() > 4);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 1.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 1.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));

        mAdministrator.getGroup(ltest_grp5, mDomain).addGroupToGroup(ltest_grp3, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than five groups (five was added 1.)", usr.getAllGroups().size() < 5);
        assertFalse("User "+TEST_USER1+" is member in more than five groups (just five was added 1.)", usr.getAllGroups().size() > 5);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 1.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 1.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 1.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp5));

        // test circular dependencies
        mAdministrator.getGroup(ltest_grp4, mDomain).addGroupToGroup(ltest_grp5, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than five groups (five was added 2.)", usr.getAllGroups().size() < 5);
        assertFalse("User "+TEST_USER1+" is member in more than five groups (just five was added 2.)", usr.getAllGroups().size() > 5);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 2.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 2.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 2.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 3.)", usr.getAllGroups().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 3.)", usr.getAllGroups().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 3.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 3.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 3.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp4, mDomain).removeGroupFromGroup(ltest_grp5, mDomain);
        mAdministrator.getGroup(ltest_grp3, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than five groups (five was added 4.)", usr.getAllGroups().size() < 5);
        assertFalse("User "+TEST_USER1+" is member in more than five groups (just five was added 4.)", usr.getAllGroups().size() > 5);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 4.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 4.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 4.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp3, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp1, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than five groups (five was added 5.)", usr.getAllGroups().size() < 5);
        assertFalse("User "+TEST_USER1+" is member in more than five groups (just five was added 5.)", usr.getAllGroups().size() > 5);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 5.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 5.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp4));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 5.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp5));
        // removing
        mAdministrator.getGroup(ltest_grp1, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp4, mDomain).removeGroupFromGroup(ltest_grp3, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than four groups (four was added 2.)", usr.getAllGroups().size() < 4);
        assertFalse("User "+TEST_USER1+" is member in more than four groups (just four was added 2.)", usr.getAllGroups().size() > 4);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 2.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp1+" (it was added 2.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp1));
        
        mAdministrator.getGroup(ltest_grp2, mDomain).removeGroupFromGroup(ltest_grp1, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in any groups (one was added 2.)", usr.getAllGroups().size() < 1);
        assertFalse("User "+TEST_USER1+" is member in more than one groups (just one was added 2.)", usr.getAllGroups().size() > 1);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" (it was added 2.)",
                        containsValue(usr.getAllGroups(), ltest_grp1));

        mAdministrator.getGroup(ltest_grp2, mDomain).addGroupToGroup(ltest_grp1, mDomain);
        assertFalse("User "+TEST_USER1+" is not member in less than four groups (four was added 3.)", usr.getAllGroups().size() < 4);
        assertFalse("User "+TEST_USER1+" is member in more than four groups (just four was added 3.)", usr.getAllGroups().size() > 4);
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 3.)",
                        containsValues(usr.getAllGroups(), ltest_grp1, ltest_grp3));
        assertTrue("User "+TEST_USER1+" is member in group "+ltest_grp2+" and "+ltest_grp1+" (it was added 3.)",
                        containsValues(usr.getAllGroups(), ltest_grp2, ltest_grp1));
        
        mAdministrator.getGroup(ltest_grp1, mDomain).removeUserFromGroup(TEST_USER1, mDomain);
        assertTrue("User "+TEST_USER1+" is already member in a group (everything was removed)", usr.getAllGroups().size() == 0);
    }
    
    public void test26_allGroupsOfGroup() throws Exception {
        String ltest_grp1 = "URM_junit_test_grpallgrps_grp1";
        String ltest_grp2 = "URM_junit_test_grpallgrps_grp2";
        String ltest_grp3 = "URM_junit_test_grpallgrps_grp3";
        String ltest_grp4 = "URM_junit_test_grpallgrps_grp4";
        String ltest_grp5 = "URM_junit_test_grpallgrps_grp5";
        mAdministrator.createGroup(ltest_grp1, mDomain); mCreatedGroupNames.add(ltest_grp1); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp2, mDomain); mCreatedGroupNames.add(ltest_grp2); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp3, mDomain); mCreatedGroupNames.add(ltest_grp3); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp4, mDomain); mCreatedGroupNames.add(ltest_grp4); mCreatedGroupDomains.add(mDomain);
        mAdministrator.createGroup(ltest_grp5, mDomain); mCreatedGroupNames.add(ltest_grp5); mCreatedGroupDomains.add(mDomain);
        // adding
        URMGroup grp = mAdministrator.getGroup(TEST_GROUP1, mDomain);
        assertTrue("Group "+TEST_GROUP1+" is already member in a group (nothing was added 1.)", grp.getAllGroupsMemberOf().size() == 0);
        mAdministrator.getGroup(ltest_grp1, mDomain).addGroupToGroup(TEST_GROUP1, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in any groups (one was added 1.)", grp.getAllGroupsMemberOf().size() < 1);
        assertFalse("Group "+TEST_GROUP1+" is member in more than one groups (just one was added 1.)", grp.getAllGroupsMemberOf().size() > 1);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" (it was added 1.)",
                        containsValue(grp.getAllGroupsMemberOf(), ltest_grp1));

        mAdministrator.getGroup(ltest_grp2, mDomain).addGroupToGroup(ltest_grp1, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than two groups (two was added)", grp.getAllGroupsMemberOf().size() < 2);
        assertFalse("Group "+TEST_GROUP1+" is member in more than two groups (just two was added)", grp.getAllGroupsMemberOf().size() > 2);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp2+" (it was added)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp2));

        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp2, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than three groups (three was added)", grp.getAllGroupsMemberOf().size() < 3);
        assertFalse("Group "+TEST_GROUP1+" is member in more than three groups (just three was added)", grp.getAllGroupsMemberOf().size() > 3);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp3+" (it was added)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp3));

        mAdministrator.getGroup(ltest_grp4, mDomain).addGroupToGroup(ltest_grp3, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than four groups (four was added 1.)", grp.getAllGroupsMemberOf().size() < 4);
        assertFalse("Group "+TEST_GROUP1+" is member in more than four groups (just four was added 1.)", grp.getAllGroupsMemberOf().size() > 4);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 1.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 1.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));

        mAdministrator.getGroup(ltest_grp5, mDomain).addGroupToGroup(ltest_grp3, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 1.)", grp.getAllGroupsMemberOf().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 1.)", grp.getAllGroupsMemberOf().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 1.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 1.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 1.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp5));

        // test circular dependencies
        mAdministrator.getGroup(ltest_grp4, mDomain).addGroupToGroup(ltest_grp5, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 2.)", grp.getAllGroupsMemberOf().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 2.)", grp.getAllGroupsMemberOf().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 2.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 2.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 2.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 3.)", grp.getAllGroupsMemberOf().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 3.)", grp.getAllGroupsMemberOf().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 3.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 3.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 3.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp4, mDomain).removeGroupFromGroup(ltest_grp5, mDomain);
        mAdministrator.getGroup(ltest_grp3, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp3, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 4.)", grp.getAllGroupsMemberOf().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 4.)", grp.getAllGroupsMemberOf().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 4.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 4.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 4.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp5));

        mAdministrator.getGroup(ltest_grp3, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp1, mDomain).addGroupToGroup(ltest_grp4, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than five groups (five was added 5.)", grp.getAllGroupsMemberOf().size() < 5);
        assertFalse("Group "+TEST_GROUP1+" is member in more than five groups (just five was added 5.)", grp.getAllGroupsMemberOf().size() > 5);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 5.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp4+" (it was added 5.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp4));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp5+" (it was added 5.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp5));
        // removing
        mAdministrator.getGroup(ltest_grp1, mDomain).removeGroupFromGroup(ltest_grp4, mDomain);
        mAdministrator.getGroup(ltest_grp4, mDomain).removeGroupFromGroup(ltest_grp3, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than four groups (four was added 2.)", grp.getAllGroupsMemberOf().size() < 4);
        assertFalse("Group "+TEST_GROUP1+" is member in more than four groups (just four was added 2.)", grp.getAllGroupsMemberOf().size() > 4);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 2.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp1+" (it was added 2.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp1));
        
        mAdministrator.getGroup(ltest_grp2, mDomain).removeGroupFromGroup(ltest_grp1, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in any groups (one was added 2.)", grp.getAllGroupsMemberOf().size() < 1);
        assertFalse("Group "+TEST_GROUP1+" is member in more than one groups (just one was added 2.)", grp.getAllGroupsMemberOf().size() > 1);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" (it was added 2.)",
                        containsValue(grp.getAllGroupsMemberOf(), ltest_grp1));

        mAdministrator.getGroup(ltest_grp2, mDomain).addGroupToGroup(ltest_grp1, mDomain);
        assertFalse("Group "+TEST_GROUP1+" is not member in less than four groups (four was added 3.)", grp.getAllGroupsMemberOf().size() < 4);
        assertFalse("Group "+TEST_GROUP1+" is member in more than four groups (just four was added 3.)", grp.getAllGroupsMemberOf().size() > 4);
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp1+" and "+ltest_grp3+" (it was added 3.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp1, ltest_grp3));
        assertTrue("Group "+TEST_GROUP1+" is member in group "+ltest_grp2+" and "+ltest_grp1+" (it was added 3.)",
                        containsValues(grp.getAllGroupsMemberOf(), ltest_grp2, ltest_grp1));
        
        mAdministrator.getGroup(ltest_grp1, mDomain).removeGroupFromGroup(TEST_GROUP1, mDomain);
        assertTrue("Group "+TEST_GROUP1+" is still member in a group (everything was removed)", grp.getAllGroupsMemberOf().size() == 0);
    }
    
    public void test27_() throws Exception {
    }
    
    public static void main(String[] args) throws Exception {
        String[] testCaseName = { "-noloading", TestURMUserDBManager.class.getName() };
        
        org.apache.log4j.PropertyConfigurator.configure("testsuite/junit/log4j.properties");
        
        junit.textui.TestRunner.main(testCaseName);
    }
}

