/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/testsuite/junit/util/org/apache/slide/util/cli/CommandLineTest.java,v 1.1 2004/03/25 16:18:17 juergen Exp $
 * $Revision: 1.1 $
 * $Date: 2004/03/25 16:18:17 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.slide.util.cli;

import java.util.List;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

public class CommandLineTest extends TestCase {
    private CommandLine cl;
    private Actuals actuals;
    
    public CommandLineTest(String name) {
        super( name );
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        cl = new CommandLine(false);
    }
    
    public void testEmpty() throws Abort {
        checkZero();
        try {
            parse("unkown");
            fail();
        } catch (Abort e) {
            // ok
        }
    }
    
    public void testSimple() throws Abort {
        cl.add(new Formal("name", "literal", "value"));
        checkZero();
        checkOne("literal", "name", "value");
        try {
            parse("unkown");
            fail();
        } catch (Abort e) {
            // ok
        }
    }
    
    public void testDuplicate() throws Abort {
        cl.add(new Formal("name", "literal", "value"));
        parse(new String[] { "literal" });
        try {
            parse("literal", "literal");
            fail();
        } catch (Abort duplicate) {
            // ok
        }
    }
    
    public void testAlternatives() throws Abort {
        cl.addAlternatives("cmd", new String[] { "a", "b" });
        checkZero();
        checkOne("a", "cmd", "a");
        checkOne("b", "cmd", "b");
    }
    
    public void testLiterals() throws Abort {
        cl.addLiterals("name", new String[] { "1", "2" }, "value");
        checkZero();
        checkOne("1", "name", "value");
        checkOne("2", "name", "value");
    }
    
    public void testSwitch() throws Abort {
        cl.addSwitch("trace");
        checkZero();
        checkOne("-trace", "trace", "trace");
        try {
            parse("trace"); // missing "-"
        } catch (Abort e) {
            // ok
        }
    }
    
    public void testOption() throws Abort {
        cl.addOption("store");
        
        checkZero();
        assertNull(actuals.getString("store"));
        
        parse("-store", "foo");
        assertEquals("foo", actuals.getString("store"));
        
        try {
            parse("-store");
            fail();
        } catch (Abort e) {
            // ok -- missing argument
        }
        try {
            parse("store", "foo" );
            fail();
        } catch (Abort e) {
            // ok -- unkown argument - dash missing
        }
    }
    
    public void testForString() throws Abort {
        List lst;
        
        cl = CommandLine.forString(
            "  a b c" +
            "+ x y"
        );
        lst = cl.getFormals();
        assertEquals(2, lst.size());
        assertEquals(new Formal("a", "b", "c"), lst.get(0));
        assertEquals(new Formal("x", "y", null), lst.get(1));
    }
    
    //--
    
    private void checkZero() throws Abort {
        parse();
        assertEquals(0, actuals.names().size());
    }
    
    private void checkOne(String actual, String name, String value) throws Abort {
        parse(actual);
        assertEquals(1, actuals.names().size());
        assertEquals(value, actuals.getString(name));
    }
    
    private void parse() throws Abort {
        parse(new String[] {});
    }
    
    private void parse(String arg) throws Abort {
        parse(new String[] { arg });
    }
    
    private void parse(String arg1, String arg2) throws Abort {
        parse(new String[] { arg1, arg2 });
    }
    
    private void parse(String[] args) throws Abort {
        actuals = cl.parse(args);
    }
    
    
    public static void main( String[] args ) {
        TestRunner.run( suite() );
    }
    
    public static Test suite() {
        return new TestSuite(CommandLineTest.class);
    }
    
}

