package javax.wvcm;
import java.util.List;

/**
 * Represents an activity resource.
 * See section 13 of RFC 3253. 
 */
public interface Activity extends Resource {

   /**
    * Create an activity on the server.
    * 
    * @throws WvcmException Preconditions:
    * <br>(resource-must-be-null): A resource MUST NOT exist at the location of this Resource.
    * <br>(location-ok): The location of this Activity MUST identify a valid location to create this Resource.
    *   A client can determine the location of folders that can contain activities
    *  from the ActivityFolderList property.
    * @throws WvcmException Postconditions:
    * <br>(initialize-resource): A new activity resource exists at the location of this Resource.
    */
   public void doCreateResource() throws WvcmException;

   /**
    * Apply {@link ControllableResource#doCheckin doCheckin}
    * to each member of the ActivityCheckoutList of this {@link Activity}.
    * @param keepCheckedOut passed into each checkin request.
    * @param forkOK passed into each checkin request. 
    * 
    * @throws WvcmException Preconditions:
    * <br>(atomic-activity-checkin): The server MAY fail the request if any of the checked-out resources
    *  in the ActivityCheckoutList of either that activity or any sub-activity of that activity cannot be checked in.
    * 
    * @throws WvcmException Postconditions:
    * <br>(activity-checkin): The server MUST have applied the doCheckin request to each checked-out resource
    *  in the ActivityCheckoutList of both that activity and any sub-activity of that activity.
    */
   public void doCheckin(boolean keepCheckedOut, boolean forkOK) throws WvcmException;

   /**
    * Return a list of {@link ControllableResource} objects that
    * identify all version-controlled resources that are checked out
    * to this {@link Activity}.
    * @throws WvcmException if this {@link Activity} was not created with
    * {@link PropertyNameList.PropertyName#ACTIVITY_CHECKOUT_LIST ACTIVITY_CHECKOUT_LIST} as a wanted property.
    */
   public List getActivityCheckoutList() throws WvcmException;

   /**
    * Return a list of {@link Version} objects that identify versions
    * that were created for this {@link Activity}.
    * Multiple versions of a single version history can be selected by an activity's
    * ActivityVersionList property, but all ActivityVersionList versions
    * from a given version history must be on a single line of descent from 
    * the root version of that version history. 
    * @throws WvcmException if this {@link Activity} was not created with
    * {@link PropertyNameList.PropertyName#ACTIVITY_VERSION_LIST ACTIVITY_VERSION_LIST} as a wanted property.
    */
   public List getActivityVersionList() throws WvcmException;

   /**
    * Return a list of {@link Workspace} objects that identify
    * workspaces that refer to this {@link Activity} in their CurrentActivityList.
    * @throws WvcmException if this {@link Activity} was not created with
    * {@link PropertyNameList.PropertyName#CURRENT_WORKSPACE_LIST CURRENT_WORKSPACE_LIST} as a wanted property.
    * @see Workspace#getCurrentActivityList
    */
   public List getCurrentWorkspaceList() throws WvcmException;

   /**
    * Return a list of {@link Activity} objects.
    * An activity behaves as if its ActivityVersionList is extended by the ActivityVersionList
    * of each activity identified in the SubactivityList.  In particular, the versions in this
    * extended set MUST be on a single line of descent, and when an activity selects a version for merging, 
    * the latest version in this extended set is the one that will be merged.
    * @throws WvcmException if this {@link Activity} was not created with
    * {@link PropertyNameList.PropertyName#SUBACTIVITY_LIST SUBACTIVITY_LIST} as a wanted property.
    * @see #getActivityVersionList
    */
   public List getSubactivityList() throws WvcmException;

   /**
    * Update the SubactivityList property of this {@link Activity}.
    * A server MAY reject attempts to modify the SubactivityList of an activity.  
    * @param subactivityList The list of {@link Activity} objects that will be the
    * new value of the SubactivityList property.
    */
   public void setSubactivityList(List subactivityList);
}
