package javax.wvcm;
import java.util.Iterator;
import java.util.List;

/**
 * Represents a baseline resource.
 * See section 12 of RFC 3253. 
 */
public interface Baseline extends Version {

   /**
    * A description of how the selection of a version
    * from a version history differs between two baselines.
    */
   public interface CompareBaselineReport { }

   /**
    * A version that is referenced
    * by the second baseline but no version from the version
    * history of that version is referenced by the first baseline.
    * @see #doCompareBaselineReport doCompareBaselineReport
    */
   public interface AddedVersion extends CompareBaselineReport {

      /**
       * Return the added version.
       */
      public Version getVersion();
   }

   /**
    * A version that is referenced
    * by the first baseline but no version from the version
    * history of that version is referenced by the second baseline.
    * @see #doCompareBaselineReport doCompareBaselineReport
    */
   public interface DeletedVersion extends CompareBaselineReport {

      /**
       * Return the deleted version.
       */
      public Version getVersion();
   }

   /**
    * A pair of different versions referenced
    * by two baseline from the same version history.
    * @see #doCompareBaselineReport doCompareBaselineReport
    */
   public interface ChangedVersion extends CompareBaselineReport {

      /**
       * Return the version from the first baseline.
       */
      public Version oldVersion();

      /**
       * Return the version from the second baseline.
       */
      public Version newVersion();
   }

   /**
    * Return an iterator of AddedVersion, DeletedVersion, and 
    * ChangedVersion objects, that enumerate the differences between the
    * versions selected by this {@link Baseline} and the baseline argument.
    * @param baseline The baseline being compared to this {@link Baseline}.
    * 
    * @throws WvcmException Preconditions:
    * <br>(baselines-from-same-history): A server MAY require that the baselines being compared
    *  be from the same baseline history.
    */
   public Iterator doCompareBaselineReport(Baseline baseline)
      throws WvcmException;

   /**
    * Return a folder containing a read-only copy of the
    * version-controlled state of the baseline-controlled folder
    * at the time the baseline identified by this {@link Baseline} was created.
    * @throws WvcmException if this {@link Baseline} was not created with
    * {@link PropertyNameList.PropertyName#BASELINE_FOLDER BASELINE_FOLDER} as a wanted property.
    */
   public ControllableFolder getBaselineFolder() throws WvcmException;

   /**
    * Return a list of {@link Baseline} objects that identify 
    * the subbaselines of the baseline identified by this {@link Baseline}.
    * created when this {@link Configuration}
    * @throws WvcmException if this {@link Baseline} was not created with
    * {@link PropertyNameList.PropertyName#SUBBASELINE_LIST SUBBASELINE_LIST} as a wanted property.
    */
   public List getSubbaselineList() throws WvcmException;

}
