package javax.wvcm;

import java.util.Iterator;
import java.util.List;

/**
 * A ControllableFolder is a folder that can be placed under control.
 * <p>
 * A "configuration" is the set of resources consisting of all
 * members of a controllable folder (called the "root folder" for the configuration).</p>
 * <p>
 * When a controllable folder is placed under version control, a version of that folder
 * only captures the named mappings to version controlled members of that folder.
 * In order to version control the entire configuration rooted at a folder,
 * the folder must be placed under "baseline control". </p>
 */
public interface ControllableFolder extends ControllableResource, Folder {

   /**
    * Return an iterator of {@link Resource} objects that contain locally
    * available properties of the resources that are locally known
    * as members of this {@link Folder}.  Locally available information is
    * information that is available on the client host
    * without communicating with the server.
    * @see #doReadMemberList
    * @param wantedPropertyList The list of properties to be retrieved.
    * @param deep Whether to retrieve properties for all members of this {@link Folder}.
    */
   public Iterator readMemberList(
      PropertyNameList wantedPropertyList,
      boolean deep) throws WvcmException;


   /**
    * Put this {@link ControllableFolder} under baseline control.
    * <p>
    * A version-controlled configuration resource is associated with
    * this {@link ControllableFolder}, which allows versions of the configuration rooted at
    * this {@link ControllableFolder} to be created by checking out and checking in the
    * version-controlled configuration resource. </p>
    * <p>
    * A version of a configuration is called a "baseline". </p>
    *
    * @throws WvcmException Preconditions:
    * <br>(controlled-configuration-must-not-exist): The ControlledConfiguration property
    *  of the folder identified by this ControllableFolder MUST NOT exist.
    *
    * @throws WvcmException Postconditions:
    * <br>(create-controlled-configuration): A new version-controlled configuration is created,
    *  whose RootFolder property identifies the folder.
    * <br>(reference-controlled-configuration): The ControlledConfiguration of the folder
    *  identifies the new version-controlled configuration.
    * <br>(create-new-baseline): The request MUST have created a new baseline history
    *  at a server-defined location, and MUST have created a new baseline in that baseline history.
    *  The BaselineFolder of the new baseline MUST identify a folder whose members
    *  have the same relative name and CheckedIn version as the version-controlled members of the request folder.
    *  The CheckedIn property of the new version-controlled configuration MUST identify the new baseline.
    */
   public void doBaselineControl() throws WvcmException;

   /**
    * Put this {@link ControllableFolder} under baseline control and initialize
    * it with the contents of the specified baseline.
    * @param baseline The baseline used to initialize the folder.
    *
    * @throws WvcmException Preconditions:
    * <br>(cannot-add-to-existing-history): This ControllableFolder MUST NOT identify an existing resource.
    * <br>(one-baseline-controlled-folder-per-history-per-workspace): There MUST NOT be another folder
    *  in the workspace of this ControllableFolder whose ControlledConfiguration property
    *  identifies a version-controlled configuration for the baseline history of that baseline.
    *
    * @throws WvcmException Postconditions:
    * <br>(create-controlled-configuration): A new folder is created at the location of this ControllableFolder,
    *  and a new version-controlled configuration is created, whose RootFolder property
    *  identifies the new folder.  The ControlledConfiguration property of the new folder
    *  identifies the new version-controlled configuration.
    * <br>(select-existing-baseline): The CheckedIn property of the new version-controlled configuration
    *  MUST have been set to identify the specified baseline.
    *  A version-controlled member of the folder will be created for each version in the baseline,
    *  where the version-controlled member will have the content of that version,
    *  and will have the same name relative to the folder as the corresponding
    *  version-controlled resource had when the baseline was created.
    *  Any nested folders that are needed to provide the appropriate name
    *  for a version-controlled member will be created.
    */
   public void doBaselineControl(Baseline baseline) throws WvcmException;

   /**
    * Return a list of String objects that identify
    * the names of version-controlled resources in this {@link ControllableFolder}
    * that are eclipsed by non-version-controlled resources.
    * @throws WvcmException if this {@link ControllableFolder} was not created with
    * {@link PropertyNameList.PropertyName#ECLIPSED_LIST ECLIPSED_LIST} as a wanted property.
    */
   public List getEclipsedList() throws WvcmException;

   /**
    * Return whether the folder can be put under baseline control.
    * @throws WvcmException if this {@link ControllableFolder} was not created with
    * {@link PropertyNameList.PropertyName#BASELINE_CONTROLLABLE BASELINE_CONTROLLABLE} as a wanted property.
    * @see #doBaselineControl
    */
   public boolean getBaselineControllable() throws WvcmException;

   /**
     * NOT YET STANDARD *** This method is normally retricted to Workspace ***
    * Return a list of {@link ControllableResource} objects
    * that identify the controllable resources that are members of
    * this {@link ControllableFolder} and whose versionHistory property identifies
    * a member of the <code>versionHistoryList</code> parameter.
    * @param versionHistoryList A list of {@link VersionHistory} objects that
    * are being located.
    * @param wantedPropertyList The properties to be retrieved for the
    * returned version-controlled resources.
    */
   public List doLocateByHistoryReport(
      List versionHistoryList,
      PropertyNameList wantedPropertyList) throws WvcmException;

}
