package javax.wvcm;

import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;

/**
 * A proxy for a persistent resource that can be placed under
 * control of a Checkout/Checkin protocol.  A controllable
 * resource is always the member of a special type of root
 * folder called a workspace.
 * <p>
 * The persistent state for a controllable resource may be
 * stored on the client, on the server, or both.
 * Storing the state on the server allows it to be shared
 * by multiple clients.  Storing the state on the client
 * provides access to that state even when the client is
 * disconnected from the server, and often provides more
 * efficient access to that state.
 * Note that the choice of what part of the state of a
 * controllable resource is stored on the client
 * is completely up to the implementation.  In particular,
 * although most implementations will cache some state on the client,
 * an implementation may chose to store no local state.
 * If some of the persistent state of a resource is stored on both
 * the client and the server, the state on the client and
 * the server must periodically be synchronized.
 * If the state on the server has changed since it was last
 * synchronized with the client, the resource is said to
 * be "stale".  If the state on the client has changed since
 * it was last synchronized with the server, the resource
 * is said to be "dirty".  Note that the resource can be
 * both dirty and stale.
 * <p>
 * A server may track the history of a controlled resource,
 * in which case it is called a version-controlled resource.</p>
 */
public interface ControllableResource extends Resource {

   // Basic Methods

   /**
    * Create a new persistent resource at the
    * location identified by this {@link ControllableResource}.  The request will
    * fail if a resource already exists at that location.
    * @throws WvcmException Preconditions:
    * <br>(resource-must-be-null): A resource MUST NOT exist at the location of this Resource.
    * <br>(location-ok): The location of this Resource MUST identify a valid location to create this Resource.
    *  For a workspace, a client can determine locations where workspaces can be created
    *  from the WorkspaceFolderList property.
    * <br>(workspace-server-location-specified): If the object being created is a workspace,
    *  and the persistent state of the workspace is stored
    *  on both the client and the server, the WorkspaceServerLocation property MUST identify
    *  the location of the server state for this Workspace.
    * @throws WvcmException Postconditions:
    * <br>(initialize-resource): A new resource of the specified type exists at the location of this Resource.
    */
   public void doCreateResource() throws WvcmException;

   /**
    * Return a {@link Resource} containing the wanted properties
    * that are available on the client host
    * without communicating with the server.
    * @param wantedPropertyList The properties available in the returned proxy.
    * @see Resource#doReadProperties doReadProperties
    */
   public Resource readProperties(PropertyNameList wantedPropertyList)
      throws WvcmException;

   /**
    * If the resource content is available on the client host,
    * the resource content is written to <code>content</code>
    * and <code>content</code> is closed.
    * @see Resource#doReadContent doReadContent
    * @return A Resource containing the wanted properties
    * that are available on the client host
    * without communicating with the server.
    */
   public Resource readContent(
      PropertyNameList wantedPropertyList,
      OutputStream content)
      throws WvcmException;

   /**
    * If both the client and the server are maintaining the
    * persistent state of the resource identified by this {@link ControllableResource},
    * synchronizes the state on the client to be that of the server.
    * If this {@link ControllableResource} is a {@link ControllableFolder}, every resource in the configuration
    * rooted at that folder is refreshed.
    * @return An iterator of {@link ControllableResource} objects
    * that are dirty or stale
    * @param wantedPropertyList The properties available in the returned proxies.
    * @param ignoreDirty If true, the content should be downloaded even if
    * it has been changed on the client since the last download from the
    * server (thereby overwriting those changes).  If false, the content should
    * be downloaded only if it is unchanged since the last download.
    * @throws WvcmException Preconditions:
    * <br>(must-not-be-dirty): If IgnoreDirty is false, then the state on the client
    *  MUST NOT have changed since it was last synchronized with the state on the server.
    * @throws WvcmException Postconditions:
    * <br>(content-synchronized): The content on the client has been updated to be the same as the content on the server.
    */
   public Iterator doRefresh(
      PropertyNameList wantedPropertyList,
      boolean ignoreDirty)
      throws WvcmException;

   /**
    * Require that the resource be checked out to be modified
    * and checked in to commit the modifications.
    * If the server tracks versions, each checkin creates
    * a new version in the version history of the resource.
    * @throws WvcmException Preconditions:
    * <br>(cannot-modify-checked-in-parent): If the parent of this ControllableResource
    *  is a checked-in controlled folder and this ControllableResource
    *  is not already under control, the request MUST fail.
    * @throws WvcmException Postconditions:
    * <br>(is-checked-in): The IsCheckedOut property of the resource
    *  identified by this ControllableResource MUST be false.
    * <br>(put-under-version-control): If this ControllableResource identified
    *  a version-controllable resource at the time of the request,
    *  the request MUST have created a new version history
    *  and MUST have created a new version resource in that version history.
    *  The resource MUST have a CheckedIn property that identifies the new version.
    *  The content of the new version MUST be the same as that of the resource.
    *  Note that an implementation can choose to locate the version history
    *  and version resources anywhere that it wishes.  In particular, it could locate them
    *  on the same host and server as the version-controlled resource,
    *  on a different virtual host maintained by the same server,
    *  on the same host maintained by a different server, or on a different host maintained by a different server.
    * <br>(must-not-change-existing-checked-in-out): If this ControllableResource
    *  identified a resource already under version control at the time of the request,
    *  the request MUST NOT change the CheckedIn or CheckedOut property of that version-controlled resource.
    * <br>(new-version-history): If the request created a new version history,
    *  the request MUST have allocated a new server-defined location for that version history
    *  that MUST NOT have previously identified any other resource,
    *  and MUST NOT ever identify a resource other than this version history.
    */
   public void doControl() throws WvcmException;

   /**
    * Apply {@link #doCheckout(boolean, List, boolean, boolean) 
    * doCheckout(true, null, false, false)}
    */
   public void doCheckout() throws WvcmException;

   /**
    * Apply {@link #doCheckin(boolean, boolean) doCheckin(false, true)}
    */
   public void doCheckin() throws WvcmException;

   // Basic Properties

   /**
    * Return the workspace of which this {@link ControllableResource} is a member.
    * By definition, the workspace property of a workspace
    * identifies itself, and the workspace property of any other
    * type of resource must identify the same workspace as its parent.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#WORKSPACE WORKSPACE} as a wanted property.
    */
   public Workspace getWorkspace() throws WvcmException;

   /**
    * Return whether the resource identified by this {@link ControllableResource} is
    * checked-out.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#IS_CHECKED_OUT IS_CHECKED_OUT} as a wanted property.
    */
   public boolean getIsCheckedOut() throws WvcmException;

   /**
    * Return the {@link ControllableResource} that identifies the resource
    * on the server that maintains the state for the client resource
    * identified by this {@link ControllableResource}.  If this Controllable
    * resource only has state on the server or only has state on the client,
    * <code>null</code> is returned.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#SERVER_STATE SERVER_STATE}
    * as a wanted property.
    * @see Workspace#setServerState
    */
   public ControllableResource getServerState() throws WvcmException;

   /**
    * Return whether both the client and server are maintaining persistent
    * state for the content of this {@link ControllableResource}, and whether the server
    * content has changed since the client and server content
    * was last synchronized.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#IS_STALE_CONTENT IS_STALE_CONTENT} as a wanted property.
    */
   public boolean getIsStaleContent() throws WvcmException;

   /**
    * Return whether both the client and server are maintaining persistent
    * state for the content of this {@link ControllableResource}, and whether the client
    * content has changed since the client and server content
    * was last synchronized.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#IS_DIRTY_CONTENT IS_DIRTY_CONTENT} as a wanted property.
    */
   public boolean getIsDirtyContent() throws WvcmException;

   /**
    * Return the list of property values whose persistent state
    * is being maintained on both the client and server,
    * and whose server-side state has changed 
    * since the client and server state was last synchronized.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#STALE_PROPERTY_LIST STALE_PROPERTY_LIST} as a wanted property.
    */
   public PropertyNameList getStalePropertyList() throws WvcmException;

   /**
    * Return the list of property values whose persistent state
    * is being maintained on both the client and server,
    * and whose client-side state has changed 
    * since the client and server state was last synchronized.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#DIRTY_PROPERTY_LIST DIRTY_PROPERTY_LIST} as a wanted property.
    */
   public PropertyNameList getDirtyPropertyList() throws WvcmException;

   // Versioning Methods

   /**
    * Create a new controlled resource
    * at the location identified by the proxy.  The resource
    * is associated with an existing version history,
    * and is initialized with a specified version from that
    * version history.  The request will
    * fail if a resource already exists at that location.
    * @param v The version used to initialize the controlled
    * resource.
    * 
    * @throws WvcmException Preconditions:
    * <br>(cannot-modify-checked-in-parent): If the parent of this ControllableResource
    *  is a checked-in version-controlled folder, the request MUST fail.
    * <br>(cannot-add-to-existing-history): This ControllableResource MUST NOT identify an existing resource.
    * <br>(one-version-controlled-resource-per-history-per-workspace): There MUST NOT already be
    *  a version-controlled member in the workspace of this ControllableResource
    *  whose CheckedIn or CheckedOut property identifies any version from the version history
    *  of the version specified in the request. 
    * 
    * @throws WvcmException Postconditions:
    * <br>(new-version-controlled-resource): A new version-controlled resource exists
    *  at the location of this ControllableResource whose content is initialized
    *  by those of the version in the request, and whose CheckedIn property identifies that version.
    * <br>(new-version-controlled-folder): If the request identified a folder version,
    *  the folder identified by this ControllableResource MUST contain a version-controlled
    *  bound member for each Binding specified in the ControlledBindingList of the folder version,
    *  where the name and version history of the bound member MUST be the name and version history
    *  specified by the Binding.   If the bound member is a member of a workspace,
    *  and there is another member of the workspace for the same version history,
    *  those two members MUST identify the same version-controlled resource;
    *  otherwise, a doCreateControlledResource request with a server selected version
    *  of the version history MUST have been applied to the location of that bound member.
    */
   public void doCreateVersionControlledResource(Version v)
      throws WvcmException;

   /**
    * Changes the state of the resource identified by this {@link ControllableResource}
    * to be checked-in, and synchronizes the persistent state
    * of the resource to be that of the persistent state
    * on the client.  If version history is being tracked for the
    * resource, the current content of the resource is captured
    * in a new version resource at a server-defined location.
    * A doCheckin request can be applied to a checked-out version-controlled configuration
    * to produce a new baseline whose Baseline-Folder captures the current state
    * of the version-controlled members of the RootFolder of the configuration.
    * @param keepCheckedOut indicates whether to checkout the
    * resource again immediately after checking it in.
    * If keepCheckedOut is <code>true</code>, the ActivityList
    * and Unreserved properties should not be changed by the checkout.
    * @param forkOK indicates whether to fork even if forking
    * is discouraged.
    * 
    * @throws WvcmException Preconditions:
    * <br>(must-be-checked-out): This ControllableResource MUST identify a resource whose IsCheckedOut property is true.
    * <br>(must-not-be-stale): If the content is being maintained on both the client and server,
    *  the state on the server MUST NOT have changed since it was last synchronized with the state on the client.
    * <br>(version-history-is-tree) The versions identified by the PredecessorList of the checked-out resource
    *  MUST be descendants of the root version of the version history for the CheckedOut version.
    * <br>(checkin-fork-forbidden): A doCheckin request MUST fail if it would cause a version
    *  whose CheckinFork is FORBIDDEN to appear in the PredecessorList of more than one version.
    * <br>(checkin-fork-discouraged): A doCheckin request MUST fail if it would cause a version
    *  whose CheckinFork is DISCOURAGED to appear in the PredecessorList of more than one version,
    *  unless forkOk is specified in the request.
    * <br>(merge-must-be-complete): The MergeList and AutoMergeList of the checked-out resource
    *  MUST be empty or not exist.
    * <br>(linear-activity): Any version which is in the version history of the checked-out resource
    *  and whose ActivityList identifies an activity from the ActivityList of the checked-out resource
    *  MUST be an ancestor of the checked-out resource. 
    * <br>(no-checked-out-baseline-controlled-folder-members): If this ControllableResource
    *  identifies a version-controlled configuration, all version-controlled members
    *  of the RootFolder of the version-controlled configuration MUST be checked-in.
    * <br>(one-version-per-history-per-baseline): If this ControllableResource identifies
    *  a version-controlled configuration, the set of versions selected
    *  by that version-controlled configuration MUST contain at most one version
    *  from any version history, where a version is selected by a version-controlled configuration
    *  if the version is identified by the CheckedIn property of any member
    *  of the baseline-controlled folder of that version-controlled configuration,
    *  or is identified by the CheckedIn property of any member of the BaselineFolder
    *  of any sub-baseline of that version-controlled configuration.
    * 
    * @throws WvcmException Postconditions:
    * <br>(update-server-content): If the content is being maintained on both the client and server,
    *  the content on the server MUST have been updated to contain the content on the client.
    * <br>(create-version): The request MUST have created a new version in the version history
    *  of the CheckedOut version.  The request MUST have allocated a distinct new location
    *  for the new version, and that location MUST NOT ever identify any resource other than that version.
    * <br>(initialize-version-content-and-properties): The content and PredecessorList of the new version
    *  MUST be copied from the checked-out resource.  The VersionName of the new version
    *  MUST be set to a server-defined value distinct from all other VersionName values
    *  of other versions in the same version history.
    * <br>(checked-in): If this ControllableResource identifies a version-controlled resource
    *  and keepCheckedOut is not specified in the request, the CheckedOut property
    *  of the version-controlled resource MUST have been removed
    *  and a CheckedIn property that identifies the new version MUST have been added.
    * <br>(keep-checked-out): If keepCheckedOut is specified in the request,
    *  the CheckedOut property of the checked-out resource MUST have been updated
    *  to identify the new version.
    * <br>(add-to-history): The new version resource MUST have been added to the VersionList
    *  of the version history of the CheckedOut version.
    * <br>(initialize-activity-list): The ActivityList of the new version MUST have been initialized
    *  to be the same as the ActivityList of the checked-out resource. 
    * <br>(initialize-version-controlled-bindings): If this ControllableResource identified
    *  a version-controlled folder, then the ControlledBindingList of the new folder version
    *  MUST contain a Binding that identifies the binding name and version history
    *  for each version-controlled binding of the version-controlled folder.
    * <br>(create-baseline-folder): If this ControllableResource identifies a version-controlled configuration,
    *  the BaselineFolder of the new baseline identifies a folder whose members have the same
    *  relative name and CheckedIn version as the members of the RootFolder
    *  of the version-controlled configuration at the time of the request.
    */
   public void doCheckin(boolean keepCheckedOut, boolean forkOK)
      throws WvcmException;

   /**
    * Checkout this {@link ControllableResource} so that its content can be modified.
    * @param forkOK Indicates whether to do the checkout even if the
    * version already has a successor or a checkout.
    * @param activityList A list of {@link Activity} objects that identify
    * the activities to which the work on the
    * checked-out resource will contribute.
    * @param newActivity Indicates whether to create a new activity
    * for the checkout.
    * @param unreserved Indicates whether to do the checkout even
    * if there already is a checkout from the currently selected
    * version in one of the activities specified in the activityList.
    * @throws WvcmException Preconditions:
    * <br>(must-be-checked-in): The IsCheckedOut property of this ControllableResource MUST be false.
    * <br>(must-not-be-stale): If the content is being maintained on both the client and server,
    *  the state on the server MUST NOT have changed since it was last synchronized with the state on the client.
    * <br>(checkout-of-version-with-descendant-is-forbidden): If the CheckoutFork property of the version
    *  being checked out is FORBIDDEN, the request MUST fail if a version identifies that version
    * in its PredecessorList.
    * <br>(checkout-of-version-with-descendant-is-discouraged): If the CheckoutFork property
    *  of the version being checked out is DISCOURAGED, the request MUST fail
    *  if a version identifies that version in its PredecessorList unless forkOk is specified in the request.
    * <br>(checkout-of-checked-out-version-is-forbidden): If the CheckoutFork property
    *  of the version being checked out is FORBIDDEN, the request MUST fail
    *  if a checked-out resource identifies that version in its CheckedOut property.
    * <br>(checkout-of-checked-out-version-is-discouraged): If the CheckoutFork property
    *  of the version being checked out is DISCOURAGED, the request MUST fail
    *  if a checked-out resource identifies that version in its CheckedOut property
    *  unless forkOk is specified in the request.
    * <br>(must-not-update-baseline-folder): If this ControllableResource identifies
    * a member of the BaselineFolder of a baseline, the request MUST fail.
    * <br>(one-checkout-per-activity-per-history): If there is a request activity set,
    *  unless Unreserved is specified, another checkout from a version of that version history
    *  MUST NOT select an activity in that activity set.
    * <br>(linear-activity): If there is a request activity set, unless Unreserved is specified,
    *  the selected version MUST be a descendant of all other versions of that version history
    *  that select that activity. 
    * @throws WvcmException Postconditions:
    * <br>(is-checked-out): The resource identified by this ControllableResource
    *  MUST have an IsCheckedOut property whose value is true.
    * <br>(has-checked-out-version): The checked-out resource MUST have a CheckedOut property
    *  that identifies the CheckedIn version preceding the checkout.
    *  The version-controlled resource MUST NOT have a CheckedIn property.
    * <br>(initialize-predecessor-list): The PredecessorList property of the checked-out resource
    *  MUST be initialized to be the CheckedOut version.
    * <br>(initialize-activity-list): The ActivityList of the checked-out resource is set as follows:
    *  - If newActivity is specified, then a new activity created by the server is used.
    *  - Otherwise, if activityList is non-empty, then those activities are used
    *  - Otherwise, if the version-controlled resource is a member of a workspace
    *    and the CurrentActivityList of the workspace is set, then those activities are used.
    *  - Otherwise, the ActivityList of the CheckedOut version is used. 
    * <br>(initialize-unreserved): If unreserved was specified in the request,
    *  then the Unreserved property of the checked-out resource MUST be true.
    */
   public void doCheckout(
      boolean forkOK,
      List activityList,
      boolean newActivity,
      boolean unreserved)
      throws WvcmException;

   /**
    * Merge the version specified in <code>source</code> into the resource
    * identified by this {@link ControllableResource} (the "target").
    * If the target is checked-in, and the source version is a descendant
    * of the target version, the target is updated to the source version.
    * If the target is checked-in, and the source version is an ancestor
    * of the target version, the target is left unmodified.
    * If the target is checked-out, or if the target version is on a different
    * line of descent than the source version, the target is checked out
    * (if it is not already checked-out), and the source version is either
    * added to the MergeList or the AutoMergeList of the target.
    * If a source version is added to the AutoMergeList, the server must
    * have updated the content of the target to contain the results of merging
    * the content of the source version with the content of the target at the
    * time of the merge.
    * 
    * @return A {@link ControllableResource} that contains the requested properties
    * of the resource modified by the merge.
    * @param source The version to be merged.
    * @param noAutoMerge The server is not allowed to auto-merge.
    * @param noCheckout The server is not allowed to checkout files for merging.
    * @param forkOK Passed into any checkout requests.
    * @param unreserved Passed into any checkout requests.
    * @param activityList Passed into any checkout requests.
    * @param newActivity Passed into any checkout requests.
    * @param wantedPropertyList The properties available in the returned proxy.
    * 
    * @throws WvcmException Preconditions:
    * <br>(checkout-not-allowed): If noCheckout is specified in the request,
    *  it MUST be possible to perform the merge without checking out the merge target.
    *  All preconditions of the doCheckout operation apply to any checkout performed by the request.
    * <br>(must-not-update-baseline-folder): Same semantics as doUpdate.
    * 
    * @throws WvcmException Postconditions:
    * <br>(ancestor-version): If the merge target is a version-controlled resource
    *  whose CheckedIn version or CheckedOut version is the merge source
    *  or is a descendant of the merge source, the merge target MUST NOT have been modified by the doMerge.
    * <br>(descendant-version): If the merge target was a checked-in version-controlled resource
    *  whose CheckedIn version was an ancestor of the merge source,
    *  a doUpdate operation MUST have been applied to the merge target to set its content
    *  to be that of the merge source.  If the doUpdate method is not supported,
    *  the merge target MUST have been checked out, the content of the merge target
    *  MUST have been set to those of the merge source, and the merge source
    *  MUST have been added to the AutoMergeList of the merge target.
    * <br>(checked-out-for-merge): If the merge target was a checked-in version-controlled resource
    *  whose CheckedIn version was neither a descendant nor an ancestor of the merge source,
    *  a doCheckout MUST have been applied to the merge target.
    *  All arguments that could appear in a doCheckout request MUST have been used
    *  as arguments to the doCheckout request.  
    * <br>(update-merge-list): If the CheckedOut version of the merge target is not equal to
    *  or a descendant of the merge source, the merge source MUST be added to either
    *  the MergeList or the AutoMergeList of the merge target.
    *  The merge target MUST appear in the result.  If a merge source has been added
    *  to the AutoMergeList, the content of the merge target MUST have been modified
    *  by the server to reflect the result of a logical merge of the merge source and the merge target.
    *  If a merge source has been added to the MergeList, the content of the merge target
    *  MUST NOT have been modified by the server.  If noAutoMerge is specified in the request,
    *  the merge source MUST NOT have been added to the AutoMergeList.
    * <br>(report-properties):  The properties of the merge target specified
    *  in the wantedPropertyList argument MUST be reported in the result.
    * <br>(update-version-controlled-folder-members): Same semantics as doUpdate.
    */
   public ControllableResource doMerge(
      Version source,
      boolean noAutoMerge,
      boolean noCheckout,
      boolean forkOK,
      boolean unreserved,
      List activityList,
      boolean newActivity,
      PropertyNameList wantedPropertyList)
      throws WvcmException;

   /**
    * Return a MergePreviewReport indicating how the resource
    * identified by this {@link ControllableResource}
    * would be modified by a {@link #doMerge doMerge}.
    * @param source The version to be merged.
    */
   public MergePreviewReport doMergePreviewReport(Version source)
      throws WvcmException;

   /**
    * A description of how the merge would affect the merge target.
    */
   public interface MergePreviewReport {
   }

   /**
    * A version-controlled resource whose CheckedIn property
    * would be updated by the merge.
    */
   public interface Update extends MergePreviewReport {

      /**
       * Return the version-controlled resource that would be
       * updated by the merge request.
       */
      public ControllableResource getTarget();

      /**
       * Return the version that would be selected as a result
       * of the merge request.
       */
      public Version getVersion();
   }

   /**
    * A version-controlled resource for which new content
    * would have to be created to perform the merge.
    */
   public interface Conflict extends MergePreviewReport {

      /**
       * Return a version-controlled resource that would be
       * updated by the merge request.
       */
      public ControllableResource getTarget();

      /**
       * Return a version that is the nearest common ancestor of all
       * the versions to be merged.
       */
      public Version getCommonAncestor();

      /**
       * Return a list of {@link Version} objects, that identifies the versions
       * to be merged to the specified target.
       */
      public List getVersionList();
   }

   /**
    * A version that would be ignored by the merge because 
    * its version history does not appear as a target of the merge.
    */
   public interface Ignore extends MergePreviewReport {

      /**
       * Return the version that would be ignored by the merge request.
       */
      public Version getVersion();
   }

   /**
    * Cancels the checkout of a version-controlled resource,
    * and restores its content to the state of its CheckedOut version.
    * 
    * @throws WvcmException Preconditions:
    * <br>(must-be-checked-out-version-controlled-resource): This ControllableResource
    *  MUST identify a version-controlled resource with a CheckedOut property.
    * 
    * @throws WvcmException Postconditions:
    * <br>(cancel-checked-out): The value of the CheckedIn property is that of the
    *  CheckedOut property prior to the request, and the CheckedOut property has been removed.
    * <br>(restore-content): The content of the version-controlled resource is a copy of that of its CheckedIn version.
    */
   public void doUncheckout() throws WvcmException;

   /**
    * Update the state of this checked-in version-controlled
    * {@link ControllableResource} to be the same as the specified version from
    * the version history of this {@link ControllableResource}.
    * @return An iterator of {@link ControllableResource} objects that
    * have been modified as a result of the doUpdate request.
    * Each version-controlled resource contains a value for each
    * of the properties specified in the WantedPropertyList.
    * @param v The version specifying the new state of the resource
    * identified by this {@link ControllableResource}.
    * @param w The list of properties that will be available
    * on each {@link ControllableResource} in the result.
    * 
    * @throws WvcmException Preconditions:
    * <br>(version-in-version-history): The Version argument must identify a version
    *  from the version history of this ControllableResource.
    * <br>(must-not-update-baseline-folder): If this ControllableResource identifies
    *  a member of the BaselineFolder of a baseline, the request MUST fail.
    * <br>(baseline-controlled-members-must-be-checked-in): If this ControllableResource
    *  identifies a version-controlled configuration, then all version-controlled members
    *  of the RootFolder of that version-controlled configuration MUST be checked-in.
    * 
    * @throws WvcmException Postconditions:
    * <br>(update-content-and-properties): If the Version argument identified a version
    *  that is in the same version history as the CheckedIn version of a version-controlled resource
    *  identified by this ControllableResource, then the content of that version-controlled resource
    *  MUST be the same as those of the version specified by the Version argument,
    *  and the CheckedIn property of the version-controlled resource MUST identify that version.
    * <br>(report-properties): The properties specified in the wantedPropertyList argument
    *  MUST be reported in the result.
    * <br>(update-version-controlled-folder-members): If the request modified the CheckedIn version
    *  of a version-controlled folder, then the version-controlled members
    *  of that version-controlled folder MUST have been updated.  In particular:
    *  - A version-controlled bound member MUST have been deleted if its version history
    *    is not identified by the ControlledBindingList of the new CheckedIn version.
    *  - A version-controlled bound member for a given version history
    *  MUST have been renamed if its binding name differs from the Binding name
    *  for that version history in the ControlledBindingList of the new CheckedIn version.
    *  - A new version-controlled bound member MUST have been created when a version history
    *  is identified by the ControlledBindingList of the CheckedIn version,
    *  but there was no member of the version-controlled folder for that version history.
    * If a new version-controlled member is in a workspace that already has
    *  a version-controlled resource for that version history, then the new version-controlled member
    *  MUST be just a binding (i.e. another name for) that existing version-controlled resource.
    *  Otherwise, the content of the new version-controlled member MUST have been initialized
    *  to be those of the version specified for that version history by the request.
    *  If no version is specified for that version history by the request, the version selected is server defined.
    * <br>(set-baseline-controlled-folder-members): If the request updated the CheckedIn property
    *  of a version-controlled configuration, then the version-controlled members of the RootFolder
    *  of that version-controlled configuration MUST have been updated so that they have the same
    *  relative name, content, as the members of the BaselineFolder of the baseline.  In particular:
    *  <br>- A version-controlled member for a given version history MUST have been deleted
    *    if there is no version-controlled member for that version history in the BaselineFolder of the baseline.
    *  <br>- A version-controlled member for a given version history MUST have been renamed if its name
    *    relative to the baseline-controlled folder is different from that of the version-controlled member
    *    for that version history in the BaselineFolder of the baseline.
    *  <br>- A new version-controlled member MUST have been created for each member of the BaselineFolder
    *    of the baseline for which there is no corresponding version-controlled member
    *    in the baseline-controlled folder.
    *  <br>- A doUpdate request MUST have been applied to each version-controlled member
    *    for a given version history whose CheckedIn version is not the same as that of
    *    the version-controlled member for that version history in the BaselineFolder of the baseline.
    * <br>(update-sub-baselines): If the request updated a version-controlled configuration
    *  whose RootFolder is a member of a workspace that contains a baseline-controlled member
    *  for one of the sub-baselines of the request baseline, then the CheckedIn property
    *  of the version-controlled configuration of that baseline-controlled member
    *  MUST have been updated to be that sub-baseline.
    */
   public Iterator doUpdate(Version v, PropertyNameList wantedPropertyList)
      throws WvcmException;

   /**
    * Apply {@link Resource#doBind doBind(locVersionByLabel+label, false)}
    * to the CheckedIn version of this {@link ControllableResource}, where
    * locVersionByLabel is the location of the VersionByLabelFolder of the
    * version history of this version.
    */
   public void doAddLabel(String label) throws WvcmException;

   /**
    * Apply {@link Resource#doUnbind doUnbind(locVersionByLabel+label)}
    * to the CheckedIn version of this {@link ControllableResource}, where
    * locVersionByLabel is the location of the VersionByLabelFolder of the
    * version history of this version.
    */
   public void doRemoveLabel(String label) throws WvcmException;

   /**
    * Apply {@link Resource#doBind doBind(locVersionByLabel+label, true)}
    * to the CheckedIn version of this {@link ControllableResource}, where
    * locVersionByLabel is the location of the VersionByLabelFolder of the
    * version history of this version.
    */
   public void doSetLabel(String label) throws WvcmException;

   /**
    * Apply {@link Resource#doReadProperties doReadProperties}
    * to the version selected by <code>label</code> from the version history
    * of this {@link ControllableResource}.
    */
   public Version doReadLabelledVersionProperties(
      String label,
      PropertyNameList wantedPropertyList)
      throws WvcmException;

   // Versioning Properties

   /**
    * Return whether version history will be tracked when the
    * resource is controlled.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_CONTROLLABLE VERSION_CONTROLLABLE} as a wanted property.
    * @see #doControl doControl
    */
   public boolean getVersionControllable() throws WvcmException;

   /**
    * Return the version history of the CheckedIn or CheckedOut version.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_HISTORY VERSION_HISTORY} as a wanted property.
    */
   public VersionHistory getVersionHistory() throws WvcmException;

   /**
    * Return the version-controlled configuration of the
    * baseline-controlled folder of which this {@link ControllableResource} is a member.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_CONTROLLED_CONFIGURATION VERSION_CONTROLLED_CONFIGURATION}
    * as a wanted property.
    */
   public Configuration getControlledConfiguration()
      throws WvcmException;

   /**
    * If this {@link ControllableResource} is checked-in,
    * return whether the version whose content
    * is identical to that of this {@link ControllableResource};
    * otherwise, return <code>null</code>.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#CHECKED_IN CHECKED_IN} as a wanted property.
    */
   public Version getCheckedIn() throws WvcmException;

   /**
    * If this {@link ControllableResource} is checked-out,
    * return the value of getCheckedIn
    * when the checkout was performed; otherwise,
    * return <code>null</code>.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#CHECKED_OUT CHECKED_OUT} as a wanted property.
    */
   public Version getCheckedOut() throws WvcmException;

   /**
    * If this {@link ControllableResource} is checked-out,
    * return a list of {@link Version} objects that identify the versions that
    * will become the predecessors of the version created
    * when this {@link ControllableResource} is checked in;
    * otherwise, return <code>null</code>.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#PREDECESSOR_LIST PREDECESSOR_LIST} as a wanted property.
    */
   public List getPredecessorList() throws WvcmException;

   /**
    * Specify the versions that will be the predecessors
    * of the version created when this {@link ControllableResource} is checked in.
    * @param versionList A list of {@link Version} objects that will
    * be the predecessors of the version created when this {@link ControllableResource}
    * is checked in.
    */
   public void setPredecessorList(List versionList);

   /**
    * If this {@link ControllableResource} is checked-out,
    * return a list of {@link Activity} objects that identify the
    * activities that will become the activityList of the version created
    * when this {@link ControllableResource} is checked in; otherwise,
    * return <code>null</code>/
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#ACTIVITY_LIST ACTIVITY_LIST} as a wanted property.
    */
   public List getActivityList() throws WvcmException;

   /**
    * Set the ActivityList property of this {@link ControllableResource}.
    * @param activityList A list of {@link Activity} objects that identify
    * the new value for the ActivityList property.
    */
   public void setActivityList(List activityList);

   /**
    * Return whether this ControllableResource is checked-out,
    * and whether another checkout of the same version
    * can be made in the same activity as the
    * one in the ActivitySet of this {@link ControllableResource}.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#UNRESERVED UNRESERVED} as a wanted property.
    */
   public boolean getUnreserved() throws WvcmException;

   /**
    * Set the value of the Unreserved property of this {@link ControllableResource}.
    * @param val The new value for the Unreserved property of this {@link ControllableResource}.
    */
   public void setUnreserved(boolean val);

   /**
    * If this {@link ControllableResource} is checked-out, return
    * a list of {@link Version} objects that identify the versions whose
    * content must be merged by the client into the content of
    * this {@link ControllableResource}; otherwise, return <code>null</code>.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#MERGE_LIST MERGE_LIST} as a wanted property.
    */
   public List getMergeList() throws WvcmException;

   /**
    * Set the MergeList of this {@link ControllableResource}.
    * In particular, a client is responsible for deleting versions from
    * the MergeList after the user has verified that the current
    * content represents a correct merge of that version.
    * @param versionList A list of {@link Version} objects that will be
    * the value of the MergeList property.
    */
   public void setMergeList(List versionList);

   /**
    * If this {@link ControllableResource} is checked-out,
    * return a list of {@link Version} objects that identify the versions whose
    * content has been merged by the server into the content of
    * this {@link ControllableResource}; otherwise, return <code>null</code>.
    * @throws WvcmException if this {@link ControllableResource} was not created with
    * {@link PropertyNameList.PropertyName#AUTO_MERGE_LIST AUTO_MERGE_LIST} as a wanted property.
    */
   public List getAutoMergeList() throws WvcmException;

   /**
    * Set the AutoMergeList of this {@link ControllableResource}.
    * A client is responsible for deleting entries from
    * the AutoMergeList after the user has verified that the automatic
    * merge was performed correctly by the server.
    * @param versionList A list of {@link Version} objects that identify
    * the new value for the AutoMergeList property.
    */
   public void setAutoMergeList(List versionList);

}
