package javax.wvcm;

import java.util.Iterator;
import java.util.List;

/**
 * A folder is a resource that contains a set of named mappings
 * to other resources, called the "bound members" of that folder.
 * The "members" of a folder are the folder itself and all members
 * of the bound members of the folder.
 * The semantics of a folder is similar to those of
 * a Windows folder, a Unix directory, or a WebDAV collection.
 */
public interface Folder extends Resource {

   /**
    * Return an iterator of {@link Resource} objects that contain properties of
    * the resources that are bound members of this {@link Folder}.
    * If <code>deep</code> is <code>true</code>, then
    * all members of the folder (including this {@link Folder}) are included.
    * @param wantedPropertyList The list of properties to be retrieved.
    * @param deep Whether to retrieve properties for
    * all members of this {@link Folder}.
    * @throws WvcmException Postconditions:
    * <br>(read-bound-members): The result contains a proxy for each bound member of the folder.
    * <br>(read-all-members): If deep is true, the result contains a proxy for
    *  every member of the folder identified by this Folder.
    */
   public Iterator doReadMemberList(
      PropertyNameList wantedPropertyList,
      boolean deep) throws WvcmException;

   /**
    * A Binding in a {@link Folder} is a mapping from a name
    * to a {@link Resource} that is a bound member of the {@link Folder}.
    */
   public interface Binding {
    
      /**
       * Return the folder containing the binding.
       */
      public Folder getBindingFolder();
        
      /**
       * Return the name of the binding.
       */
      public String getBindingName();

      /**
       * Return the Resource of the binding.
       */
      public Resource getBoundMember();

   }

   /**
    * Return a list of Binding objects that identify the
    * bound members of this {@link Folder}.
    * @throws WvcmException if this {@link Folder} was not created with
    * {@link PropertyNameList.PropertyName#BINDING_LIST BINDING_LIST}
    * as a wanted property.
    */
   public List getBindingList() throws WvcmException;

}
