package javax.wvcm;

import java.util.Hashtable;

/** 
 * This factory class is used to create instances of WVCM providers. 
 * The providers are identified by their class names and are runtime 
 * loaded. 
 */
public final class ProviderFactory {

   /**
    * Provides callback functions that can be invoked by a provider.
    */
   public interface Callback {

      /**
       * Authentication information for the current user.
       */
      public interface Authentication {
         /** Return the login name of the current user. */
         public String loginName();
         /** Return the password of the current user. */
         public String password();
      }

      /**
       * Return authentication information for the current user.
       * @param realm The authentication realm for the provider.
       * @param retryCount The number of times the provider has previously 
       * attempted to get authentication for this operation.
       */
      public Authentication getAuthentication(String realm, Integer retryCount);

   }

   /**
    * Return an instance of the Provider whose class name is providerName
    * @param providerName The class name of the provider.
    * @param callback The object containing the getAuthentication
    * method that the Provider will call whenever it needs
    * to authenticate the current user.
    */
   public static Provider createProvider(
      String providerName,
      Callback callback)
      throws Exception {

      Class[] parameterTypes = { Callback.class};
      Object[] initargs = { callback };

      return (Provider) Class
         .forName(providerName)
         .getConstructor(parameterTypes)
         .newInstance(initargs);
   }

   /**
    * Return an instance of the Provider whose class name is providerName
    * @param providerName The class name of the provider.
    * @param callback The object containing the getAuthentication
    * method that the Provider will call whenever it needs
    * to authenticate the current user.
    * @param h A hashtable containing provider-specific initialization values.
    */
   public static Provider createProvider(
      String providerName,
      Callback callback,
      Hashtable h)
      throws Exception {

      Class[] parameterTypes = { Callback.class, Hashtable.class};
      Object[] initargs = { callback, h };

      return (Provider) Class
         .forName(providerName)
         .getConstructor(parameterTypes)
         .newInstance(initargs);
   }

}