package javax.wvcm;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * A proxy for a persistent resource, used to retrieve information from
 * the resource and to perform operations on the resource.
 * A resource has both properties, which are named values whose
 * semantics are specified by the API, as well as "attributes"
 * whose names are arbitrary strings selected by the client.
 * An attribute also has a namespace which allows different clients
 * to use simple attribute names without conflicting with each other.
 * An attribute is called a "dead property" in RFC 2518.
 * <p>
 * A proxy for a persistent resource contains the location of
 * that persistent resource on the server that maintains that resource.
 * A proxy can be created as the return value of a request to the
 * server, in which case the proxy is initialized with a set of
 * properties reflecting the state of the persistent resource at
 * the time of the request (the list of wanted
 * properties is specified as an argument to the request).
 * Alternatively, a proxy can be created locally by a client
 * (using the Provider interface), in which case it is initialized
 * with an empty set of properties.
 * </p><p>
 * Methods whose name begin with the prefix "do", as in "doCheckin",
 * are redirected by the proxy to the persistent resource
 * (all other methods operate just on the proxy itself).
 * The state of the proxy itself is not modified by a "do" method.
 * In particular, if the request resulted in a change to any of the
 * property values of the persistent resource of the proxy, those
 * changes are not reflected in the proxy, but must be obtained
 * in a new proxy by a subsequent {@link #doReadProperties} request.
 * The "do" methods can be expected to send and receive
 * messages over the network to the remote server that maintains
 * the persistent resource.
 * </p><p>
 * Methods whose name begin with "get" return the value of properties
 * that were requested as a wanted property when the proxy was created.
 * If a caller attempts to get the value of a property from a proxy
 * that was not created with that property as a "wanted property",
 * then a <code>WvcmException</code> is thrown with a reason code
 * of <code>WvcmException.VALUE_UNAVAILABLE</code>.
 * If a wanted property could not be retrieved from the server,
 * then the "get" method will throw a <code>WvcmException</code>
 * with an appropriate response code and condition failure value.</p>
 * <p>
 * Methods whose name begin with "set" (for example
 * <code>r.setXxx(newValue)</code>), update the value of a
 * proxy property, mark that property as "updated", and subsequent
 * requests to "get" that property value from that proxy
 * (for example, <code>r.getXxx()</code>) will return the new value.
 * These updates are accumulated by a proxy,
 * until a "doXxx" method is applied to the proxy, at which time
 * the property updates are sent along with the "doXxx" method
 * to be applied to the persistent resource.  If the property
 * updates succeed on the server, the "update flags" are cleared
 * on the proxy.
 * For the "doCreateXxx" methods that create new persistent resources,
 * the property updates are applied after the persistent resource
 * is created.
 * For all other "doXxx" methods, the property updates are applied
 * before the "doXxx" method is applied, and the "doXxx" method
 * is only attempted if the property updates are successful.</p>
 * <p>
 * An "attribute" is a user or client defined property.
 * In order to prevent property name collisions between different
 * clients, an attribute is identified by both a name and a namespace.
 * <p>
 * As an example of getting and setting a property value on a
 * persistent resource, the following example appends some text
 * to the <code>comment</code> property of a resource:
 * <pre>
 Provider myProvider = createProvider("com.myprovider.wvcm", null);
 PropertyNameList wantComment
 = new PropertyNameList(new PropertyName[] { PropertyName.COMMENT });
 Resource myResource
 = myProvider.location("/my/resource/loc").resource().doReadProperties(wantComment);
 String comment = myResource.getComment();
 myResource.setComment(comment + "addition to comment");
 myResource.doWriteProperties();
 * </pre> </p>
 */
public interface Resource {
    
    /**
     * Return the location of the persistent resource for which
     * this {@link Resource} is a proxy.
     * <p>
     * The format of the location string is specific to the
     * repository that stores the persistent resource.
     * A URL, a UNC filename, and an NFS filename are examples
     * of possible formats for a location string.</p>
     */
    public Location location();
    
    // Resource properties
    
    /**
     * Return the list of names of properties available on this {@link Resource}.
     * @param onlyAttributes Only return the names of attributes.
     */
    public PropertyNameList getPropertyNameList(boolean onlyAttributes)
        throws WvcmException;
    
    /**
     * Return a {@link Resource} containing the wanted properties.
     * If the property state is being maintained on both the
     * client and the server, and if {@link ControllableResource#getServerState}
     * returns a non-NULL value, the client value is returned.
     * A requested property named XXX can be retrieved from
     * the resource with the <code>getXxx</code> method.
     */
    public Resource doReadProperties(PropertyNameList wantedPropertyList) throws WvcmException;
    
    /**
     * Return the names of properties that have been updated in
     * the proxy, but the updates have not yet been successfully
     * applied to the resource.
     */
    public PropertyNameList getUpdatedPropertyList() throws WvcmException;
    
    /**
     * Persists property changes to this {@link Resource}.
     */
    public void doWriteProperties() throws WvcmException;
    
    /**
     * Return a resource containing the wanted properties.
     * A requested property named XXX can be retrieved from
     * the resource with the <code>getXxx</code> method.
     * The resource content is written to <code>content</code>
     * and <code>content</code> is closed.
     * If state is being maintained on both the
     * client and the server, the client state is retrieved.
     */
    public Resource doReadContent(PropertyNameList wantedPropertyList, OutputStream content)
        throws WvcmException;
    
    /**
     * Persists content changes to a resource.
     * <p>
     * If content for a resource is being maintained persistently on
     * both the client and the server, only the client copy of the content
     * is updated.</p>
     * <p>
     * If <code>contentIdentifier</code> matches the current
     * state identifier of the persistent resource,
     * the content of the resource is replaced with the
     * bytes read from <code>content</code>, and <code>content</code>
     * is then closed.</p>
     * <p>
     * If reading from the stream throws a <code>java.io.IOException</code>,
     * then no further data will be read from the stream,
     * and after attempting to close the stream, a <code>WvcmException</code>
     * wrapping the <code> IOException</code> will be thrown,
     * possibly leading to incomplete data being stored on the resource.</p>
     * @throws WvcmException if the resource identified by this {@link Resource}
     * does not exist.
     *
     * @throws WvcmException Preconditions:
     * <br>(cannot-modify-controlled-content): If this ControllableResource identifies
     *  a resource whose IsCheckedOut property is false, the request MUST fail.
     * <br>(cannot-modify-version): If this proxy identifies a version, the request MUST fail.
     */
    public void doWriteContent(InputStream content, String contentIdentifier)
        throws WvcmException;
    
    /**
     * Create a copy of the resource identified by this {@link Resource}
     * at the location identified by the <code>destination</code>.
     * The content of the copy is the same as the content of the
     * resource identified by this {@link Resource}, but the properties of the
     * copy are the default properties for a new resource.
     * @param destination The location of the new resource created by doCopy.
     * @param overwrite If <code>false</code> the existence of a resource
     * at the destination will cause the copy to fail; otherwise,
     * doCopy will replace the destination resource.
     *
     * @throws WvcmException Preconditions:
     * <br>(can-overwrite): If there already is a resource at the destination,
     *  <code>overwrite</code> MUST be <code>true</code>.
     * <br>(cannot-copy-folder-version): This proxy MUST NOT identify a folder version.
     * <br>(cannot-copy-history): If this proxy identifies a version history, the request MUST fail.
     *  In order to create another version history whose versions have the same content,
     *  the appropriate sequence of doControl, doCheckout, doWriteContent, and doCheckin requests must be made.
     *
     * @throws WvcmException Postconditions:
     * <br>(must-not-copy-property): A property defined by this document MUST NOT
     * have been copied to the new resource created by this request, but instead
     * that property of the new resource MUST have the default initial value it
     * would have had if the new resource had been created by doCreateResource.
     * <br>(copy-creates-new-resource): If the source of a doCopy is a controlled resource,
     * and if there is no resource at the destination of the doCopy,
     * then the doCopy creates a new uncontrolled resource at the destination of the doCopy.
     * <br>(copy-creates-new-resource): If this proxy identifies a version,
     *  and if there is no resource at the destination of the doCopy,
     *  then the doCopy creates a new uncontrolled resource at the destination of the doCopy.
     */
    public void doCopy(String destination, boolean overwrite)
        throws WvcmException;
    
    /**
     * Unbinds the resource identified by the locator of this {@link Resource}.
     * The deletion of a resource only guarantees that the resource
     * is no longer accessible at the specified location; it does
     * not affect its accessibility at other locations.
     * If a folder is unbound, no resource is accessible at any
     * location that has the location of the unbound folder as its prefix.
     *
     * @throws WvcmException Preconditions:
     * <br>(cannot-modify-checked-in-parent): If this Resource identifies a controlled resource,
     *  the doUnbind MUST fail when the folder containing the controlled resource
     *  is a checked-in controlled folder.
     * <br>(no-version-unbind): A server MAY fail an attempt to apply doUnbind to a version.
     *
     * @throws WvcmException Postconditions:
     * <br>(resource-unbound): There is no resource at the location identified by this Resource.
     * <br>(unbind-activity-reference): If an activity is unbound, any reference to that activity
     *  in an ActivityList, SubactivityList, or CurrentActivityList MUST be removed.
     * <br>(update-predecessor-list): If a version was unbound, the server MUST have replaced
     *  any reference to that version in a PredecessorList by a copy of the PredecessorList of the unbound version.
     * <br>(version-history-has-root): If the request unbound the root version of a version history,
     *  the request MUST have updated the RootVersion of the version history to refer to
     *  another version that is an ancestor of all other remaining versions in that version history.
     *  A result of this postcondition is that every version history will have at least one version,
     *  and the only way to delete all versions is to unbind the version history resource.
     * <br>(delete-version-reference): If a version is unbound, any reference to that version in a MergeList
     * or AutoMergeList property MUST be removed.
     * <br>(delete-version-set): If the request unbound a version history,
     *  the request MUST have unbound all versions in the VersionList of that version history,
     *  and MUST have satisfied the postconditions for version deletion.
     */
    public void doUnbind() throws WvcmException;
    
    /**
     * Unbinds the resource identified by this {@link Resource}
     * from its current location and binds it
     * to the location identified by the <code>destination</code>.
     * The content and properties of a resource are not modified
     * by doRebind, except for the properties that are location dependent.
     * @param destination The new location of the resource.
     * @param overwrite If <code>false</code> the existence of a resource
     * at the destination will cause doRebind to fail; otherwise,
     * doRebind will replace the destination resource.
     *
     * @throws WvcmException Preconditions:
     * <br>(can-overwrite): If there already is a resource at the destination,
     *  <code>overwrite</code> MUST be <code>true</code>.
     * <br>(cannot-modify-checked-in-parent): If this ControllableResource is a controlled resource,
     *  the request MUST fail when the folder containing this ControllableResource
     *  is a checked-in controlled folder.
     * <br>(cannot-modify-destination-checked-in-parent): If this ControllableResource
     *  is a controlled resource, the request MUST fail when the folder containing
     *  the destination location is a checked-in controlled folder.
     * <br>(cannot-rename-history): This proxy MUST NOT identify a version history.
     * <br>(cannot-rename-version): If this proxy identifies a version, the request MUST fail.
     *
     * @throws WvcmException Postconditions:
     * <br>(preserve-properties): The property values of the resource identified by this Resource
     *  MUST NOT have been modified by the doRebind request unless this specification states otherwise.
     * <br>(workspace-member-moved): If this Resource did not identify a workspace,
     *  the Workspace property MUST have been updated to have the same value as
     *  the Workspace of its parent folder at the destination location.
     * <br>(workspace-moved): If this proxy identified a workspace,
     *  any reference to that workspace in a Workspace property MUST have been updated
     *  to refer to the new location of that workspace.
     * <br>(update-checked-out-reference): If a checked-out resource is moved, any reference
     *  to that resource in a ActivityCheckoutList property MUST be updated
     *  to refer to the new location of that resource.
     * <br>(update-workspace-reference): If this proxy identifies a workspace,
     *  any reference to that workspace in a CurrentWorkspaceList property
     *  MUST be updated to refer to the new location of that workspace.
     * <br>(update-activity-reference): If this proxy identifies an activity, any reference to that activity
     *  in a ActivityList, SubactivityList, or CurrentActivityList MUST be updated to refer to
     *  the new location of that activity.
     */
    public void doRebind(Location destination, boolean overwrite)
        throws WvcmException;
    
    /**
     * Binds the resource identified by this {@link Resource}
     * to the location identified by the <code>destination</code>.
     * The content and properties of a resource are not modified
     * by doBind.
     * @param destination The location of the new binding to the resource.
     * @param overwrite If <code>false</code> the existence of a resource
     * at the destination will cause doBind to fail; otherwise,
     * doBind will first unbind the existing resource at the destination.
     *
     * @throws WvcmException Preconditions:
     * <br>(cannot-modify-destination-checked-in-parent): If this Resource
     *  is a controlled resource, the request MUST fail when the folder containing
     *  the destination location is a checked-in controlled folder.
     * <br>(binding-allowed): This Resource supports multiple bindings to it.
     * <br>(cross-server-binding): If this Resource is on another server from
     *  the folder that contains the destination, the destination server
     *  MUST support cross-server bindings.
     * <br>(can-overwrite): If there already is a resource at the destination,
     *  <code>overwrite</code> MUST be <code>true</code>.
     * <br>(cycle-allowed): If this Resource is a folder, and the folder that contains
     *  the destination is a member of this Resource, then the server MUST support cycles
     *  in the location namespace.
     *
     * @throws WvcmException Postconditions:
     * <br>(preserve-properties): The property values of the resource identified by this Resource
     *  MUST NOT have been modified by the doBind request unless this specification states otherwise.
     * <br>(new-binding): The destination MUST identify the resource identified by this Resource.
     */
    public void doBind(Location destination, boolean overwrite)
        throws WvcmException;
    
    // Resource Properties
    
    /**
     * Return a comment describing this {@link Resource}
     * that is suitable for display to a user.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#COMMENT COMMENT} as a wanted property.
     */
    public String getComment() throws WvcmException;
    
    /**
     * Adds or replaces the comment string of this {@link Resource}.
     * @param comment The new comment to apply to this {@link Resource}.
     */
    public void setComment(String comment);
    
    /**
     * Return an implementation-defined <code>String</code>
     * that identifies the persistent state of the resource.
     * The semantics of a ContentIdentifier is similar to that
     * of an HTTP ETag (see RFC-2616).
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CONTENT_IDENTIFIER CONTENT_IDENTIFIER} as a wanted property.
     */
    public String getContentIdentifier() throws WvcmException;
    
    /**
     * Return a description of the natural language used in the
     * resource content.
     * The format of the description is an ISO 3316 language string
     * followed by an optional underscore and ISO 639 country code
     * (see RFC1766).
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CONTENT_LANGUAGE CONTENT_LANGUAGE} as a wanted property.
     */
    public Locale getContentLanguage() throws WvcmException;
    
    /**
     * Sets the content language of the resource.
     * @param locale The content language for the resource.
     */
    public void setContentLanguage(Locale contentLanguage);
    
    /**
     * Return the content length as an integer number of bytes.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CONTENT_LENGTH CONTENT_LENGTH} as a wanted property
     */
    public long getContentLength() throws WvcmException;
    
    /**
     * Return a description of the media-type of the resource content.
     * The format is a MIME type string (see RFC1590).
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CONTENT_TYPE CONTENT_TYPE} as a wanted property.
     */
    public String getContentType() throws WvcmException;
    
    /**
     * Sets the content type of the resource.
     * @param contentType The content type for the resource.
     */
    public void setContentType(String contentType);
    
    /**
     * Return a description of the character set of the resource content (see RFC2278).
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CONTENT_CHARACTER_SET CONTENT_CHARACTER_SET} as a wanted property.
     */
    public String getContentCharacterSet() throws WvcmException;
    
    /**
     * Sets the content character set of the resource.
     * @param contentCharacterSet The content character set for the resource.
     */
    public void setContentCharacterSet(String contentCharacterSet);
    
    /**
     * Return the date that the resource was originally created.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CREATION_DATE </code> CREATION_DATE </code>} as a wanted property
     */
    public Date getCreationDate() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return the date that the resource was modified.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#MODIFICATION_DATE </code> MODIFICATION_DATE </code>} as a wanted property
     */
    public Date getModificationDate() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return the principal that originally created the resource.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CREATION_USER </code> CREATION_USER </code>} as a wanted property
     */
    public Principal getCreationUser() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return the principal that modified the resource.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#MODIFICATION_USER </code> MODIFICATION_USER </code>} as a wanted property
     */
    public Principal getModificationUser() throws WvcmException;
    
    /**
     * Return a description of the user that created the resource,
     * in a format that is suitable for display to an end user.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#CREATOR_DISPLAY_NAME CREATOR_DISPLAY_NAME} as a wanted property.
     */
    public String getCreatorDisplayName() throws WvcmException;
    
    /**
     * Adds or replaces the creator display name string of the resource.
     * @param comment The new creator display name to apply to the resource.
     */
    public void setCreatorDisplayName(String val);
    
    /**
     * Return a short description of the resource, in a format
     * that is suitable for display to an end user in a tree display.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#DISPLAY_NAME DISPLAY_NAME} as a wanted property.
     */
    public String getDisplayName() throws WvcmException;
    
    /**
     * Adds or replaces the display name string of the resource.
     * @param comment The new display name to apply to the resource.
     */
    public void setDisplayName(String val);
    
    /**
     * Return the date the content of the resource was last modified.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#LAST_MODIFIED LAST_MODIFIED} as a wanted property.
     */
    public Date getLastModified() throws WvcmException;
    
    /**
     * Return a list of {@link Folder} objects that identify folders
     * that contain workspaces.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#WORKSPACE_FOLDER_LIST WORKSPACE_FOLDER_LIST} as a wanted property.
     */
    public List getWorkspaceFolderList() throws WvcmException;
    
    /**
     * Return an attribute of this {@link Resource}.
     * @param name The name of the attribute.
     * @throws WvcmException if this {@link Resource} was not created with
     * the specified attribute as a wanted property.
     */
    public Object getAttribute(PropertyNameList.AttributeName name)
        throws WvcmException;
    
    /**
     * Adds or replaces the value of the specified attribute of this {@link Resource}.
     * @param name The name of the attribute.
     * @param value The new value of the specified attribute.
     */
    public void setAttribute(PropertyNameList.AttributeName name, Object value);
    
    /**
     * Removes the specified attribute of this {@link Resource}.
     * @param name The name of the attribute.
     */
    public void removeAttribute(PropertyNameList.AttributeName name);
    
    /**
     * Return a list of String objects that identify
     * the names of providers for this resource, with the preferred
     * providers specified earlier in the list.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#PROVIDER_LIST PROVIDER_LIST} as a wanted property.
     */
    public List getProviderList()
        throws WvcmException;
    
    /**
     * Return a unique identifier for this Resource.
     * The resource identifier enables clients to determine
     * whether two bindings are to the same resource. The value of this property
     * is a URI, and may use any registered URI scheme that guarantees the
     * uniqueness of the value across all resources for all time.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#RESOURCE_IDENTIFIER RESOURCE_IDENTIFIER} as a wanted property.
     */
    public String getResourceIdentifier()
        throws WvcmException;
    
    /**
     * Return the list of bindings to this resource.
     * The ParentBindingList property enables
     * clients to discover what folders contain a binding to this resource
     * (i.e. what folders have this resource as a bound member).
     * The returned list contains {@link Folder.Binding} instances.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#PARENT_BINDING_LIST PARENT_BINDING_LIST}
     * as a wanted property.
     */
    public List getParentBindingList()
        throws WvcmException;
    
    /**
     * NOT YET STANDARD !!!
     * Return a property of this {@link Resource}.
     */
    public Object getProperty(PropertyNameList.PropertyName name)
        throws WvcmException;
    
    /**
     * NOT YET STANDARD !!!
     * Adds or replaces the value of the specified property of this {@link Resource}.
     */
    public void setProperty(PropertyNameList.PropertyName name, Object value);
    
    /**
     * NOT YET STANDARD !!!
     * Return a list of {@link Resource} objects containing the wanted properties
     * according to the conditions of the specified searct token from the scope
     * defined by this {@link Resource}.
     * A requested property named XXX can be retrieved from
     * the resource with the <code>getXxx</code> method.
     */
    public List doSearch(PropertyNameList wantedPropertyList, SearchToken searchToken) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Get the owner of this resource
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#OWNER OWNER} as a wanted property.
     */
    public Principal getOwner() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return a list of {@link Folder} objects that identify folders
     * that contain principals.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#PRINCIPAL_FOLDER_LIST PRINCIPAL_FOLDER_LIST} as a wanted property.
     */
    public List getPrincipalFolderList() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return a list of {@link Folder} objects that identify folders
     * that contain privileges.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#PRIVILEGE_FOLDER_LIST PRIVILEGE_FOLDER_LIST} as a wanted property.
     */
    public List getPrivilegeFolderList() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return a list of {@link AccessControlElement.Privilege} objects defined for the resource.
     * @throws WvcmException if this {@link Resource} was not created with
     * {@link PropertyNameList.PropertyName#SUPPORTED_PRIVILEGE_LIST SUPPORTED_PRIVILEGE_LIST} as a wanted property.
     */
    public List getSupportedPrivilegeList() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Return the list of {@link AccessControlElement} instances (ACL) defined
     * on this resource.
     * The ACL specifies the list of access control elements (ACEs), which define what principals
     * are to get what privileges for this resource.
     * Each ACE specifies the set of privileges to be either granted or denied to a single principal.
     * If the ACL is empty, no principal is granted any privilege.
     *
     * @param    includeInherited    if false, only ACEs defined for the resource are returned;
     *                               otherwise, the ACL includes all inherited ACEs
     * @return   a List
     * @throws   WvcmException
     */
    public List doReadAccessControlList( boolean includeInherited ) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Modifies the access control list (ACL) of this resource. Specifically, this method only
     * permits modification to ACEs that are not inherited, and are not protected.
     *
     * @param    acl                 a list of {@link AccessControlElement} instances
     * @throws   WvcmException
     */
    public void doWriteAccessControlList( List acl ) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Locks this resource.
     *
     * @param    timeout             a  Timeout
     * @param    deep                a  boolean
     * @throws   WvcmException
     */
    public void doLock(LockToken.Timeout timeout, boolean deep) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Locks this resource.
     *
     * @param    timeout             a  Timeout
     * @param    deep                a  boolean
     * @param    exclusive           a  boolean
     * @param    owner               a  String
     * @throws   WvcmException
     */
    public void doLock(LockToken.Timeout timeout, boolean deep, boolean exclusive, String owner) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Releases the active lock of this resource.
     *
     * @throws   WvcmException
     */
    public void doUnlock() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Releases the specified lock of this resource.
     *
     * @param    lockToken           a  LockToken
     * @throws   WvcmException
     */
    public void doUnlock(LockToken lockToken) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Returns the list of lock tokens available at this resource.
     *
     * @return   a List
     * @throws   WvcmException
     */
    public List getLockTokens() throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Implementation of a generic Resource.
     *
     * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
     * @version $Revision: 1.11 $
     */
    
    public abstract class GenericResourceImpl implements Resource {
        
        private Location location;
        
        /**
         * Private Constructor
         *
         * @param    location            a  Location
         */
        public GenericResourceImpl(Location location) {
            this.location = location;
        }
        
        /**
         * Sets the content language of the resource.
         * @param locale The content language for the resource.
         */
        public void setContentLanguage(Locale contentLanguage) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return a list of {@link Folder} objects that identify folders
         * that contain principals.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#PRINCIPAL_FOLDER_LIST PRINCIPAL_FOLDER_LIST} as a wanted property.
         */
        public List getPrincipalFolderList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Removes the specified attribute of this {@link Resource}.
         * @param name The name of the attribute.
         */
        public void removeAttribute(javax.wvcm.PropertyNameList.AttributeName name) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Get the owner of this resource
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#OWNER OWNER} as a wanted property.
         */
        public Principal getOwner() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a resource containing the wanted properties.
         * A requested property named XXX can be retrieved from
         * the resource with the <code>getXxx</code> method.
         * The resource content is written to <code>content</code>
         * and <code>content</code> is closed.
         * If state is being maintained on both the
         * client and the server, the client state is retrieved.
         */
        public Resource doReadContent(PropertyNameList wantedPropertyList, OutputStream content) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a {@link Resource} containing the wanted properties.
         * If the property state is being maintained on both the
         * client and the server, and if {@link ControllableResource#getServerState}
         * returns a non-NULL value, the client value is returned.
         * A requested property named XXX can be retrieved from
         * the resource with the <code>getXxx</code> method.
         */
        public Resource doReadProperties(PropertyNameList wantedPropertyList) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a short description of the resource, in a format
         * that is suitable for display to an end user in a tree display.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#DISPLAY_NAME DISPLAY_NAME} as a wanted property.
         */
        public String getDisplayName() throws WvcmException {
            return location.string();
        }
        
        /**
         * Return a list of String objects that identify
         * the names of providers for this resource, with the preferred
         * providers specified earlier in the list.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#PROVIDER_LIST PROVIDER_LIST} as a wanted property.
         */
        public List getProviderList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the date that the resource was originally created.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CREATION_DATE </code> CREATION_DATE </code>} as a wanted property
         */
        public Date getCreationDate() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Adds or replaces the display name string of the resource.
         * @param comment The new display name to apply to the resource.
         */
        public void setDisplayName(String val) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the content length as an integer number of bytes.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CONTENT_LENGTH CONTENT_LENGTH} as a wanted property
         */
        public long getContentLength() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Unbinds the resource identified by the locator of this {@link Resource}.
         * The deletion of a resource only guarantees that the resource
         * is no longer accessible at the specified location; it does
         * not affect its accessibility at other locations.
         * If a folder is unbound, no resource is accessible at any
         * location that has the location of the unbound folder as its prefix.
         *
         * @throws WvcmException Preconditions:
         * <br>(cannot-modify-checked-in-parent): If this Resource identifies a controlled resource,
         *  the doUnbind MUST fail when the folder containing the controlled resource
         *  is a checked-in controlled folder.
         * <br>(no-version-unbind): A server MAY fail an attempt to apply doUnbind to a version.
         *
         * @throws WvcmException Postconditions:
         * <br>(resource-unbound): There is no resource at the location identified by this Resource.
         * <br>(unbind-activity-reference): If an activity is unbound, any reference to that activity
         *  in an ActivityList, SubactivityList, or CurrentActivityList MUST be removed.
         * <br>(update-predecessor-list): If a version was unbound, the server MUST have replaced
         *  any reference to that version in a PredecessorList by a copy of the PredecessorList of the unbound version.
         * <br>(version-history-has-root): If the request unbound the root version of a version history,
         *  the request MUST have updated the RootVersion of the version history to refer to
         *  another version that is an ancestor of all other remaining versions in that version history.
         *  A result of this postcondition is that every version history will have at least one version,
         *  and the only way to delete all versions is to unbind the version history resource.
         * <br>(delete-version-reference): If a version is unbound, any reference to that version in a MergeList
         * or AutoMergeList property MUST be removed.
         * <br>(delete-version-set): If the request unbound a version history,
         *  the request MUST have unbound all versions in the VersionList of that version history,
         *  and MUST have satisfied the postconditions for version deletion.
         */
        public void doUnbind() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a unique identifier for this Resource.
         * The resource identifier enables clients to determine
         * whether two bindings are to the same resource. The value of this property
         * is a URI, and may use any registered URI scheme that guarantees the
         * uniqueness of the value across all resources for all time.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#RESOURCE_IDENTIFIER RESOURCE_IDENTIFIER} as a wanted property.
         */
        public String getResourceIdentifier() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return a list of {@link Folder} objects that identify folders
         * that contain privileges.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#PRIVILEGE_FOLDER_LIST PRIVILEGE_FOLDER_LIST} as a wanted property.
         */
        public List getPrivilegeFolderList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return the date that the resource was modified.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#MODIFICATION_DATE </code> MODIFICATION_DATE </code>} as a wanted property
         */
        public Date getModificationDate() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the date the content of the resource was last modified.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#LAST_MODIFIED LAST_MODIFIED} as a wanted property.
         */
        public Date getLastModified() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the names of properties that have been updated in
         * the proxy, but the updates have not yet been successfully
         * applied to the resource.
         */
        public PropertyNameList getUpdatedPropertyList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Unbinds the resource identified by this {@link Resource}
         * from its current location and binds it
         * to the location identified by the <code>destination</code>.
         * The content and properties of a resource are not modified
         * by doRebind, except for the properties that are location dependent.
         * @param destination The new location of the resource.
         * @param overwrite If <code>false</code> the existence of a resource
         * at the destination will cause doRebind to fail; otherwise,
         * doRebind will replace the destination resource.
         *
         * @throws WvcmException Preconditions:
         * <br>(can-overwrite): If there already is a resource at the destination,
         *  <code>overwrite</code> MUST be <code>true</code>.
         * <br>(cannot-modify-checked-in-parent): If this ControllableResource is a controlled resource,
         *  the request MUST fail when the folder containing this ControllableResource
         *  is a checked-in controlled folder.
         * <br>(cannot-modify-destination-checked-in-parent): If this ControllableResource
         *  is a controlled resource, the request MUST fail when the folder containing
         *  the destination location is a checked-in controlled folder.
         * <br>(cannot-rename-history): This proxy MUST NOT identify a version history.
         * <br>(cannot-rename-version): If this proxy identifies a version, the request MUST fail.
         *
         * @throws WvcmException Postconditions:
         * <br>(preserve-properties): The property values of the resource identified by this Resource
         *  MUST NOT have been modified by the doRebind request unless this specification states otherwise.
         * <br>(workspace-member-moved): If this Resource did not identify a workspace,
         *  the Workspace property MUST have been updated to have the same value as
         *  the Workspace of its parent folder at the destination location.
         * <br>(workspace-moved): If this proxy identified a workspace,
         *  any reference to that workspace in a Workspace property MUST have been updated
         *  to refer to the new location of that workspace.
         * <br>(update-checked-out-reference): If a checked-out resource is moved, any reference
         *  to that resource in a ActivityCheckoutList property MUST be updated
         *  to refer to the new location of that resource.
         * <br>(update-workspace-reference): If this proxy identifies a workspace,
         *  any reference to that workspace in a CurrentWorkspaceList property
         *  MUST be updated to refer to the new location of that workspace.
         * <br>(update-activity-reference): If this proxy identifies an activity, any reference to that activity
         *  in a ActivityList, SubactivityList, or CurrentActivityList MUST be updated to refer to
         *  the new location of that activity.
         */
        public void doRebind(Location destination, boolean overwrite) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Binds the resource identified by this {@link Resource}
         * to the location identified by the <code>destination</code>.
         * The content and properties of a resource are not modified
         * by doBind.
         * @param destination The location of the new binding to the resource.
         * @param overwrite If <code>false</code> the existence of a resource
         * at the destination will cause doBind to fail; otherwise,
         * doBind will first unbind the existing resource at the destination.
         *
         * @throws WvcmException Preconditions:
         * <br>(cannot-modify-destination-checked-in-parent): If this Resource
         *  is a controlled resource, the request MUST fail when the folder containing
         *  the destination location is a checked-in controlled folder.
         * <br>(binding-allowed): This Resource supports multiple bindings to it.
         * <br>(cross-server-binding): If this Resource is on another server from
         *  the folder that contains the destination, the destination server
         *  MUST support cross-server bindings.
         * <br>(can-overwrite): If there already is a resource at the destination,
         *  <code>overwrite</code> MUST be <code>true</code>.
         * <br>(cycle-allowed): If this Resource is a folder, and the folder that contains
         *  the destination is a member of this Resource, then the server MUST support cycles
         *  in the location namespace.
         *
         * @throws WvcmException Postconditions:
         * <br>(preserve-properties): The property values of the resource identified by this Resource
         *  MUST NOT have been modified by the doBind request unless this specification states otherwise.
         * <br>(new-binding): The destination MUST identify the resource identified by this Resource.
         */
        public void doBind(Location destination, boolean overwrite) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a comment describing this {@link Resource}
         * that is suitable for display to a user.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#COMMENT COMMENT} as a wanted property.
         */
        public String getComment() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a list of {@link Folder} objects that identify folders
         * that contain workspaces.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#WORKSPACE_FOLDER_LIST WORKSPACE_FOLDER_LIST} as a wanted property.
         */
        public List getWorkspaceFolderList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return the principal that originally created the resource.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CREATION_USER </code> CREATION_USER </code>} as a wanted property
         */
        public Principal getCreationUser() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Sets the content type of the resource.
         * @param contentType The content type for the resource.
         */
        public void setContentType(String contentType) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a description of the media-type of the resource content.
         * The format is a MIME type string (see RFC1590).
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CONTENT_TYPE CONTENT_TYPE} as a wanted property.
         */
        public String getContentType() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Adds or replaces the comment string of this {@link Resource}.
         * @param comment The new comment to apply to this {@link Resource}.
         */
        public void setComment(String comment) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Persists property changes to this {@link Resource}.
         */
        public void doWriteProperties() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD !!!
         * Return a list of {@link Resource} objects containing the wanted properties
         * according to the conditions of the specified searct token from the scope
         * defined by this {@link Resource}.
         * A requested property named XXX can be retrieved from
         * the resource with the <code>getXxx</code> method.
         */
        public List doSearch(PropertyNameList wantedPropertyList, SearchToken searchToken) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return the principal that modified the resource.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#MODIFICATION_USER </code> MODIFICATION_USER </code>} as a wanted property
         */
        public Principal getModificationUser() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD !!!
         * Return a property of this {@link Resource}.
         */
        public Object getProperty(javax.wvcm.PropertyNameList.PropertyName name) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return an attribute of this {@link Resource}.
         * @param name The name of the attribute.
         * @throws WvcmException if this {@link Resource} was not created with
         * the specified attribute as a wanted property.
         */
        public Object getAttribute(javax.wvcm.PropertyNameList.AttributeName name) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the location of the persistent resource for which
         * this {@link Resource} is a proxy.
         * <p>
         * The format of the location string is specific to the
         * repository that stores the persistent resource.
         * A URL, a UNC filename, and an NFS filename are examples
         * of possible formats for a location string.</p>
         */
        public Location location() {
            return location;
        }
        
        /**
         * NOT YET STANDARD
         * Return the list of {@link AccessControlElement} instances (ACL) defined
         * on this resource.
         * The ACL specifies the list of access control elements (ACEs), which define what principals
         * are to get what privileges for this resource.
         * Each ACE specifies the set of privileges to be either granted or denied to a single principal.
         * If the ACL is empty, no principal is granted any privilege.
         *
         * @param    includeInherited    if false, only ACEs defined for the resource are returned;
         *                               otherwise, the ACL includes all inherited ACEs
         * @return   a List
         * @throws   WvcmException
         */
        public List doReadAccessControlList(boolean includeInherited) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return an implementation-defined <code>String</code>
         * that identifies the persistent state of the resource.
         * The semantics of a ContentIdentifier is similar to that
         * of an HTTP ETag (see RFC-2616).
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CONTENT_IDENTIFIER CONTENT_IDENTIFIER} as a wanted property.
         */
        public String getContentIdentifier() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the list of bindings to this resource.
         * The ParentBindingList property enables
         * clients to discover what folders contain a binding to this resource
         * (i.e. what folders have this resource as a bound member).
         * The returned list contains {@link Folder.Binding} instances.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#PARENT_BINDING_LIST PARENT_BINDING_LIST}
         * as a wanted property.
         */
        public List getParentBindingList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a description of the character set of the resource content (see RFC2278).
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CONTENT_CHARACTER_SET CONTENT_CHARACTER_SET} as a wanted property.
         */
        public String getContentCharacterSet() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Return a list of {@link AccessControlElement.Privilege} objects defined for the resource.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#SUPPORTED_PRIVILEGE_LIST SUPPORTED_PRIVILEGE_LIST} as a wanted property.
         */
        public List getSupportedPrivilegeList() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a description of the user that created the resource,
         * in a format that is suitable for display to an end user.
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#CREATOR_DISPLAY_NAME CREATOR_DISPLAY_NAME} as a wanted property.
         */
        public String getCreatorDisplayName() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Modifies the access control list (ACL) of this resource. Specifically, this method only
         * permits modification to ACEs that are not inherited, and are not protected.
         *
         * @param    acl                 a list of {@link AccessControlElement} instances
         * @throws   WvcmException
         */
        public void doWriteAccessControlList(List acl) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Create a copy of the resource identified by this {@link Resource}
         * at the location identified by the <code>destination</code>.
         * The content of the copy is the same as the content of the
         * resource identified by this {@link Resource}, but the properties of the
         * copy are the default properties for a new resource.
         * @param destination The location of the new resource created by doCopy.
         * @param overwrite If <code>false</code> the existence of a resource
         * at the destination will cause the copy to fail; otherwise,
         * doCopy will replace the destination resource.
         *
         * @throws WvcmException Preconditions:
         * <br>(can-overwrite): If there already is a resource at the destination,
         *  <code>overwrite</code> MUST be <code>true</code>.
         * <br>(cannot-copy-folder-version): This proxy MUST NOT identify a folder version.
         * <br>(cannot-copy-history): If this proxy identifies a version history, the request MUST fail.
         *  In order to create another version history whose versions have the same content,
         *  the appropriate sequence of doControl, doCheckout, doWriteContent, and doCheckin requests must be made.
         *
         * @throws WvcmException Postconditions:
         * <br>(must-not-copy-property): A property defined by this document MUST NOT
         * have been copied to the new resource created by this request, but instead
         * that property of the new resource MUST have the default initial value it
         * would have had if the new resource had been created by doCreateResource.
         * <br>(copy-creates-new-resource): If the source of a doCopy is a controlled resource,
         * and if there is no resource at the destination of the doCopy,
         * then the doCopy creates a new uncontrolled resource at the destination of the doCopy.
         * <br>(copy-creates-new-resource): If this proxy identifies a version,
         *  and if there is no resource at the destination of the doCopy,
         *  then the doCopy creates a new uncontrolled resource at the destination of the doCopy.
         */
        public void doCopy(String destination, boolean overwrite) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Persists content changes to a resource.
         * <p>
         * If content for a resource is being maintained persistently on
         * both the client and the server, only the client copy of the content
         * is updated.</p>
         * <p>
         * If <code>contentIdentifier</code> matches the current
         * state identifier of the persistent resource,
         * the content of the resource is replaced with the
         * bytes read from <code>content</code>, and <code>content</code>
         * is then closed.</p>
         * <p>
         * If reading from the stream throws a <code>java.io.IOException</code>,
         * then no further data will be read from the stream,
         * and after attempting to close the stream, a <code>WvcmException</code>
         * wrapping the <code> IOException</code> will be thrown,
         * possibly leading to incomplete data being stored on the resource.</p>
         * @throws WvcmException if the resource identified by this {@link Resource}
         * does not exist.
         *
         * @throws WvcmException Preconditions:
         * <br>(cannot-modify-controlled-content): If this ControllableResource identifies
         *  a resource whose IsCheckedOut property is false, the request MUST fail.
         * <br>(cannot-modify-version): If this proxy identifies a version, the request MUST fail.
         */
        public void doWriteContent(InputStream content, String contentIdentifier) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD !!!
         * Adds or replaces the value of the specified property of this {@link Resource}.
         */
        public void setProperty(javax.wvcm.PropertyNameList.PropertyName name, Object value) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return the list of names of properties available on this {@link Resource}.
         * @param onlyAttributes Only return the names of attributes.
         */
        public PropertyNameList getPropertyNameList(boolean onlyAttributes) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Return a description of the natural language used in the
         * resource content.
         * The format of the description is an ISO 3316 language string
         * followed by an optional underscore and ISO 639 country code
         * (see RFC1766).
         * @throws WvcmException if this {@link Resource} was not created with
         * {@link PropertyNameList.PropertyName#COMMENT COMMENT} as a wanted property.
         */
        public Locale getContentLanguage() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Sets the content character set of the resource.
         * @param contentCharacterSet The content character set for the resource.
         */
        public void setContentCharacterSet(String contentCharacterSet) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Adds or replaces the value of the specified attribute of this {@link Resource}.
         * @param name The name of the attribute.
         * @param value The new value of the specified attribute.
         */
        public void setAttribute(javax.wvcm.PropertyNameList.AttributeName name, Object value) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * Adds or replaces the creator display name string of the resource.
         * @param comment The new creator display name to apply to the resource.
         */
        public void setCreatorDisplayName(String val) {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Returns the list of lock tokens available at this resource.
         *
         * @return   a List
         * @throws   WvcmException
         */
        public List getLockTokens() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Locks this resource.
         *
         * @param    timeout             a  Timeout
         * @param    deep                a  boolean
         * @throws   WvcmException
         */
        public void doLock(javax.wvcm.LockToken.Timeout timeout, boolean deep) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Releases the active lock of this resource.
         *
         * @throws   WvcmException
         */
        public void doUnlock() throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Locks this resource.
         *
         * @param    timeout             a  Timeout
         * @param    deep                a  boolean
         * @param    exclusive           a  boolean
         * @param    owner               a  String
         * @throws   WvcmException
         */
        public void doLock(javax.wvcm.LockToken.Timeout timeout, boolean deep, boolean exclusive, String owner) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        /**
         * NOT YET STANDARD
         * Releases the specified lock of this resource.
         *
         * @param    lockToken           a  LockToken
         * @throws   WvcmException
         */
        public void doUnlock(LockToken lockToken) throws WvcmException {
            throw new UnsupportedOperationException("Not supported for generic resources");
        }
        
        public String toString() {
            return location().string();
        }
    }
}
