package javax.wvcm;
import java.util.List;

/**
 * A version is an immutable copy of a particular state of a version-controlled
 * resource.
 */
public interface Version extends Resource {

   /**
    * Return a list of {@link Activity} objects that identify the
    * activities to which this {@link Version} contributes, and on which
    * lines of descent this version appears.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#ACTIVITY_LIST ACTIVITY_LIST} as a wanted property.
    */
   public List getActivityList() throws WvcmException;

   /**
    * Set the ActivityList property.
    * A server MAY restrict the ActivityList to identify a single activity.
    * A server MAY refuse to allow the value of the ActivityList property of a version to be modified. 
    * @param activityList A list of {@link Activity} objects that specify
    * the new ActivityList property for this {@link Version}.
    */
   public void setActivityList(List activityList);

   /**
    * Valid values for the CheckinFork and CheckoutFork properties.
    */
   public final class Fork {
      public static final int OK = 1;
      public static final int DISCOURAGED = 2;
      public static final int FORBIDDEN = 3;
   }

   /**
    * Return the CheckinFork property, which determines whether
    * this {@link Version} can be given more than one successor.
    * If the CheckinFork of a version is FORBIDDEN, a doCheckin request MUST fail
    * if it would result in that version appearing in the PredecessorList of more than one version.  
    * If CheckinFork is DISCOURAGED, such a doCheckin request MUST fail unless forkOk is specified 
    * in the doCheckin request.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#CHECKIN_FORK CHECKIN_FORK} as a wanted property.
    */
   public int getCheckinFork() throws WvcmException;

   /**
    * Set the CheckinFork property.
    * A server MAY reject attempts to modify the CheckinFork of a version.
    * @param val The new CheckinFork value for this {@link Version}.
    */
   public void setCheckinFork(int val);

   /**
    * Return the CheckoutFork property, which determines whether
    * a version-controlled resource selecting this {@link Version}
    * can be checked out when it already has a successor version.
    * If the CheckoutFork of a version is FORBIDDEN, a doCheckout request MUST fail
    * if it would result in that version appearing in the PredecessorList or CheckedOut property
    * of more than one version or checked-out resource.
    * If CheckoutFork is DISCOURAGED, such a doCheckout request MUST fail unless forkOk is specified
    * in the doCheckout request.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#CHECKOUT_FORK CHECKOUT_FORK} as a wanted property.
    */
   public int getCheckoutFork() throws WvcmException;

   /**
    * Set the CheckoutFork property.
    * A server MAY reject attempts to modify the CheckoutFork of a version.
    * @param val The new CheckoutFork value for this {@link Version}.
    */
   public void setCheckoutFork(int val);

   /**
    * Return a list of {@link ControllableResource} objects
    * that identify the checked-out version-controlled
    * resources that refer to this {@link Version} in their CheckedOut property.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#CHECKOUT_LIST CHECKOUT_LIST} as a wanted property.
    * @see ControllableResource#getCheckedOut ControllableResource.getCheckedOut
    */
   public List getCheckoutList()
      throws WvcmException;

   /**
    * Return a list of String objects that identify
    * labels that currently select this {@link Version}.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#LABEL_NAME_LIST LABEL_NAME_LIST} as a wanted property.
    */
   public List getLabelNameList() throws WvcmException;

   /**
    * Return a list of {@link Version} objects that identify
    * the predecessors of this {@link Version}.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#PREDECESSOR_LIST PREDECESSOR_LIST} as a wanted property.
    */
   public List getPredecessorList() throws WvcmException;

   /**
    * Return a list of {@link Version} objects that identify
    * the versions that are the successors of this {@link Version}.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#SUCCESSOR_LIST SUCCESSOR_LIST} as a wanted property.
    */
   public List getSuccessorList() throws WvcmException;

   /**
    * Return the version history resource that contains this {@link Version}.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_HISTORY VERSION_HISTORY} as a wanted property.
    */
   public VersionHistory getVersionHistory() throws WvcmException;

   /**
    * Return the version name of this {@link Version}.
    * This property contains a server-defined string that is different for each version
    * in a given version history.  This string is intended for display to a user, 
    * unlike the location of a version, which is normally only used by a client and not displayed to a user.
    * @throws WvcmException if this {@link Version} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_NAME VERSION_NAME} as a wanted property.
    */
   public String getVersionName() throws WvcmException;

}
