package javax.wvcm;
import java.util.Iterator;
import java.util.List;

/**
 * Represents workspace resource.
 * See section 6 of RFC 3253. 
 */
public interface Workspace extends ControllableFolder {

   /**
    * When this {@link Workspace} is located on the client,
    * this routine is used to specify the {@link Workspace}
    * on the server that maintains the server state for
    * this {@link Workspace}
    * This property must be set before creating a workspace that
    * is located on the client.
    * @param serverState The {@link Workspace} located
    * on the server that maintains the server state
    * for this {@link Workspace}.
    * @see ControllableResource#getServerState
    */
   public void setServerState(Workspace workspace);

	/**
	 * Return a list of {@link Folder} objects that identify folders that
	 * can contain activities for this {@link Workspace}.
	 * This list is empty if this {@link Workspace}
	 * does not provide activity support.
    * @throws WvcmException if this {@link Workspace} was not created with
    * {@link PropertyNameList.PropertyName#ACTIVITY_FOLDER_LIST ACTIVITY_FOLDER_LIST}
    * as a wanted property.
	 */
	public List getActivityFolderList() throws WvcmException;

	/**
	 * Return a list of {@link Folder} objects that identify folders that
	 * can contain version histories for this {@link Workspace}.
	 * This list is empty if this {@link Workspace}
	 * does not provide versioning support.
    * @throws WvcmException if this {@link Workspace} was not created with
    * {@link PropertyNameList.PropertyName#VERSION_HISTORY_FOLDER_LIST VERSION_HISTORY_FOLDER_LIST}
    * as a wanted property.
	 */
	public List getVersionHistoryFolderList() throws WvcmException;
	
   /**
    * Return a list of {@link ControllableFolder} objects that identify the folders
    * that are the roots of the baseline-controlled
    * configurations in that workspace.
    * @throws WvcmException if this {@link Workspace} was not created with
    * {@link PropertyNameList.PropertyName#BASELINE_CONTROLLED_FOLDER_LIST BASELINE_CONTROLLED_FOLDER_LIST}
    * as a wanted property.
    */
   public List getBaselineControlledFolderList()
      throws WvcmException;

   /**
    * Return a list of {@link Activity} objects that identify
    * the activities that will initialize the ActivityList property
    * of a version-controlled member of this {@link Workspace} when that member
    * is checked out and no explicit ActivityList is
    * specified in the checkout request.
    * This allows an activity-unaware client to update a workspace in which activity tracking is required.
    * The CurrentActivityList MAY be restricted to identify at most one activity.
    * @throws WvcmException if this {@link Workspace} was not created with
    * {@link PropertyNameList.PropertyName#CURRENT_ACTIVITY_LIST CURRENT_ACTIVITY_LIST} as a wanted property.
    */
   public List getCurrentActivityList() throws WvcmException;

   /**
    * Set the CurrentActivityList property.
    * @param activityList The list of {@link Activity} objects that
    * identify the new CurrentActivityList for this {@link Workspace}.
    */
   public void setCurrentActivityList(List activityList);

   /**
    * Return a list of {@link ControllableResource} objects
    * that identify the members of this {@link Workspace} that are checked-out
    * version-controlled resources.
    * @throws WvcmException if this {@link Workspace} was not created with
    * {@link PropertyNameList.PropertyName#WORKSPACE_CHECKOUT_LIST WORKSPACE_CHECKOUT_LIST} as a wanted property.
    */
   public List getWorkspaceCheckoutList() throws WvcmException;

   /**
    * Return a list of {@link ControllableResource} objects
    * that identify the controllable resources that are members of
    * this {@link Workspace} and whose versionHistory property identifies
    * a member of the <code>versionHistoryList</code> parameter.
    * @param versionHistoryList A list of {@link VersionHistory} objects that
    * are being located.
    * @param wantedPropertyList The properties to be retrieved for the
    * returned version-controlled resources.
    */
   public List doLocateByHistoryReport(
      List versionHistoryList,
      PropertyNameList wantedPropertyList) throws WvcmException;

   /**
    * Merge the versions specified in the <code>sourceList</code> into
    * this {@link ControllableResource}.
    * The "target" of a particular source version is the version-controlled 
    * member of the workspace identified by this {@link Workspace}
    * that is associated with the version history for that version.
    * @return An iterator of {@link ControllableResource} objects that
    * identifies the resources modified by the merge.
    * @param sourceList A list of {@link Resource} objects that
    * identifies the versions to be merged.
    * If a member of sourceList is an activity,
    * each version in the ActivityVersionList is a source version.
    * If a member of sourceList is a version-controlled resource,
    * the checked-in version of the member is a source version.
    * If a member of sourceList is a folder, all members of the configuration
    * rooted at that folder are considered members of the sourceList.
    * @param checkinActivity If a member of the sourceList is an activity,
    * apply doCheckin to that activity before performing the merge.
    * @param noAutoMerge See ControllableResource#doMerge.
    * @param noCheckout See ControllableResource#doMerge.
    * @param forkOK See ControllableResource#doMerge.
    * @param unreserved See ControllableResource#doMerge.
    * @param activityList See ControllableResource#doMerge.
    * @param newActivity See ControllableResource#doMerge.
    * @param wantedPropertyList See ControllableResource#doMerge.
    * 
    * @throws WvcmException Preconditions:
    * <br>(cannot-merge-checked-out-resource): The sourceList argument member
    *  MUST NOT identify a checked-out resource.  If the sourceList argument member
    *  identifies a folder, the folder MUST NOT have a member that is a checked-out resource.
    * 
    * @throws WvcmException Postconditions:
    * <br>(merge-baseline): If the merge target is a version-controlled configuration
    *  whose CheckedOut baseline is not a descendant of the merge baseline,
    *  then the merge baseline MUST have been added to the AutoMergeList
    *  of a version-controlled configuration. 
    *  The CheckedIn version of each member of the BaselineFolder of that baseline
    *  MUST have been merged into the RootFolder of that version-controlled configuration.
    * <br>(merge-sub-baselines): If the merge target is a version-controlled configuration
    *  whose RootFolder contains a baseline-controlled member for one of the sub-baselines
    *  of the merge baseline, then that sub-baseline MUST have been merged into
    *  the version-controlled configuration of that baseline-controlled member. 
    *  If the merge target is a version-controlled configuration whose RootFolder
    *  is a member of a workspace that contains a baseline-controlled member
    *  for one of the sub-baselines of the merge baseline, then that sub-baseline
    *  MUST have been merged into the version-controlled configuration of that baseline-controlled member.
    * <br>(set-baseline-controlled-folder-members): Same semantics as doUpdate.
    */
   public Iterator doMerge(
      List sourceList,
      boolean checkinActivity,
      boolean noAutoMerge,
      boolean noCheckout,
      boolean forkOK,
      boolean unreserved,
      List activityList,
      boolean newActivity,
      PropertyNameList wantedPropertyList)
      throws WvcmException;

   /**
    * Return an iterator of MergePreviewReport objects,
    * indicating what resources would be modified by a
    * {@link #doMerge doMerge}.
    * @param sourceList A list of {@link Resource} objects that identify
    * the versions to be merged.
    */
   public Iterator doMergePreviewReport(List sourceList)
      throws WvcmException;

}
