package javax.wvcm;

import java.util.HashMap;

/**
 * Signals that an failure of some sort has occurred in the WVCM provider.
 * A WvcmException contains a code describing the reason for the failure
 * and optionally a nested exception to further explain the cause of the error.
 * Some of the reason codes were based on pre or post condition failures
 * as defined in <a href="http://www.ietf.org/rfc/rfc3253.txt">
 * Versioning Extensions to WebDAV</a>.
 */
public class WvcmException extends Exception {
    
    public static class ReasonCode {
        
        /** The provider suffered an I/O failure, the operation may be retried. */
        public static final ReasonCode READ_FAILED
            = new ReasonCode(){public String toString(){return "READ_FAILED";}};

        /** The provider suffered an I/O failure, the operation may be retried. */
        public static final ReasonCode WRITE_FAILED
            = new ReasonCode(){public String toString(){return "WRITE_FAILED";}};

        /** The property value is unavailable because it was not in the property name
         * list when the proxy was created. */
        public static final ReasonCode VALUE_UNAVAILABLE
            = new ReasonCode(){public String toString(){return "VALUE_UNAVAILABLE";}};

        /** The property value is maintained only on the server and so is not available
         * locally. */
        public static final ReasonCode VALUE_UNAVAILABLE_LOCALLY
            = new ReasonCode(){public String toString(){return "VALUE_UNAVAILABLE_LOCALLY";}};

        /** The requested property does not exist on the persistent resource. */
        public static final ReasonCode PROPERTY_MISSING
            = new ReasonCode(){public String toString(){return "PROPERTY_MISSING";}};

        /** The resource has no content. */
        public static final ReasonCode NO_CONTENT
            = new ReasonCode(){public String toString(){return "NO_CONTENT";}};

        /** The resource has no content stored locally. */
        public static final ReasonCode NO_LOCAL_CONTENT
            = new ReasonCode(){public String toString(){return "NO_LOCAL_CONTENT";}};

        /** The property value update would overwrite an earlier change. */
        public static final ReasonCode PROPERTY_OVERWRITE_FORBIDDEN
            = new ReasonCode(){public String toString(){return "PROPERTY_OVERWRITE_FORBIDDEN";}};

        /** The user is not authorized to execute the attempted operation.*/
        public static final ReasonCode UNAUTHORIZED
            = new ReasonCode(){public String toString(){return "UNAUTHORIZED";}};

        /** The provider was unable to complete the operation for an unspecified reason. */
        public static final ReasonCode FORBIDDEN
            = new ReasonCode(){public String toString(){return "FORBIDDEN";}};

        /** The corresponding remote resource no longer exists or was never created. */
        public static final ReasonCode NOT_FOUND
            = new ReasonCode(){public String toString(){return "NOT_FOUND";}};

        /** The operation cannot be performed due to a conflict with resource state. */
        public static final ReasonCode CONFLICT
            = new ReasonCode(){public String toString(){return "CONFLICT";}};

        /** The server workspace location is required to perform the operation. */
        public static final ReasonCode WORKSPACE_SERVER_LOCATION_UNDEFINED
            = new ReasonCode(){public String toString(){return "WORKSPACE_SERVER_LOCATION_UNDEFINED";}};

        /** <code>Precondition:</code> Resource specified was not an activity. */
        public static final ReasonCode NOT_AN_ACTIVITY
            = new ReasonCode(){public String toString(){return "NOT_AN_ACTIVITY";}};

        /** <code>Postcondition:</code> The operation failed to delete all versions in the version set. */
        public static final ReasonCode CANNOT_DELETE_VERSION_LIST
            = new ReasonCode(){public String toString(){return "CANNOT_DELETE_VERSION_LIST";}};

        /** <code>Postcondition:</code> Failed to initialize the bindings in the folder. */
        public static final ReasonCode CANNOT_INITIALIZE_BINDINGS
            = new ReasonCode(){public String toString(){return "CANNOT_INITIALIZE_BINDINGS";}};

        /** <code>Precondition:</code> This folder already has a controlled configuration. */
        public static final ReasonCode CONTROLLED_CONFIGURATION_ALREADY_EXISTS
            = new ReasonCode(){public String toString(){return "CONTROLLED_CONFIGURATION_ALREADY_EXISTS";}};

        /** <code>Postcondition:</code> Failed to create a new controlled configuration. */
        public static final ReasonCode CANNOT_CREATE_CONTROLLED_CONFIGURATION
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_CONTROLLED_CONFIGURATION";}};

        /** <code>Postcondition:</code> Failed to create reference to a new controlled configuration. */
        public static final ReasonCode CANNOT_REFERENCE_CONTROLLED_CONFIGURATION
            = new ReasonCode(){public String toString(){return "CANNOT_REFERENCE_CONTROLLED_CONFIGURATION";}};

        /** <code>Postcondition:</code> The new version could not be added to the version history. */
        public static final ReasonCode CANNOT_ADD_TO_HISTORY
            = new ReasonCode(){public String toString(){return "CANNOT_ADD_TO_HISTORY";}};

        /** <code>Postcondition:</code> he workspace could not be deleted because members still reference it. */
        public static final ReasonCode CANNOT_DELETE_WORKSPACE_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_DELETE_WORKSPACE_MEMBERS";}};

        /** <code>Precondition:</code> A version resource cannot be renamed. */
        public static final ReasonCode CANNOT_RENAME_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_RENAME_VERSION";}};

        /** <code>Postcondition:</code> The specified properties cannot be reported. */
        public static final ReasonCode CANNOT_REPORT_PROPERTIES
            = new ReasonCode(){public String toString(){return "CANNOT_REPORT_PROPERTIES";}};

        /** <code>Postcondition:</code> The activity set of the checked-out resource could not be initialized. */
        public static final ReasonCode CANNOT_INITIALIZE_ACTIVITY_LIST
            = new ReasonCode(){public String toString(){return "CANNOT_INITIALIZE_ACTIVITY_LIST";}};

        /** <code>Precondition:</code> A fork in the version tree is not allowed. */
        public static final ReasonCode CANNOT_FORK
            = new ReasonCode(){public String toString(){return "CANNOT_FORK";}};

        /** <code>Postcondition:</code> The folders controlled configuration property could not be set. */
        public static final ReasonCode CANNOT_SET_CONTROLLED_CONFIGURATION
            = new ReasonCode(){public String toString(){return "CANNOT_SET_CONTROLLED_CONFIGURATION";}};

        /** <code>Postcondition:</code> The uncheckout operation could not restore content and properties. */
        public static final ReasonCode CANNOT_RESTORE_CONTENT_AND_DEAD_PROPERTIES
            = new ReasonCode(){public String toString(){return "CANNOT_RESTORE_CONTENT_AND_DEAD_PROPERTIES";}};

        /** <code>Precondition:</code> The members of a baseline folder cannot be modified. */
        public static final ReasonCode CANNOT_UPDATE_BASELINE_FOLDER
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_BASELINE_FOLDER";}};

        /** <code>Precondition:</code> A baseline controlled folder already exists in this workspace for this baseline history. */
        public static final ReasonCode CANNOT_HAVE_MULTIPLE_BASELINE_CONTROLLED_FOLDERS
            = new ReasonCode(){public String toString(){return "CANNOT_HAVE_MULTIPLE_BASELINE_CONTROLLED_FOLDERS";}};

        /** <code>Precondition:</code> Cannot create this resource at the specified location. */
        public static final ReasonCode CANNOT_CREATE_AT_THIS_LOCATION
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_AT_THIS_LOCATION";}};

        /** <code>Postcondition:</code> Update of the merge set or auto merge set properties failed. */
        public static final ReasonCode CANNOT_UPDATE_MERGE_LIST
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_MERGE_LIST";}};

        /** <code>Precondition:</code> Cannot modify the properties of a checked-in resource. */
        public static final ReasonCode CANNOT_MODIFY_CONTROLLED_PROPERTY
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_CONTROLLED_PROPERTY";}};

        /**
         * <code>Precondition:</code> Failed to checkin this resource because auto update does not
         * reference a controlled resource whose checked-in version is not the
         * same as, or a predecessor of this resource. */
        public static final ReasonCode CANNOT_OVERWRITE_BY_AUTO_UPDATE
            = new ReasonCode(){public String toString(){return "CANNOT_OVERWRITE_BY_AUTO_UPDATE";}};

        /** <code>Precondition:</code> Failed to perform the merge because the target could not be checked-out. */
        public static final ReasonCode CANNOT_MERGE_CHECKOUT_NOT_ALLOWED
            = new ReasonCode(){public String toString(){return "CANNOT_MERGE_CHECKOUT_NOT_ALLOWED";}};

        /** <code>Postcondition:</code> Failed to create working resource. */
        public static final ReasonCode CANNOT_CREATE_WORKING_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_WORKING_RESOURCE";}};

        /** <code>Postcondition:</code> Failed to checkin the folder version. */
        public static final ReasonCode CANNOT_CONTROL_FOLDER_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_CONTROL_FOLDER_MEMBERS";}};

        /** <code>Precondition:</code> Cannot modify the content of a checked-in resource. */
        public static final ReasonCode CANNOT_MODIFY_CONTROLLED_CONTENT
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_CONTROLLED_CONTENT";}};

        /** <code>Precondition:</code> A version history cannot be renamed. */
        public static final ReasonCode CANNOT_RENAME_HISTORY
            = new ReasonCode(){public String toString(){return "CANNOT_RENAME_HISTORY";}};

        /** <code>Postcondition:</code> A vesioning property was copied to the destination. */
        public static final ReasonCode VERSIONING_PROPERTY_COPIED
            = new ReasonCode(){public String toString(){return "VERSIONING_PROPERTY_COPIED";}};

        /** <code>Postcondition:</code> Failed to checkin all checked-out resources in the activity checkout set. */
        public static final ReasonCode CANNOT_CHECKIN_ACTIVITY_RESOURCES
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKIN_ACTIVITY_RESOURCES";}};

        /** <code>Postcondition:</code> Operation failed because references to the resource could not be updated. */
        public static final ReasonCode CANNOT_UPDATE_CHECKED_OUT_REFERENCE
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_CHECKED_OUT_REFERENCE";}};

        /** <code>Precondition:</code> Operation failed because the resource passed in was not a baseline. */
        public static final ReasonCode NOT_A_BASELINE
            = new ReasonCode(){public String toString(){return "NOT_A_BASELINE";}};

        /**
         * <code>Precondition:</code> Failed to checkout the resource because a descendent exists and forking
         * is forbidden.
         */
        public static final ReasonCode CANNOT_CHECKOUT_FORKING_IS_FORBIDDEN
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKOUT_FORKING_IS_FORBIDDEN";}};

        /**
         * <code>Precondition:</code> Failed to checkout resource because multiple checkout is discouraged
         * and the caller did not specify fork-ok.
         */
        public static final ReasonCode CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_DISCOURAGED
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_DISCOURAGED";}};

        /** <code>Postcondition:</code> Failed to move a resource into a new workspace. */
        public static final ReasonCode CANNOT_MOVE_INTO_WORKSPACE
            = new ReasonCode(){public String toString(){return "CANNOT_MOVE_INTO_WORKSPACE";}};

        /** <code>Postcondition:</code> Deleting a version failed to update the root of the version history. */
        public static final ReasonCode CANNOT_UPDATE_VERSION_HISTORY_ROOT
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_VERSION_HISTORY_ROOT";}};

        /** <code>Precondition:</code> Failed to retrieve a property that should be supported. */
        public static final ReasonCode CANNOT_RETRIEVE_SUPPORTED_PROPERTY
            = new ReasonCode(){public String toString(){return "CANNOT_RETRIEVE_SUPPORTED_PROPERTY";}};

        /** <code>Precondition:</code> Checkout of an already checked-out resource is forbidden. */
        public static final ReasonCode CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_FORBIDDEN
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_FORBIDDEN";}};

        /** <code>Postcondition:</code> Checkout failed to initialize the predecessor set of the checked-out resource. */
        public static final ReasonCode CANNOT_INITIALIZE_PREDECESSOR_LIST
            = new ReasonCode(){public String toString(){return "CANNOT_INITIALIZE_PREDECESSOR_LIST";}};

        /** <code>Postcondition:</code> Unlock failed to automatically checkin the resource. */
        public static final ReasonCode AUTO_CHECKIN_FAILED
            = new ReasonCode(){public String toString(){return "AUTO_CHECKIN_FAILED";}};

        /** <code>Precondition:</code> Cannot remove the specified label because it does not used by this resource. */
        public static final ReasonCode CANNOT_REMOVE_LABEL_DOES_NOT_EXIST
            = new ReasonCode(){public String toString(){return "CANNOT_REMOVE_LABEL_DOES_NOT_EXIST";}};

        /** <code>Precondition:</code> Report failed because a resource specified was not a version history. */
        public static final ReasonCode BAD_VERSION_HISTORY
            = new ReasonCode(){public String toString(){return "BAD_VERSION_HISTORY";}};

        /** <code>Postcondition:</code> Workspace was moved but references to it failed to be updated. */
        public static final ReasonCode CANNOT_UPDATE_WORKSPACE_REFERENCE
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_WORKSPACE_REFERENCE";}};

        /** <code>Precondition:</code> Cannot checkin since it would cause a fork and forking is discouraged */
        public static final ReasonCode CANNOT_CHECKIN_FORK_DISCOURAGED
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKIN_FORK_DISCOURAGED";}};

        /** <code>Postcondition:</code> Controlled folder was updated but members failed to be updated. */
        public static final ReasonCode CANNOT_UPDATE_CONTROLLED_FOLDER_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_CONTROLLED_FOLDER_MEMBERS";}};

        /** <code>Postcondition:</code> Checkin of a controlled configuration failed to create a new baseline folder. */
        public static final ReasonCode CANNOT_CREATE_BASELINE_FOLDER
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_BASELINE_FOLDER";}};

        /** <code>Postcondition:</code> Failed to preserve the properties when moving a resource. */
        public static final ReasonCode CANNOT_PRESERVE_VERSIONING_PROPERTIES
            = new ReasonCode(){public String toString(){return "CANNOT_PRESERVE_VERSIONING_PROPERTIES";}};

        /** <code>Postcondition:</code> Failed to merge baseline. */
        public static final ReasonCode CANNOT_MERGE_BASELINE
            = new ReasonCode(){public String toString(){return "CANNOT_MERGE_BASELINE";}};

        /** <code>Postcondition:</code> Failed to put the resource under control. */
        public static final ReasonCode CANNOT_PUT_UNDER_CONTROL
            = new ReasonCode(){public String toString(){return "CANNOT_PUT_UNDER_CONTROL";}};

        /** <code>Precondition:</code> Delete failed because you cannot delete a version. */
        public static final ReasonCode CANNOT_DELETE_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_DELETE_VERSION";}};

        /** <code>Postcondition:</code> Adding or setting the specified label failed. */
        public static final ReasonCode CANNOT_ADD_OR_SET_LABEL
            = new ReasonCode(){public String toString(){return "CANNOT_ADD_OR_SET_LABEL";}};

        /** <code>Precondition:</code> Uncheckout failed, resource was not a checked-out controlled resource. */
        public static final ReasonCode CANNOT_UNCHECKOUT_NOT_CHECKED_OUT_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_UNCHECKOUT_NOT_CHECKED_OUT_RESOURCE";}};

        /** <code>Postcondition:</code> Cannot correctly set the members of the baseline controlled folder. */
        public static final ReasonCode CANNOT_SET_BASELINE_CONTROLLED_FOLDER_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_SET_BASELINE_CONTROLLED_FOLDER_MEMBERS";}};

        /** <code>Precondition:</code> resource specified must be a version. */
        public static final ReasonCode BAD_VERSION
            = new ReasonCode(){public String toString(){return "BAD_VERSION";}};

        /**
         * <code>Precondition:</code> Failed to modify content/properties because the resource specified
         * was a version.
         */
        public static final ReasonCode CANNOT_MODIFY_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_VERSION";}};

        /**
         * <code>Precondition:</code> The operation failed because the server does not allow
         * compare/merge of baselines from different baseline histories.
         */
        public static final ReasonCode BASELINES_FROM_SAME_HISTORY
            = new ReasonCode(){public String toString(){return "BASELINES_FROM_SAME_HISTORY";}};

        /**
         * <code>Precondition:</code> Activity cannot be checked-in because checkin of all
         * referenced resources failed.
         */
        public static final ReasonCode CANNOT_CHECKIN_ALL_RESOURCES
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKIN_ALL_RESOURCES";}};

        /** <code>Precondition:</code> Failed to move because the resource to be moved is a working resource. */
        public static final ReasonCode CANNOT_RENAME_WORKING_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_RENAME_WORKING_RESOURCE";}};

        /** <code>Postcondition:</code> Failed to create a new controlled resource. */
        public static final ReasonCode CANNOT_CREATE_NEW_CONTROLLED_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_NEW_CONTROLLED_RESOURCE";}};

        /** <code>Postcondition:</code> Baseline control failed to create members or set properties. */
        public static final ReasonCode CANNOT_SELECT_EXISTING_BASELINE
            = new ReasonCode(){public String toString(){return "CANNOT_SELECT_EXISTING_BASELINE";}};

        /** <code>Precondition:</code> Creating a resource failed because a resource already exists at the specified location. */
        public static final ReasonCode RESOURCE_ALREADY_EXISTS_AT_LOCATION
            = new ReasonCode(){public String toString(){return "RESOURCE_ALREADY_EXISTS_AT_LOCATION";}};

        /** <code>Postcondition:</code> The report failed because it modified resources properties. */
        public static final ReasonCode CANNOT_MODIFY_PROPERTIES
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_PROPERTIES";}};

        /** <code>Postcondition:</code> The operation failed to modify the resource because auto checkout failed. */
        public static final ReasonCode CANNOT_AUTO_CHECKOUT
            = new ReasonCode(){public String toString(){return "CANNOT_AUTO_CHECKOUT";}};

        /** <code>Precondition:</code> Failed to checkout because descendent already exists and forking is discouraged. */
        public static final ReasonCode CHECKOUT_OF_VERSION_WITH_DESCENDANT_IS_DISCOURAGED
            = new ReasonCode(){public String toString(){return "CHECKOUT_OF_VERSION_WITH_DESCENDANT_IS_DISCOURAGED";}};

        public static final ReasonCode CANNOT_UPDATE_ACTIVITY_REFERENCE
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_ACTIVITY_REFERENCE";}};

        /**
         * <code>Precondition:</code> Cannot checkin because the resources predecessors are not descendents
         * of the root of the version history.
         */
        public static final ReasonCode VERSION_HISTORY_IS_NOT_TREE
            = new ReasonCode(){public String toString(){return "VERSION_HISTORY_IS_NOT_TREE";}};

        /**
         * <code>Precondition:</code> Label failed to select a version in the history of the
         * specified controlled resource.
         */
        public static final ReasonCode MUST_SELECT_VERSION_IN_HISTORY
            = new ReasonCode(){public String toString(){return "MUST_SELECT_VERSION_IN_HISTORY";}};

        /** <code>Precondition:</code> Failed because the resource specified is a folder version. */
        public static final ReasonCode CANNOT_COPY_FOLDER_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_COPY_FOLDER_VERSION";}};

        /** <code>Postcondition:</code> Update failed to set content and properties correctly. */
        public static final ReasonCode CANNOT_UPDATE_CONTENT_AND_PROPERTIES
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_CONTENT_AND_PROPERTIES";}};

        /** <code>Precondition:</code> baseline control failed because the folder already has controlled resources. */
        public static final ReasonCode CANNOT_HAVE_CONTROLLED_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_HAVE_CONTROLLED_MEMBERS";}};

        /** <code>Postcondition:</code> The operation changed an existing controlled resource. */
        public static final ReasonCode CANNOT_CHANGE_EXISTING_CHECKED_IN_OUT
            = new ReasonCode(){public String toString(){return "CANNOT_CHANGE_EXISTING_CHECKED_IN_OUT";}};

        /** <code>Postcondition:</code> Target version is an ancestor of the source but update failed. */
        public static final ReasonCode CANNOT_UPDATE_DESCENDANT_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_DESCENDANT_VERSION";}};

        /** <code>Postcondition:</code> Rebind failed to update references to the resource in auto update properties. */
        public static final ReasonCode CANNOT_UPDATE_AUTO_UPDATE
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_AUTO_UPDATE";}};

        /** <code>Precondition:</code> Copy failed because you cannot copy a history resource. */
        public static final ReasonCode CANNOT_COPY_HISTORY
            = new ReasonCode(){public String toString(){return "CANNOT_COPY_HISTORY";}};

        /** <code>Postcondition:</code> Checkout operation failed to checkout the controlled resource. */
        public static final ReasonCode CANNOT_CHECK_OUT
            = new ReasonCode(){public String toString(){return "CANNOT_CHECK_OUT";}};

        /**
         * <code>Precondition:</code> The operation failed because it would result in more than one controlled
         * resource for this version history in a workspace.
         * */
        public static final ReasonCode ONE_CONTROLLED_RESOURCE_PER_HISTORY_PER_WORKSPACE
            = new ReasonCode(){public String toString(){return "ONE_CONTROLLED_RESOURCE_PER_HISTORY_PER_WORKSPACE";}};

        public static final ReasonCode INITIALIZE_CONTROLLED_BINDINGS
            = new ReasonCode(){public String toString(){return "INITIALIZE_CONTROLLED_BINDINGS";}};

        /**
         * <code>Postcondition:</code> Update request failed because member of baseline controlled folder
         * could not be updated.
         */
        public static final ReasonCode CANNOT_MODIFY_CONFIGURATION
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_CONFIGURATION";}};

        /** <code>Precondition:</code> Update failed because members of the controlled configuration are checked-out. */
        public static final ReasonCode CANNOT_UPDATE_MEMBERS_ARE_CHECKED_OUT
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_MEMBERS_ARE_CHECKED_OUT";}};

        /**
         * <code>Postcondition:</code> Operation could not be carried out on the version
         * identified by the specified label.
         */
        public static final ReasonCode APPLY_REQUEST_TO_LABELED_VERSION
            = new ReasonCode(){public String toString(){return "APPLY_REQUEST_TO_LABELED_VERSION";}};

        /** <code>Postcondition:</code> Unbind failed to remove all outstanding merge references to the resource. */
        public static final ReasonCode CANNOT_DELETE_VERSION_REFERENCE
            = new ReasonCode(){public String toString(){return "CANNOT_DELETE_VERSION_REFERENCE";}};

        public static final ReasonCode CANNOT_MERGE_SUB_BASELINES
            = new ReasonCode(){public String toString(){return "CANNOT_MERGE_SUB_BASELINES";}};

        /** <code>Precondition:</code> Merge failed because source identified or contains a checked-out resource. */
        public static final ReasonCode CANNOT_MERGE_CHECKED_OUT_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_MERGE_CHECKED_OUT_RESOURCE";}};

        /** <code>Postcondition:</code> Uncheckout failed to reset the state of the controlled resource. */
        public static final ReasonCode CANNOT_CANCEL_CHECKED_OUT
            = new ReasonCode(){public String toString(){return "CANNOT_CANCEL_CHECKED_OUT";}};

        /** <code>Postcondition:</code> Unbind failed to update the predecessor set of the version history. */
        public static final ReasonCode CANNOT_UPDATE_PREDECESSOR_LIST
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_PREDECESSOR_LIST";}};

        /**
         * <code>Precondition:</code> Failed because more than one version of this resource is referenced
         * in the specified activity.
         */
        public static final ReasonCode ONE_CHECKOUT_PER_ACTIVITY_PER_HISTORY
            = new ReasonCode(){public String toString(){return "ONE_CHECKOUT_PER_ACTIVITY_PER_HISTORY";}};

        /** <code>Precondition:</code> The operation failed because the resource must be in the checked-in state. */
        public static final ReasonCode MUST_BE_CHECKED_IN
            = new ReasonCode(){public String toString(){return "MUST_BE_CHECKED_IN";}};

        /** <code>Precondition:</code> Could not update all resources matching the depth header. */
        public static final ReasonCode CANNOT_DEPTH_UPDATE
            = new ReasonCode(){public String toString(){return "CANNOT_DEPTH_UPDATE";}};

        /** <code>Postcondition:</code> The resource could not be checked-in. */
        public static final ReasonCode CANNOT_CHECK_IN
            = new ReasonCode(){public String toString(){return "CANNOT_CHECK_IN";}};

        /** <code>Precondition:</code> The operation failed because the resource must first be checked-out. */
        public static final ReasonCode MUST_BE_CHECKED_OUT
            = new ReasonCode(){public String toString(){return "MUST_BE_CHECKED_OUT";}};

        /**
         * <code>Postcondition:</code> Cannot unbind activity because references to the activity could
         * not be deleted.
         */
        public static final ReasonCode CANNOT_UNBIND_ACTIVITY_REFERENCE
            = new ReasonCode(){public String toString(){return "CANNOT_UNBIND_ACTIVITY_REFERENCE";}};

        /** <code>Precondition:</code> The resource being checked-in is in the process of being merged. */
        public static final ReasonCode CANNOT_CHECKIN_MERGE_NOT_COMPLETE
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKIN_MERGE_NOT_COMPLETE";}};

        /** <code>Precondition:</code> Each version in an activity for a given version history
         * must be on the same line of descent. */
        public static final ReasonCode CANNOT_CREATE_BRANCH_IN_ACTIVITY
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_BRANCH_IN_ACTIVITY";}};

        /**
         * <code>Precondition:</code> The operation failed because the resource is a member of a
         * checked-in controlled configuration and auto versioning is not enabled.
         */
        public static final ReasonCode CANNOT_MODIFY_CONTROLLED_CONFIGURATION
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_CONTROLLED_CONFIGURATION";}};

        /**
         * <code>Precondition:</code> Cannot checkin the controlled configuration because more than one
         * member exists for a given version history.
         */
        public static final ReasonCode ONE_VERSION_PER_HISTORY_PER_BASELINE
            = new ReasonCode(){public String toString(){return "ONE_VERSION_PER_HISTORY_PER_BASELINE";}};

        public static final ReasonCode CANNOT_CREATE_NEW_VERSION_HISTORY
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_NEW_VERSION_HISTORY";}};

        /** <code>Postcondition:</code> Copy failed to create new resource when copying a controlled resource. */
        public static final ReasonCode CANNOT_CREATE_NEW_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_NEW_RESOURCE";}};

        /** <code>Postcondition:</code> Merge modified an ancestor of the merge source. */
        public static final ReasonCode CANNOT_MERGE_ANCESTOR_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_MERGE_ANCESTOR_VERSION";}};

        /** <code>Postcondition:</code> Checkin failed to update a controlled resource specified in the auto update property. */
        public static final ReasonCode CANNOT_AUTO_UPDATE
            = new ReasonCode(){public String toString(){return "CANNOT_AUTO_UPDATE";}};

        /** <code>Postcondition:</code> Target needed to be checked-out for merge but it failed. */
        public static final ReasonCode CANNOT_CHECKOUT_FOR_MERGE
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKOUT_FOR_MERGE";}};

        public static final ReasonCode CANNOT_KEEP_CHECKED_OUT
            = new ReasonCode(){public String toString(){return "CANNOT_KEEP_CHECKED_OUT";}};

        /** <code>Precondition:</code> The label is already in use by this resource. */
        public static final ReasonCode ADD_MUST_BE_NEW_LABEL
            = new ReasonCode(){public String toString(){return "ADD_MUST_BE_NEW_LABEL";}};

        /**
         * <code>Precondition:</code> Operation failed to modify a resource because it's parent folder is a
         * checked-in controlled folder.
         */
        public static final ReasonCode CANNOT_MODIFY_CHECKED_IN_PARENT
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_CHECKED_IN_PARENT";}};

        /**
         * <code>Precondition:</code> Failed to checkin the controlled configuration because some of it's
         * members are still checked-out.
         */
        public static final ReasonCode NO_CHECKED_OUT_BASELINE_CONTROLLED_FOLDER_MEMBERS
            = new ReasonCode(){public String toString(){return "NO_CHECKED_OUT_BASELINE_CONTROLLED_FOLDER_MEMBERS";}};

        /** <code>Precondition:</code> Operation failed because it attempted to set a protected property. */
        public static final ReasonCode CANNOT_MODIFY_PROTECTED_PROPERTY
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_PROTECTED_PROPERTY";}};

        /** <code>Postcondition:</code> Update failed to correctly set the checked-in property of controlled configurations. */
        public static final ReasonCode CANNOT_UPDATE_SUB_BASELINES
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_SUB_BASELINES";}};

        /** <code>Precondition:</code> Rebind failed because the destination is a checked-in controlled folder. */
        public static final ReasonCode CANNOT_MODIFY_DESTINATION_CHECKED_IN_PARENT
            = new ReasonCode(){public String toString(){return "CANNOT_MODIFY_DESTINATION_CHECKED_IN_PARENT";}};

        /** <code>Postcondition:</code> Workspace was moved but failed to update all references to the workspace. */
        public static final ReasonCode CANNOT_MOVE_WORKSPACE
            = new ReasonCode(){public String toString(){return "CANNOT_MOVE_WORKSPACE";}};

        /** <code>Postcondition:</code> The request to modify the resource failed to auto checkout and checkin the resource. */
        public static final ReasonCode CANNOT_AUTO_CHECKOUT_CHECKIN
            = new ReasonCode(){public String toString(){return "CANNOT_AUTO_CHECKOUT_CHECKIN";}};

        /** <code>Postcondition:</code> Baseline control failed to create a new baseline. */
        public static final ReasonCode CANNOT_CREATE_NEW_BASELINE
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_NEW_BASELINE";}};

        /** <code>Postcondition:</code> Failed to remove label from the specified resource. */
        public static final ReasonCode CANNOT_REMOVE_LABEL
            = new ReasonCode(){public String toString(){return "CANNOT_REMOVE_LABEL";}};

        /** <code>Precondition:</code> Request failed because both a label and apply to version were specified. */
        public static final ReasonCode CANNOT_HAVE_LABEL_AND_APPLY_TO_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_HAVE_LABEL_AND_APPLY_TO_VERSION";}};

        /** <code>Postcondition:</code> Failed to create a working resource based on checked-in resource. */
        public static final ReasonCode CANNOT_APPLY_TO_CHECKED_IN_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_APPLY_TO_CHECKED_IN_VERSION";}};

        /** <code>Postcondition:</code> Checkin failed to create a new version. */
        public static final ReasonCode CANNOT_CREATE_VERSION
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_VERSION";}};

        /** <code>Precondition:</code> Report failed since the resource does not support the specified report. */
        public static final ReasonCode BAD_REPORT
            = new ReasonCode(){public String toString(){return "BAD_REPORT";}};

        /** <code>Postcondition:</code> Failed to unbind the resource. */
        public static final ReasonCode CANNOT_UNBIND_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_UNBIND_RESOURCE";}};

        /** <code>Postcondition:</code> Failed to checkin all checked-out resources in the activity checkout set. */
        public static final ReasonCode CANNOT_CHECKIN_ACTIVITY
            = new ReasonCode(){public String toString(){return "CANNOT_CHECKIN_ACTIVITY";}};

        /** <code>Postcondition:</code> Could not set the unreserved property of the resource. */
        public static final ReasonCode CANNOT_INITIALIZE_UNRESERVED
            = new ReasonCode(){public String toString(){return "CANNOT_INITIALIZE_UNRESERVED";}};

        /** Illegal syntax for location string value */
        public static final ReasonCode ILLEGAL_LOCATION_SYNTAX
            = new ReasonCode(){public String toString(){return "ILLEGAL_LOCATION_SYNTAX";}};

        /** Cannot refresh dirty resource without ignoreDirty flag set */
        public static final ReasonCode CANNOT_REFRESH_DIRTY_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_REFRESH_DIRTY_RESOURCE";}};

        /** Operation cannot be performed if the resource content is stale */
        public static final ReasonCode MUST_NOT_BE_STALE
            = new ReasonCode(){public String toString(){return "MUST_NOT_BE_STALE";}};

        /** Attempt to synchronize content failed */
        public static final ReasonCode CANNOT_SYNCHRONIZE_CONTENT
            = new ReasonCode(){public String toString(){return "CANNOT_SYNCHRONIZE_CONTENT";}};

        /** The new resource could not be properly initialized */
        public static final ReasonCode CANNOT_INITIALIZE_RESOURCE
            = new ReasonCode(){public String toString(){return "CANNOT_INITIALIZE_RESOURCE";}};

        /** Cannot read members of folder */
        public static final ReasonCode CANNOT_READ_MEMBERS
            = new ReasonCode(){public String toString(){return "CANNOT_READ_MEMBERS";}};

        /** Checkin cannot update the server content */
        public static final ReasonCode CANNOT_UPDATE_SERVER_CONTENT
            = new ReasonCode(){public String toString(){return "CANNOT_UPDATE_SERVER_CONTENT";}};

        /** Not Cannot create multiple bindings to one resource */
        public static final ReasonCode MULTIPLE_BINDINGS_NOT_ALLOWED
            = new ReasonCode(){public String toString(){return "MULTIPLE_BINDINGS_NOT_ALLOWED";}};

        /** Cannot create cross-server binding */
        public static final ReasonCode NO_CROSS_SERVER_BINDING
            = new ReasonCode(){public String toString(){return "NO_CROSS_SERVER_BINDING";}};

        /** Cannot overwrite existing binding */
        public static final ReasonCode CANNOT_OVERWRITE
            = new ReasonCode(){public String toString(){return "CANNOT_OVERWRITE";}};

        /** Cannot create location cycle */
        public static final ReasonCode CYCLE_NOT_ALLOWED
            = new ReasonCode(){public String toString(){return "CYCLE_NOT_ALLOWED";}};

        /** Cannot create specified binding */
        public static final ReasonCode CANNOT_CREATE_BINDING
            = new ReasonCode(){public String toString(){return "CANNOT_CREATE_BINDING";}};

        // NOT YET STANDARD - ACL
        public static final ReasonCode ACE_CONFLICT
            = new ReasonCode(){public String toString(){return "ACE_CONFLICT";}};
        public static final ReasonCode DENY_BEFORE_GRANT
            = new ReasonCode(){public String toString(){return "DENY_BEFORE_GRANT";}};
        public static final ReasonCode GRANT_ONLY
            = new ReasonCode(){public String toString(){return "GRANT_ONLY";}};
        public static final ReasonCode PRIVILEGE_NOT_SUPPORTED
            = new ReasonCode(){public String toString(){return "PRIVILEGE_NOT_SUPPORTED";}};
        public static final ReasonCode MISSING_REQUIRED_PRINCIPAL
            = new ReasonCode(){public String toString(){return "MISSING_REQUIRED_PRINCIPAL";}};
        public static final ReasonCode PRINCIPAL_NOT_FOUND
            = new ReasonCode(){public String toString(){return "PRINCIPAL_NOT_FOUND";}};
        public static final ReasonCode INVALID_PRINCIPAL
            = new ReasonCode(){public String toString(){return "INVALID_PRINCIPAL";}};

        // NOT YET STANDARD - Locking
        public static final ReasonCode LOCKED
            = new ReasonCode(){public String toString(){return "LOCKED";}};
        public static final ReasonCode CANNOT_LOCK
            = new ReasonCode(){public String toString(){return "CANNOT_LOCK";}};
        public static final ReasonCode CANNOT_UNLOCK
            = new ReasonCode(){public String toString(){return "CANNOT_UNLOCK";}};
        
        
        /** Map of RFC 3253 pre & post condition names to WvcmException.ReasonCode */
        private static final HashMap _rfc3253ToWvcmMap = new HashMap();
        
        static {
            
            //  VERSION-CONTROL
            _rfc3253ToWvcmMap.put( "put-under-version-control",                 CANNOT_PUT_UNDER_CONTROL );
            _rfc3253ToWvcmMap.put( "must-not-change-existing-checked-in-out",   CANNOT_CHANGE_EXISTING_CHECKED_IN_OUT );
            
            // REPORT
            _rfc3253ToWvcmMap.put( "supported-report",                          BAD_REPORT );
            //(acceptable-depth)
            _rfc3253ToWvcmMap.put( "no-modification",                           CANNOT_MODIFY_PROPERTIES);
            
            // PUT
            _rfc3253ToWvcmMap.put( "cannot-modify-version-controlled-content",  CANNOT_MODIFY_CONTROLLED_CONTENT );
            _rfc3253ToWvcmMap.put( "cannot-modify-version",                     CANNOT_MODIFY_VERSION );
            _rfc3253ToWvcmMap.put( "auto-checkout",                             CANNOT_AUTO_CHECKOUT );
            _rfc3253ToWvcmMap.put( "auto-checkout-checkin",                     CANNOT_AUTO_CHECKOUT_CHECKIN );
            
            // PROPFIND
            _rfc3253ToWvcmMap.put( "supported-live-property",                   CANNOT_RETRIEVE_SUPPORTED_PROPERTY );
            
            // PROPPATCH
            _rfc3253ToWvcmMap.put( "cannot-modify-version-controlled-property", CANNOT_MODIFY_CONTROLLED_PROPERTY );
            _rfc3253ToWvcmMap.put( "cannot-modify-version",                     CANNOT_MODIFY_VERSION );
            _rfc3253ToWvcmMap.put( "cannot-modify-protected-property",          CANNOT_MODIFY_PROTECTED_PROPERTY );
            _rfc3253ToWvcmMap.put( "supported-live-property",                   CANNOT_RETRIEVE_SUPPORTED_PROPERTY );
            _rfc3253ToWvcmMap.put( "auto-checkout",                             CANNOT_AUTO_CHECKOUT );
            _rfc3253ToWvcmMap.put( "auto-checkout-checkin",                     CANNOT_AUTO_CHECKOUT_CHECKIN );
            
            // DELETE
            _rfc3253ToWvcmMap.put( "no-version-delete",                         CANNOT_DELETE_VERSION );
            _rfc3253ToWvcmMap.put( "update-predecessor-set",                    CANNOT_UPDATE_PREDECESSOR_LIST );
            
            // COPY
            _rfc3253ToWvcmMap.put( "must-not-copy-versioning-property",         VERSIONING_PROPERTY_COPIED ); // ??
            _rfc3253ToWvcmMap.put( "auto-checkout",                             CANNOT_AUTO_CHECKOUT );
            _rfc3253ToWvcmMap.put( "auto-checkout-checkin",                     CANNOT_AUTO_CHECKOUT_CHECKIN );
            _rfc3253ToWvcmMap.put( "copy-creates-new-resource",                 CANNOT_CREATE_NEW_RESOURCE );
            
            // MOVE
            _rfc3253ToWvcmMap.put( "cannot-rename-version",                     CANNOT_RENAME_VERSION );
            _rfc3253ToWvcmMap.put( "preserve-versioning-properties",            CANNOT_PRESERVE_VERSIONING_PROPERTIES );
            
            // UNLOCK
            _rfc3253ToWvcmMap.put( "version-history-is-tree",                   VERSION_HISTORY_IS_NOT_TREE );
            _rfc3253ToWvcmMap.put( "auto-checkin",                              AUTO_CHECKIN_FAILED );
            
            // CHECKOUT
            _rfc3253ToWvcmMap.put( "must-be-checked-in",                        MUST_BE_CHECKED_IN );
            _rfc3253ToWvcmMap.put( "checkout-of-version-with-descendent-is-forbidden",    CANNOT_CHECKOUT_FORKING_IS_FORBIDDEN );
            _rfc3253ToWvcmMap.put( "checkout-of-version-with-descendent-is-discouraged",  CHECKOUT_OF_VERSION_WITH_DESCENDANT_IS_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "checkout-of-checked-out-version-is-forbidden",        CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_FORBIDDEN );
            _rfc3253ToWvcmMap.put( "checkout-of-checked-out-version-is-discouraged",      CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "is-checked-out",                            CANNOT_CHECK_OUT );
            _rfc3253ToWvcmMap.put( "initialize-predecessor-set",                CANNOT_INITIALIZE_PREDECESSOR_LIST );
            
            // CHECKIN
            _rfc3253ToWvcmMap.put( "must-be-checked-out",                       MUST_BE_CHECKED_OUT );
            _rfc3253ToWvcmMap.put( "version-history-is-tree",                   VERSION_HISTORY_IS_NOT_TREE );
            _rfc3253ToWvcmMap.put( "checkin-fork-forbidden",                    CANNOT_FORK );
            _rfc3253ToWvcmMap.put( "checkin-fork-discouraged",                  CANNOT_CHECKIN_FORK_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "create-version",                            CANNOT_CREATE_VERSION );
            _rfc3253ToWvcmMap.put( "initialize-version-content-and-properties", CANNOT_INITIALIZE_RESOURCE );
            _rfc3253ToWvcmMap.put( "checked-in",                                CANNOT_CHECK_IN );
            _rfc3253ToWvcmMap.put( "keep-checked-out",                          CANNOT_KEEP_CHECKED_OUT );
            
            // UNCHECKOUT
            _rfc3253ToWvcmMap.put( "must-be-checked-out-version-controlled-resource", CANNOT_UNCHECKOUT_NOT_CHECKED_OUT_RESOURCE );
            _rfc3253ToWvcmMap.put( "cancel-checked-out",                        CANNOT_CANCEL_CHECKED_OUT );
            _rfc3253ToWvcmMap.put( "restore-content-and-dead-properties",       CANNOT_RESTORE_CONTENT_AND_DEAD_PROPERTIES );
            
            // REPORT (DAV:locate-by-history)
            _rfc3253ToWvcmMap.put( "must-be-version-history",                   BAD_VERSION_HISTORY );
            
            // DELETE (version history feature)
            _rfc3253ToWvcmMap.put( "delete-version-set",                        CANNOT_DELETE_VERSION_LIST );
            _rfc3253ToWvcmMap.put( "version-history-has-root",                  CANNOT_UPDATE_VERSION_HISTORY_ROOT );
            
            // COPY (version history feature)
            _rfc3253ToWvcmMap.put( "cannot-copy-history",                       CANNOT_COPY_HISTORY );
            
            // MOVE (version history feature)
            _rfc3253ToWvcmMap.put( "cannot-rename-history",                     CANNOT_RENAME_HISTORY );
            
            // VERSION-CONTROL (version history feature)
            _rfc3253ToWvcmMap.put( "new-version-history",                       CANNOT_CREATE_NEW_VERSION_HISTORY );
            
            // CHECKIN (version history feature)
            _rfc3253ToWvcmMap.put( "add-to-history",                            CANNOT_ADD_TO_HISTORY );
            
            // MKWORKSPACE
            _rfc3253ToWvcmMap.put( "resource-must-be-null",                     RESOURCE_ALREADY_EXISTS_AT_LOCATION );
            _rfc3253ToWvcmMap.put( "workspace-location-ok",                     CANNOT_CREATE_AT_THIS_LOCATION );
            _rfc3253ToWvcmMap.put( "initialize-workspace",                      CANNOT_INITIALIZE_RESOURCE );
            
            // DELETE (workspace)
            _rfc3253ToWvcmMap.put( "delete-workspace-members",                  CANNOT_DELETE_WORKSPACE_MEMBERS );
            
            // MOVE (workspace)
            _rfc3253ToWvcmMap.put( "workspace-member-moved",                    CANNOT_MOVE_INTO_WORKSPACE );
            _rfc3253ToWvcmMap.put( "workspace-moved",                           CANNOT_MOVE_WORKSPACE );
            
            // VERSION-CONTROL (workspace)
            _rfc3253ToWvcmMap.put( "cannot-add-to-existing-history",            RESOURCE_ALREADY_EXISTS_AT_LOCATION );
            _rfc3253ToWvcmMap.put( "must-be-version",                           BAD_VERSION );
            _rfc3253ToWvcmMap.put( "one-version-controlled-resource-per-history-per-workspace", ONE_CONTROLLED_RESOURCE_PER_HISTORY_PER_WORKSPACE );
            _rfc3253ToWvcmMap.put( "new-version-controlled-resource",           CANNOT_CREATE_NEW_CONTROLLED_RESOURCE );
            
            // UPDATE (workspace)
            _rfc3253ToWvcmMap.put( "update-content-and-properties",             CANNOT_UPDATE_CONTENT_AND_PROPERTIES );
            _rfc3253ToWvcmMap.put( "report-properties",                         CANNOT_REPORT_PROPERTIES );
            
            // LABEL
            _rfc3253ToWvcmMap.put( "must-be-checked-in",                        MUST_BE_CHECKED_IN );
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "add-must-be-new-label",                     ADD_MUST_BE_NEW_LABEL );
            _rfc3253ToWvcmMap.put( "label-must-exist",                          CANNOT_REMOVE_LABEL_DOES_NOT_EXIST );
            _rfc3253ToWvcmMap.put( "add-or-set-label",                          CANNOT_ADD_OR_SET_LABEL );
            _rfc3253ToWvcmMap.put( "remove-label",                              CANNOT_REMOVE_LABEL );
            
            // GET (label)
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "apply-request-to-labeled-version",          APPLY_REQUEST_TO_LABELED_VERSION );
            
            // PROPFIND (label)
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "apply-request-to-labeled-version",          APPLY_REQUEST_TO_LABELED_VERSION );
            
            // COPY (label)
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "apply-request-to-labeled-version",          APPLY_REQUEST_TO_LABELED_VERSION );
            
            // CHECKOUT (label)
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "must-not-have-label-and-apply-to-version",  CANNOT_HAVE_LABEL_AND_APPLY_TO_VERSION );
            _rfc3253ToWvcmMap.put( "apply-request-to-labeled-version",          APPLY_REQUEST_TO_LABELED_VERSION );
            
            // UPDATE (label)
            _rfc3253ToWvcmMap.put( "must-select-version-in-history",            MUST_SELECT_VERSION_IN_HISTORY );
            _rfc3253ToWvcmMap.put( "depth-update",                              CANNOT_DEPTH_UPDATE );
            _rfc3253ToWvcmMap.put( "apply-request-to-labeled-version",          APPLY_REQUEST_TO_LABELED_VERSION );
            
            // CHECKOUT (working resource)
            _rfc3253ToWvcmMap.put( "checkout-of-version-with-descendent-is-forbidden",    CANNOT_CHECKOUT_FORKING_IS_FORBIDDEN );
            _rfc3253ToWvcmMap.put( "checkout-of-version-with-descendent-is-discouraged",  CHECKOUT_OF_VERSION_WITH_DESCENDANT_IS_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "checkout-of-checked-out-version-is-forbidden",        CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_FORBIDDEN );
            _rfc3253ToWvcmMap.put( "checkout-of-checked-out-version-is-discouraged",      CANNOT_CHECKOUT_MULTI_CHECKOUT_IS_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "create-working-resource",                             CANNOT_CREATE_WORKING_RESOURCE );
            _rfc3253ToWvcmMap.put( "create-working-resource-from-checked-in-version",     CANNOT_APPLY_TO_CHECKED_IN_VERSION );
            
            // CHECKIN (working resource)
            _rfc3253ToWvcmMap.put( "must-be-checked-in",                                  MUST_BE_CHECKED_IN );
            _rfc3253ToWvcmMap.put( "version-history-is-tree",                             VERSION_HISTORY_IS_NOT_TREE );
            _rfc3253ToWvcmMap.put( "checkin-fork-forbidden",                              CANNOT_FORK );
            _rfc3253ToWvcmMap.put( "checkin-fork-discouraged",                            CANNOT_CHECKIN_FORK_DISCOURAGED );
            _rfc3253ToWvcmMap.put( "no-overwrite-by-auto-update",                         CANNOT_OVERWRITE_BY_AUTO_UPDATE );
            _rfc3253ToWvcmMap.put( "create-version",                                      CANNOT_CREATE_VERSION );
            _rfc3253ToWvcmMap.put( "initialize-version-content-and-properties",           CANNOT_INITIALIZE_RESOURCE );
            _rfc3253ToWvcmMap.put( "auto-update",                                         CANNOT_AUTO_UPDATE );
            _rfc3253ToWvcmMap.put( "delete-working-resource",                             CANNOT_UNBIND_RESOURCE );
            
            // COPY (working resource)
            _rfc3253ToWvcmMap.put( "copy-creates-new-resource",                           CANNOT_CREATE_NEW_RESOURCE );
            
            // MOVE (working resource)
            _rfc3253ToWvcmMap.put( "cannot-rename-working-resource",                      CANNOT_RENAME_WORKING_RESOURCE );
            _rfc3253ToWvcmMap.put( "update-auto-update",                                  CANNOT_UPDATE_AUTO_UPDATE );
            
            // MERGE
            _rfc3253ToWvcmMap.put( "cannot-merge-checked-out-resource",                   CANNOT_MERGE_CHECKED_OUT_RESOURCE );
            _rfc3253ToWvcmMap.put( "checkout-not-allowed",                                CANNOT_MERGE_CHECKOUT_NOT_ALLOWED );
            _rfc3253ToWvcmMap.put( "ancestor-version",                                    CANNOT_MERGE_ANCESTOR_VERSION );
            _rfc3253ToWvcmMap.put( "decendent-version",                                   CANNOT_UPDATE_DESCENDANT_VERSION );
            _rfc3253ToWvcmMap.put( "checked-out-for-merge",                               CANNOT_CHECKOUT_FOR_MERGE );
            _rfc3253ToWvcmMap.put( "update-merge-set",                                    CANNOT_UPDATE_MERGE_LIST );
            _rfc3253ToWvcmMap.put( "report-properties",                                   CANNOT_REPORT_PROPERTIES );
            
            // DELETE (merge)
            _rfc3253ToWvcmMap.put( "delete-version-reference",                            CANNOT_DELETE_VERSION_REFERENCE );
            
            // CHECKIN (merge)
            _rfc3253ToWvcmMap.put( "merge-must-be-complete",                              CANNOT_CHECKIN_MERGE_NOT_COMPLETE );
            
            // BASELINE-CONTROL
            _rfc3253ToWvcmMap.put( "version-controlled-configuration-must-not-exist",     CANNOT_SET_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "must-be-baseline",                                    NOT_A_BASELINE );
            _rfc3253ToWvcmMap.put( "must-have-no-version-controlled-members",             CANNOT_HAVE_CONTROLLED_MEMBERS );
            _rfc3253ToWvcmMap.put( "one-baseline-controlled-collection-per-history-per-workspace",  CANNOT_HAVE_MULTIPLE_BASELINE_CONTROLLED_FOLDERS );
            _rfc3253ToWvcmMap.put( "create-version-controlled-configuration",             CANNOT_CREATE_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "reference-version-controlled-configuration",          CANNOT_REFERENCE_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "select-existing-baseline",                            CANNOT_SELECT_EXISTING_BASELINE );
            _rfc3253ToWvcmMap.put( "create-new-baseline",                                 CANNOT_CREATE_NEW_BASELINE );
            
            // REPORT (DAV:compare-baseline report)
            _rfc3253ToWvcmMap.put( "must-be-baseline",                                    NOT_A_BASELINE );
            _rfc3253ToWvcmMap.put( "baselines-from-same-history",                         BASELINES_FROM_SAME_HISTORY );
            
            // CHECKOUT (baseline)
            _rfc3253ToWvcmMap.put( "must-not-update-baseline-collection",                 CANNOT_UPDATE_BASELINE_FOLDER ); // ??
            
            // CHECKIN (baseline)
            _rfc3253ToWvcmMap.put( "no-checked-out-baseline-controlled-collection-members", NO_CHECKED_OUT_BASELINE_CONTROLLED_FOLDER_MEMBERS );
            _rfc3253ToWvcmMap.put( "one-version-per-history-per-baseline",                ONE_VERSION_PER_HISTORY_PER_BASELINE );
            _rfc3253ToWvcmMap.put( "cannot-modify-version-controlled-configuration",      CANNOT_MODIFY_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "create-baseline-collection",                          CANNOT_CREATE_BASELINE_FOLDER );
            _rfc3253ToWvcmMap.put( "modify-configuration",                                CANNOT_MODIFY_CONFIGURATION );
            
            // UPDATE (baseline)
            _rfc3253ToWvcmMap.put( "baseline-controlled-members-must-be-checked-in",      CANNOT_UPDATE_MEMBERS_ARE_CHECKED_OUT );
            _rfc3253ToWvcmMap.put( "must-not-update-baseline-collection",                 CANNOT_UPDATE_BASELINE_FOLDER ); // ??
            _rfc3253ToWvcmMap.put( "cannot-modify-version-controlled-configuration",      CANNOT_MODIFY_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "set-baseline-controlled-collection-members",          CANNOT_SET_BASELINE_CONTROLLED_FOLDER_MEMBERS );
            _rfc3253ToWvcmMap.put( "update-subbaselines",                                 CANNOT_UPDATE_SUB_BASELINES );
            _rfc3253ToWvcmMap.put( "modify-configuration",                                CANNOT_MODIFY_CONFIGURATION );
            
            // MERGE (baseline)
            _rfc3253ToWvcmMap.put( "must-not-update-baseline-collection",                 CANNOT_UPDATE_BASELINE_FOLDER ); // ??
            _rfc3253ToWvcmMap.put( "cannot-modify-version-controlled-configuration",      CANNOT_MODIFY_CONTROLLED_CONFIGURATION );
            _rfc3253ToWvcmMap.put( "merge-baseline",                                      CANNOT_MERGE_BASELINE );
            _rfc3253ToWvcmMap.put( "merge-subbaselines",                                  CANNOT_MERGE_SUB_BASELINES );
            _rfc3253ToWvcmMap.put( "set-baseline-controlled-collection-members",          CANNOT_SET_BASELINE_CONTROLLED_FOLDER_MEMBERS );
            _rfc3253ToWvcmMap.put( "modify-configuration",                                CANNOT_MODIFY_CONFIGURATION );
            
            // ACTIVITY
            _rfc3253ToWvcmMap.put( "resource-must-be-null",                               RESOURCE_ALREADY_EXISTS_AT_LOCATION );
            _rfc3253ToWvcmMap.put( "activity-location-ok",                                CANNOT_CREATE_AT_THIS_LOCATION );
            _rfc3253ToWvcmMap.put( "initialize-activity",                                 CANNOT_INITIALIZE_RESOURCE );
            
            // REPORT (DAV:latest-activity-version)
            _rfc3253ToWvcmMap.put( "must-be-activity",                                    NOT_AN_ACTIVITY );
            _rfc3253ToWvcmMap.put( "delete-activity-reference",                           CANNOT_UNBIND_ACTIVITY_REFERENCE );
            _rfc3253ToWvcmMap.put( "update-checked-out-reference",                        CANNOT_UPDATE_CHECKED_OUT_REFERENCE );
            _rfc3253ToWvcmMap.put( "update-activity-reference",                           CANNOT_UPDATE_ACTIVITY_REFERENCE );
            _rfc3253ToWvcmMap.put( "update-workspace-reference",                          CANNOT_UPDATE_WORKSPACE_REFERENCE );
            
            // CHECKOUT (activity)
            _rfc3253ToWvcmMap.put( "one-checkout-per-activity-per-history",               ONE_CHECKOUT_PER_ACTIVITY_PER_HISTORY );
            _rfc3253ToWvcmMap.put( "linear-activity",                                     CANNOT_CREATE_BRANCH_IN_ACTIVITY );
            _rfc3253ToWvcmMap.put( "initialize-activity-set",                             CANNOT_INITIALIZE_ACTIVITY_LIST );
            _rfc3253ToWvcmMap.put( "initialize-unreserved",                               CANNOT_INITIALIZE_UNRESERVED );
            
            // CHECKIN (activity)
            _rfc3253ToWvcmMap.put( "linear-activity",                                     CANNOT_CREATE_BRANCH_IN_ACTIVITY );
            _rfc3253ToWvcmMap.put( "atomic-activity-checkin",                             CANNOT_CHECKIN_ALL_RESOURCES );
            _rfc3253ToWvcmMap.put( "initialize-activity-set",                             CANNOT_INITIALIZE_ACTIVITY_LIST );
            _rfc3253ToWvcmMap.put( "activity-checkin",                                    CANNOT_CHECKIN_ACTIVITY_RESOURCES );
            
            // MERGE (activity)
            _rfc3253ToWvcmMap.put( "checkin-activity",                                    CANNOT_CHECKIN_ACTIVITY );
            
            // DELETE (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "cannot-modify-checked-in-parent",                     CANNOT_MODIFY_CHECKED_IN_PARENT );
            _rfc3253ToWvcmMap.put( "delete-working-collection-binding",                   CANNOT_UNBIND_RESOURCE );
            
            // COPY (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "cannot-copy-collection-version",                      CANNOT_COPY_FOLDER_VERSION );
            _rfc3253ToWvcmMap.put( "cannot-modify-checked-in-parent",                     CANNOT_MODIFY_CHECKED_IN_PARENT );
            _rfc3253ToWvcmMap.put( "cannot-modify-destination-checked-in-parent",         CANNOT_MODIFY_DESTINATION_CHECKED_IN_PARENT );
            
            // VERSION-CONTROL (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "cannot-modify-checked-in-parent",                     CANNOT_MODIFY_CHECKED_IN_PARENT );
            _rfc3253ToWvcmMap.put( "new-version-controlled-collection",                   CANNOT_INITIALIZE_RESOURCE );
            
            // CHECKOUT (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "initialize-version-history-bindings",                 CANNOT_INITIALIZE_BINDINGS );
            
            // CHECKIN (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "initialize-version-controlled-bindings",              CANNOT_CONTROL_FOLDER_MEMBERS );
            _rfc3253ToWvcmMap.put( "version-control-working-collection-members",          CANNOT_CONTROL_FOLDER_MEMBERS );
            
            // UNCHECKOUT, UPDATE and MERGE (version-controlled-collection)
            _rfc3253ToWvcmMap.put( "update-version-controlled-collection-members",        CANNOT_UPDATE_CONTROLLED_FOLDER_MEMBERS );
            
            // BIND
            _rfc3253ToWvcmMap.put( "bind-source-exists",                                  NOT_FOUND );
            _rfc3253ToWvcmMap.put( "name-allowed",                                        CANNOT_CREATE_AT_THIS_LOCATION );
            _rfc3253ToWvcmMap.put( "binding-allowed",                                     MULTIPLE_BINDINGS_NOT_ALLOWED );
            _rfc3253ToWvcmMap.put( "cross-server-binding",                                NO_CROSS_SERVER_BINDING );
            _rfc3253ToWvcmMap.put( "can-overwrite",                                       CANNOT_OVERWRITE );
            _rfc3253ToWvcmMap.put( "cycle-allowed",                                       CYCLE_NOT_ALLOWED );
            _rfc3253ToWvcmMap.put( "new-binding",                                         CANNOT_CREATE_BINDING );
            
            // UNBIND
            _rfc3253ToWvcmMap.put( "unbind-source-exists",                                NOT_FOUND );
            
            // REBIND
            _rfc3253ToWvcmMap.put( "rebind-source-exists",                                NOT_FOUND );
            
            // NOT YET STANDARD - ACL
            _rfc3253ToWvcmMap.put( "no-ace-conflict",                                     ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "no-protected-ace-conflict",                           ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "no-inherited-ace-conflict",                           ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "limited-number-of-aces",                              ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "deny-before-grant",                                   DENY_BEFORE_GRANT );
            _rfc3253ToWvcmMap.put( "grant-only",                                          GRANT_ONLY );
            _rfc3253ToWvcmMap.put( "no-invert",                                           ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "no-abstract",                                         ACE_CONFLICT );
            _rfc3253ToWvcmMap.put( "not-supported-privilege",                             PRIVILEGE_NOT_SUPPORTED );
            _rfc3253ToWvcmMap.put( "missing-required-principal",                          MISSING_REQUIRED_PRINCIPAL );
            _rfc3253ToWvcmMap.put( "recognized-principal",                                PRINCIPAL_NOT_FOUND );
            _rfc3253ToWvcmMap.put( "allowed-principal",                                   INVALID_PRINCIPAL );
        }
        
        /**  Hidden constuctor for type-safety. */
        private ReasonCode() {}
        
        /**
         * The ReasonCode for an RFC-3253 pre- or post-condition.
         *
         * @param condition the RFC-3253 pre- or post- condition local name.
         * @return the corresponding ReasonCode.
         */
        public static ReasonCode getReasonCodeForRFC3253Condition( String condition ) {
            return (ReasonCode) _rfc3253ToWvcmMap.get( condition );
        }
        
    }
    
    /**
     * The location of the resource that caused the exception.
     */
    private String _location;
    
    /**
     * The reason for the exception.
     */
    private ReasonCode _reasonCode;
    
    /**
     * An array of nested exceptions that further explain the cause of the error.
     */
    private Exception[] _nestedExceptions = null;
    
    /**
     * Constructs a WvcmException.
     * @param message The detail message for the exception.
     * @param location The location of the resource causing the exception.
     * @param reasonCode The reason for the exception.
     * @param nestedExceptions Any nested exception.
     */
    public WvcmException(
        String message,
        String location,
        ReasonCode  reasonCode,
        Exception[] nestedExceptions) {
        super(message);
        _location = location;
        _reasonCode = reasonCode;
        _nestedExceptions = nestedExceptions;
    }
    
    /**
     * Returns the location of the resource causing the exception.
     */
    public String getLocation() {
        return _location;
    }
    
    /**
     * Returns the reason code that describes the nature of the error.
     */
    public ReasonCode getReasonCode() {
        return _reasonCode;
    }
    
    /**
     * Returns any nested exception that further explains the cause of the error.
     */
    public Exception[] getNestedExceptions() {
        return _nestedExceptions;
        
    }
}
