/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/ControllableResourceAccessor.java,v 1.14 2004/07/30 06:52:26 ozeigermann Exp $
 * $Revision: 1.14 $
 * $Date: 2004/07/30 06:52:26 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store;

import java.util.Iterator;
import java.util.List;
import javax.wvcm.PropertyNameList;
import javax.wvcm.Version;
import javax.wvcm.WvcmException;

/**
 * Persistent controllable resource state accessor.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.14 $
 */
public interface ControllableResourceAccessor extends ResourceAccessor {
    
    /**
     * Create a new persistent controllable resource at the location of the
     * associated resource.
     *
     * @throws   WvcmException
     */
    public void doCreateResource() throws WvcmException;
    
    /**
     * Put the associated resource under (version) control.
     *
     * @throws   WvcmException
     */
    public void doControl() throws WvcmException;
    
    /**
     * Checkout the associated resource so that its content can be modified.
     * @param forkOK Indicates whether to do the checkout even if the
     * version already has a successor or a checkout.
     * @param activityList A list of {@link Activity} objects that identify
     * the activities to which the work on the
     * checked-out resource will contribute.
     * @param newActivity Indicates whether to create a new activity
     * for the checkout.
     * @param unreserved Indicates whether to do the checkout even
     * if there already is a checkout from the currently selected
     * version in one of the activities specified in the activityList.
     */
    public void doCheckout(boolean forkOK, List activityList, boolean newActivity, boolean unreserved) throws WvcmException;

    /**
     * Changes the state of the resource identified by the associated {@link ControllableResource}
     * to be checked-in.  If version history is being tracked for the
     * resource, the current content of the resource is captured
     * in a new version resource at a server-defined location.
     * @param keepCheckedOut indicates whether to checkout the
     * resource again immediately after checking it in.
     * If keepCheckedOut is <code>true</code>, the ActivityList
     * and Unreserved properties should not be changed by the checkout.
     * @param forkOK indicates whether to fork even if forking
     * is discouraged.
     */
    public void doCheckin(boolean keepCheckedOut, boolean forkOK) throws WvcmException;
    
    /**
     * Cancels the checkout of a version-controlled resource,
     * and restores its content to the state of its CheckedOut version.
     */
    public void doUncheckout() throws WvcmException;
        
    /**
     * Update the state of the associated checked-in version-controlled
     * {@link ControllableResource} to be the same as the specified version from
     * the version history of the associated {@link ControllableResource}.
     * @param version The version specifying the new state of the resource
     * @param wantedPropertyList The list of properties that will be available
     * @return An iterator of {@link ControllableResource} objects that
     * have been modified as a result of the doUpdate request.
     * Each version-controlled resource contains a value for each
     * of the properties specified in the WantedPropertyList.
     */
    public Iterator doUpdate(Version version, PropertyNameList wantedPropertyList) throws WvcmException;
    
    /**
     * Create a new controlled resource
     * at the location identified by the proxy.  The resource
     * is associated with an existing version history,
     * and is initialized with a specified version from that
     * version history.  The request will
     * fail if a resource already exists at that location.
     * @param v The version used to initialize the controlled
     * resource.
     */
    public void doCreateVersionControlledResource(Version v) throws WvcmException;
    
    /**
     * Apply {@link Version#doAddLabel Version.doAddLabel} to the CheckedIn version of this {@link ControllableResource}.
     */
    public void doAddLabel(String label) throws WvcmException;
    
    
    /**
     * Apply {@link Version#doSetLabel Version.doSetLabel} to the CheckedIn version of this {@link ControllableResource}.
     */
    public void doSetLabel(String label) throws WvcmException;
        
    /**
     * Apply {@link Version#doRemoveLabel Version.doRemoveLabel} to the CheckedIn version of this {@link ControllableResource}.
     */
    public void doRemoveLabel(String label) throws WvcmException;
    
    /**
     * Apply {@link VersionHistory#doReadLabelledVersionProperties
     * Version.doReadLabelledVersionProperties}
     * to the version history of this {@link ControllableResource}.
     */
    public Version doReadLabelledVersionProperties(String label, PropertyNameList wantedPropertyList) throws WvcmException;
}



